#!/usr/bin/env python
"""Execute the tests for the razers3 program.

The golden test outputs are generated by the script generate_outputs.sh.

You have to give the root paths to the source and the binaries as arguments to
the program.  These are the paths to the directory that contains the 'projects'
directory.

Usage:  run_tests.py SOURCE_ROOT_PATH BINARY_ROOT_PATH
"""
import logging
import os.path
import sys

# Automagically add util/py_lib to PYTHONPATH environment variable.
path = os.path.abspath(os.path.join(os.path.dirname(__file__), '..', '..',
                                    '..', '..', 'util', 'py_lib'))
sys.path.insert(0, path)

import seqan.app_tests as app_tests

class RemovePairIdColumn(object):
    """Transformation to remove pair id column."""
    
    def __init__(self, col_no=8, min_cols=8):
        # The index of the column to remove.
        self.col_no = col_no
        # If there are less than min_col columns then we don't remove.
        self.min_cols = min_cols

    def apply(self, text, is_left):
        lines = text.splitlines(True)
        lines2 = []
        for line in lines:
            cols = line.split('\t')
            if len(cols) > self.min_cols:
                cols = cols[0:self.col_no] + cols[self.col_no + 1:]
            lines2.append('\t'.join(cols))
        return ''.join(lines2)


def main(source_base, binary_base, num_threads=1):
    """Main entry point of the script."""

    print 'Executing test for razers3'
    print '==========================='
    print

    ph = app_tests.TestPathHelper(
        source_base, binary_base,
        'extras/apps/razers3/tests')  # tests dir

    # ============================================================
    # Auto-detect the binary path.
    # ============================================================

    path_to_program = app_tests.autolocateBinary(
      binary_base, 'extras/apps/razers3', 'razers3')

    # ============================================================
    # Built TestConf list.
    # ============================================================

    # Build list with TestConf objects, analoguely to how the output
    # was generated in generate_outputs.sh.
    conf_list = []

    # We prepare a list of transforms to apply to the output files.  This is
    # used to strip the input/output paths from the programs' output to
    # make it more canonical and host independent.
    ph.outFile('-')  # To ensure that the out path is set.
    transforms = [
        app_tests.ReplaceTransform(os.path.join(ph.source_base_path, 'extras/apps/razers3/tests') + os.sep, '', right=True),
        app_tests.ReplaceTransform(ph.temp_dir + os.sep, '', right=True),
        ]

    # Transforms for SAM output format only.  Make VN field of @PG header canonical.
    sam_transforms = [app_tests.RegexpReplaceTransform(r'\tVN:[^\t]*', r'\tVN:VERSION', right=True, left=True)]

    # Transforms for RazerS output format only.  Remove pair id column.
    razers_transforms = [RemovePairIdColumn()]

    # ============================================================
    # Run Adeno Single-End Tests
    # ============================================================

    # We run the following for all read lengths we have reads for.
    for rl in [36, 100]:
        # Run with default options.
        conf = app_tests.TestConf(
            program=path_to_program,
            redir_stdout=ph.outFile('se-adeno-reads%d_1-tc%d.stdout' % (rl, num_threads)),
            args=['-tc', str(num_threads),
                  ph.inFile('adeno-genome.fa'),
                  ph.inFile('adeno-reads%d_1.fa' % rl),
                  '-o', ph.outFile('se-adeno-reads%d_1-tc%d.razers' % (rl, num_threads))],
            to_diff=[(ph.inFile('se-adeno-reads%d_1-tc%d.razers' % (rl, num_threads)),
                      ph.outFile('se-adeno-reads%d_1-tc%d.razers' % (rl, num_threads))),
                     (ph.inFile('se-adeno-reads%d_1-tc%d.stdout' % (rl, num_threads)),
                      ph.outFile('se-adeno-reads%d_1-tc%d.stdout' % (rl, num_threads)))])
        conf_list.append(conf)

        # Allow indels.
        conf = app_tests.TestConf(
            program=path_to_program,
            redir_stdout=ph.outFile('se-adeno-reads%d_1-ng-tc%d.stdout' % (rl, num_threads)),
            args=['-tc', str(num_threads),
                  '-ng',
                  ph.inFile('adeno-genome.fa'),
                  ph.inFile('adeno-reads%d_1.fa' % rl),
                  '-o', ph.outFile('se-adeno-reads%d_1-ng-tc%d.razers' % (rl, num_threads))],
            to_diff=[(ph.inFile('se-adeno-reads%d_1-ng-tc%d.razers' % (rl, num_threads)),
                      ph.outFile('se-adeno-reads%d_1-ng-tc%d.razers' % (rl, num_threads))),
                     (ph.inFile('se-adeno-reads%d_1-ng-tc%d.stdout' % (rl, num_threads)),
                      ph.outFile('se-adeno-reads%d_1-ng-tc%d.stdout' % (rl, num_threads)))])
        conf_list.append(conf)

        # Compute forward/reverse matches only.
        for o in ['-r', '-f']:
            conf = app_tests.TestConf(
                program=path_to_program,
                redir_stdout=ph.outFile('se-adeno-reads%d_1%s-tc%d.stdout' % (rl, o, num_threads)),
                args=['-tc', str(num_threads),
                      o,
                      ph.inFile('adeno-genome.fa'),
                      ph.inFile('adeno-reads%d_1.fa' % rl),
                      '-o', ph.outFile('se-adeno-reads%d_1%s-tc%d.razers' % (rl, o, num_threads))],
                to_diff=[(ph.inFile('se-adeno-reads%d_1%s-tc%d.razers' % (rl, o, num_threads)),
                          ph.outFile('se-adeno-reads%d_1%s-tc%d.razers' % (rl, o, num_threads))),
                         (ph.inFile('se-adeno-reads%d_1%s-tc%d.stdout' % (rl, o, num_threads)),
                          ph.outFile('se-adeno-reads%d_1%s-tc%d.stdout' % (rl, o, num_threads)))])
            conf_list.append(conf)

        # Compute with different identity rates.
        for i in range(90, 101):
            conf = app_tests.TestConf(
                program=path_to_program,
                redir_stdout=ph.outFile('se-adeno-reads%d_1-i%d-tc%d.stdout' % (rl, i, num_threads)),
                args=['-tc', str(num_threads),
                      '-i', str(i),
                      ph.inFile('adeno-genome.fa'),
                      ph.inFile('adeno-reads%d_1.fa' % rl),
                      '-o', ph.outFile('se-adeno-reads%d_1-i%d-tc%d.razers' % (rl, i, num_threads))],
                to_diff=[(ph.inFile('se-adeno-reads%d_1-i%d-tc%d.razers' % (rl, i, num_threads)),
                          ph.outFile('se-adeno-reads%d_1-i%d-tc%d.razers' % (rl, i, num_threads))),
                         (ph.inFile('se-adeno-reads%d_1-i%d-tc%d.stdout' % (rl, i, num_threads)),
                          ph.outFile('se-adeno-reads%d_1-i%d-tc%d.stdout' % (rl, i, num_threads)))])
            conf_list.append(conf)

        # Compute with different output formats.
        for of, suffix in enumerate(['razers', 'fa', 'eland', 'gff', 'sam', 'afg']):
            this_transforms = list(transforms)
            if suffix == 'razers':
                this_transforms += razers_transforms
            elif suffix == 'sam':
                this_transforms += sam_transforms
            conf = app_tests.TestConf(
                program=path_to_program,
                redir_stdout=ph.outFile('se-adeno-reads%d_1-of%d-tc%d.stdout' % (rl, of, num_threads)),
                args=['-tc', str(num_threads),
                      ph.inFile('adeno-genome.fa'),
                      ph.inFile('adeno-reads%d_1.fa' % rl),
                      '-o', ph.outFile('se-adeno-reads%d_1-of%d-tc%d.%s' % (rl, of, num_threads, suffix))],
                to_diff=[(ph.inFile('se-adeno-reads%d_1-of%d-tc%d.%s' % (rl, of, num_threads, suffix)),
                          ph.outFile('se-adeno-reads%d_1-of%d-tc%d.%s' % (rl, of, num_threads, suffix)),
                          this_transforms),
                         (ph.inFile('se-adeno-reads%d_1-of%d-tc%d.stdout' % (rl, of, num_threads)),
                          ph.outFile('se-adeno-reads%d_1-of%d-tc%d.stdout' % (rl, of, num_threads)),
                          transforms)])
            conf_list.append(conf)

        # Compute with different sort orders.
        for so in [0, 1]:
            conf = app_tests.TestConf(
                program=path_to_program,
                redir_stdout=ph.outFile('se-adeno-reads%d_1-so%d-tc%d.stdout' % (rl, so, num_threads)),
                args=['-tc', str(num_threads),
                      '-so', str(so),
                      ph.inFile('adeno-genome.fa'),
                      ph.inFile('adeno-reads%d_1.fa' % rl),
                      '-o', ph.outFile('se-adeno-reads%d_1-so%d-tc%d.razers' % (rl, so, num_threads))],
                to_diff=[(ph.inFile('se-adeno-reads%d_1-so%d-tc%d.razers' % (rl, so, num_threads)),
                          ph.outFile('se-adeno-reads%d_1-so%d-tc%d.razers' % (rl, so, num_threads))),
                         (ph.inFile('se-adeno-reads%d_1-so%d-tc%d.stdout' % (rl, so, num_threads)),
                          ph.outFile('se-adeno-reads%d_1-so%d-tc%d.stdout' % (rl, so, num_threads)))])
            conf_list.append(conf)

    # ============================================================
    # Run Adeno Paired-End Tests
    # ============================================================

    # We run the following for all read lengths we have reads for.
    for rl in [36, 100]:
        # Run with default options.
        conf = app_tests.TestConf(
            program=path_to_program,
            redir_stdout=ph.outFile('pe-adeno-reads%d_2-tc%d.stdout' % (rl, num_threads)),
            args=['-tc', str(num_threads),
                  ph.inFile('adeno-genome.fa'),
                  ph.inFile('adeno-reads%d_1.fa' % rl),
                  ph.inFile('adeno-reads%d_2.fa' % rl),
                  '-o', ph.outFile('pe-adeno-reads%d_2-tc%d.razers' % (rl, num_threads))],
            to_diff=[(ph.inFile('pe-adeno-reads%d_2-tc%d.razers' % (rl, num_threads)),
                      ph.outFile('pe-adeno-reads%d_2-tc%d.razers' % (rl, num_threads)),
                      razers_transforms),
                     (ph.inFile('pe-adeno-reads%d_2-tc%d.stdout' % (rl, num_threads)),
                      ph.outFile('pe-adeno-reads%d_2-tc%d.stdout' % (rl, num_threads)))])
        conf_list.append(conf)

        # Allow indels.
        conf = app_tests.TestConf(
            program=path_to_program,
            redir_stdout=ph.outFile('pe-adeno-reads%d_2-tc%d.stdout' % (rl, num_threads)),
            args=['-tc', str(num_threads),
                  ph.inFile('adeno-genome.fa'),
                  ph.inFile('adeno-reads%d_1.fa' % rl),
                  ph.inFile('adeno-reads%d_2.fa' % rl),
                  '-o', ph.outFile('pe-adeno-reads%d_2-tc%d.razers' % (rl, num_threads))],
            to_diff=[(ph.inFile('pe-adeno-reads%d_2-tc%d.razers' % (rl, num_threads)),
                      ph.outFile('pe-adeno-reads%d_2-tc%d.razers' % (rl, num_threads)),
                      razers_transforms),
                     (ph.inFile('pe-adeno-reads%d_2-tc%d.stdout' % (rl, num_threads)),
                      ph.outFile('pe-adeno-reads%d_2-tc%d.stdout' % (rl, num_threads)))])
        conf_list.append(conf)

        # Compute forward/reverse matches only.
        for o in ['-r', '-f']:
            conf = app_tests.TestConf(
                program=path_to_program,
                redir_stdout=ph.outFile('pe-adeno-reads%d_2%s-tc%d.stdout' % (rl, o, num_threads)),
                args=['-tc', str(num_threads),
                      o,
                      ph.inFile('adeno-genome.fa'),
                      ph.inFile('adeno-reads%d_1.fa' % rl),
                      ph.inFile('adeno-reads%d_2.fa' % rl),
                      '-o', ph.outFile('pe-adeno-reads%d_2%s-tc%d.razers' % (rl, o, num_threads))],
                to_diff=[(ph.inFile('pe-adeno-reads%d_2%s-tc%d.razers' % (rl, o, num_threads)),
                          ph.outFile('pe-adeno-reads%d_2%s-tc%d.razers' % (rl, o, num_threads)),
                          razers_transforms),
                         (ph.inFile('pe-adeno-reads%d_2%s-tc%d.stdout' % (rl, o, num_threads)),
                          ph.outFile('pe-adeno-reads%d_2%s-tc%d.stdout' % (rl, o, num_threads)))])
            conf_list.append(conf)

        # Compute with different identity rates.
        for i in range(90, 101):
            conf = app_tests.TestConf(
                program=path_to_program,
                redir_stdout=ph.outFile('pe-adeno-reads%d_2-i%d-tc%d.stdout' % (rl, i, num_threads)),
                args=['-tc', str(num_threads),
                      '-i', str(i),
                      ph.inFile('adeno-genome.fa'),
                      ph.inFile('adeno-reads%d_1.fa' % rl),
                      ph.inFile('adeno-reads%d_2.fa' % rl),
                      '-o', ph.outFile('pe-adeno-reads%d_2-i%d-tc%d.razers' % (rl, i, num_threads))],
                to_diff=[(ph.inFile('pe-adeno-reads%d_2-i%d-tc%d.razers' % (rl, i, num_threads)),
                          ph.outFile('pe-adeno-reads%d_2-i%d-tc%d.razers' % (rl, i, num_threads)),
                          razers_transforms),
                         (ph.inFile('pe-adeno-reads%d_2-i%d-tc%d.stdout' % (rl, i, num_threads)),
                          ph.outFile('pe-adeno-reads%d_2-i%d-tc%d.stdout' % (rl, i, num_threads)))])
            conf_list.append(conf)

        # Compute with different output formats.
        for of, suffix in enumerate(['razers', 'fa', 'eland', 'gff', 'sam', 'afg']):
            this_transforms = list(transforms)
            if suffix == 'razers':
                this_transforms += razers_transforms
            elif suffix == 'sam':
                this_transforms += sam_transforms
            conf = app_tests.TestConf(
                program=path_to_program,
                redir_stdout=ph.outFile('pe-adeno-reads%d_2-of%d-tc%d.stdout' % (rl, of, num_threads)),
                args=['-tc', str(num_threads),
                      ph.inFile('adeno-genome.fa'),
                      ph.inFile('adeno-reads%d_1.fa' % rl),
                      ph.inFile('adeno-reads%d_2.fa' % rl),
                      '-o', ph.outFile('pe-adeno-reads%d_2-of%d-tc%d.%s' % (rl, of, num_threads, suffix))],
                to_diff=[(ph.inFile('pe-adeno-reads%d_2-of%d-tc%d.%s' % (rl, of, num_threads, suffix)),
                          ph.outFile('pe-adeno-reads%d_2-of%d-tc%d.%s' % (rl, of, num_threads, suffix)),
                          this_transforms),
                         (ph.inFile('pe-adeno-reads%d_2-of%d-tc%d.stdout' % (rl, of, num_threads)),
                          ph.outFile('pe-adeno-reads%d_2-of%d-tc%d.stdout' % (rl, of, num_threads)),
                          this_transforms)])
            conf_list.append(conf)

        # Compute with different sort orders.
        for so in [0, 1]:
            conf = app_tests.TestConf(
                program=path_to_program,
                redir_stdout=ph.outFile('pe-adeno-reads%d_2-so%d-tc%d.stdout' % (rl, so, num_threads)),
                args=['-tc', str(num_threads),
                      '-so', str(so),
                      ph.inFile('adeno-genome.fa'),
                      ph.inFile('adeno-reads%d_1.fa' % rl),
                      ph.inFile('adeno-reads%d_2.fa' % rl),
                      '-o', ph.outFile('pe-adeno-reads%d_2-so%d-tc%d.razers' % (rl, so, num_threads))],
                to_diff=[(ph.inFile('pe-adeno-reads%d_2-so%d-tc%d.razers' % (rl, so, num_threads)),
                          ph.outFile('pe-adeno-reads%d_2-so%d-tc%d.razers' % (rl, so, num_threads)),
                          razers_transforms),
                         (ph.inFile('pe-adeno-reads%d_2-so%d-tc%d.stdout' % (rl, so, num_threads)),
                          ph.outFile('pe-adeno-reads%d_2-so%d-tc%d.stdout' % (rl, so, num_threads)))])
            conf_list.append(conf)

    # Execute the tests.
    failures = 0
    for conf in conf_list:
        res = app_tests.runTest(conf)
        # Output to the user.
        print ' '.join(['razers3'] + conf.args),
        if res:
             print 'OK'
        else:
            failures += 1
            print 'FAILED'

    # Cleanup.
    ph.deleteTempDir()

    print '=============================='
    print '     total tests: %d' % len(conf_list)
    print '    failed tests: %d' % failures
    print 'successful tests: %d' % (len(conf_list) - failures)
    print '=============================='
    # Compute and return return code.
    return failures != 0


if __name__ == '__main__':
    sys.exit(app_tests.main(main))
