///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
#include "rheolef/index_set.h"

namespace rheolef {

index_set::index_set (const index_set& x)
 : base()
{
  for (std::set<size_type>::const_iterator iter = x.base::begin(), last = x.base::end(); iter != last; iter++) {
    base::insert (*iter);
  }
}
index_set&
index_set::operator= (const index_set& x)
{
  base::clear();
  for (std::set<size_type>::const_iterator iter = x.base::begin(), last = x.base::end(); iter != last; iter++) {
    base::insert (*iter);
  }
  return *this;
}
index_set&
index_set::operator+= (const index_set& x)
{
  for (std::set<size_type>::const_iterator iter = x.base::begin(), last = x.base::end(); iter != last; iter++) {
    base::insert (*iter);
  }
  return *this;
}
std::istream&
operator>> (std::istream& is, index_set& x)
{
  typedef index_set::size_type size_type;
  size_type n;
  is >> n;
  x.clear();
  for (size_type i = 0; i < n; i++) {
    size_type xi;
    is >> xi;
    x.insert (xi);
  }
  return is;
}
std::ostream&
operator<< (std::ostream& os, const index_set& x)
{
  typedef index_set::size_type size_type;
  os << x.base::size() << "\t";	
  for (std::set<size_type>::const_iterator iter = x.base::begin(), last = x.base::end(); iter != last; iter++) {
    os << " " << *iter;	
  }
  return os;
}
// c := a union b
void 
set_union (const index_set& a, const index_set& b, index_set& c) 
{
    c.erase(c.begin(), c.end());
    std::set_union (a.begin(), a.end(), b.begin(), b.end(), std::inserter((index_set::base&)c, c.end()));
}
// c := a inter b
void 
set_intersection (const index_set& a, const index_set& b, index_set& c) 
{
    c.erase(c.begin(), c.end());
    std::set_intersection (a.begin(), a.end(), b.begin(), b.end(), std::inserter((index_set::base&)c, c.end()));
}
// a := a union b
void 
index_set::inplace_union (const index_set& b) 
{
    const size_type infty = std::numeric_limits<size_type>::max();
          iterator iter_a =   begin();
          iterator last_a =   end();
    const_iterator iter_b = b.begin();
    const_iterator last_b = b.end();
    while (iter_a != last_a || iter_b != last_b) {
      if  (iter_b == last_b) return;
      if  (iter_a == last_a) {
	base::insert(iter_b,last_b);
        return;
      }
      size_type ia = (*iter_a);
      size_type ib = (*iter_b);
      if (ia == ib) {
	iter_a++;
	iter_b++;
      } else if (ia < ib) {
	iter_a++;
      } else { // ib < ia
        // Set has the important property that inserting a new element
        // into a set does not invalidate iterators that point to existing elements.
        insert(ib);
	iter_b++;
      }
    }
}
// a := a union b
void 
index_set::inplace_intersection (const index_set& b) 
{
    const size_type infty = std::numeric_limits<size_type>::max();
          iterator iter_a =   begin();
          iterator last_a =   end();
    const_iterator iter_b = b.begin();
    const_iterator last_b = b.end();
    while (iter_a != last_a || iter_b != last_b) {
      if  (iter_a == last_a) return;
      if  (iter_b == last_b) { // last_b is reached
	base::erase(iter_a, last_a);
        return;
      }
      size_type ia = (*iter_a);
      size_type ib = (*iter_b);
      if (ia == ib) {
	iter_a++;
	iter_b++;
      } else if (ia < ib) {
        // Erasing an element from a set also does not invalidate any iterators, except,
        // of course, for iterators that actually point to the element that is being erased.
        iterator curr_a = iter_a++;
	base::erase(curr_a);
      } else { // ib < ia
	iter_b++;
      }
    }
}

} // namespace rheolef
