/* SPDX-License-Identifier: GPL-2.0-only OR GPL-3.0-only */
/* Copyright (c) 2025 Brett Sheffield <bacs@librecast.net> */

#ifndef _LIBRECAST_KEY_H
#define _LIBRECAST_KEY_H 1

#include <assert.h>
#include <limits.h>
#include <stdint.h>
#include <sys/param.h>
#include <sys/types.h>
#include <librecast/types.h>

/* ensure uint8_t is the same as unsigned char */
static_assert(CHAR_BIT == 8, "CHAR_BIT != 8");

/* generate new public keypair
 * Flags:
 * - LC_KEY_SIG => signing keypair
 * - LC_KEY_ENC => encrypt keypair
 */
int lc_keypair_new(lc_keypair_t *key, int flags);

/* manage keyring */
int lc_keyring_init(lc_keyring_t *keyring, size_t nkeys);
void lc_keyring_free(lc_keyring_t *keyring);
int lc_keyring_add(lc_keyring_t *keyring, uint8_t *key);
int lc_keyring_del(lc_keyring_t *keyring, uint8_t *key);

/* return true (-1) if key found in keyring, false (0) if not */
int lc_keyring_has(lc_keyring_t *keyring, uint8_t *key);

/* sign token */
int lc_token_new(lc_token_t *token, lc_keypair_t *signing_key, uint8_t *bearer_key,
	lc_channel_t *chan, uint8_t capbits, uint64_t valid_sec);

/* set sending token for channel */
void lc_channel_token_set(lc_channel_t *chan, lc_token_t *token);

/* apply filter to channel */
void lc_channel_filter_set(lc_channel_t *chan, lc_filter_t *filter);

#endif /* _LIBRECAST_KEY_H */
