/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010.
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 *
 *  Authors:
 *  2004-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl>
 *
 */


#ifndef PUBLIC_VERIFY_X509_DATATYPES
 #define PUBLIC_VERIFY_X509_DATATYPES


#define _GNU_SOURCE

#include <sys/types.h>
#include <sys/stat.h>

#include <openssl/x509.h>
#include <openssl/x509v3.h>
#include <openssl/x509_vfy.h>
#include <openssl/err.h>
#include <openssl/pem.h>

#include <openssl/rsa.h>
#include <openssl/evp.h>
#include <openssl/bio.h>
#include <openssl/des.h>
#include <openssl/rand.h>

#include <openssl/buffer.h>
#include <openssl/objects.h>
#include <openssl/asn1.h>


#define VERIFY_X509_OK X509_V_OK
#define VERIFY_ENABLE  1
#define VERIFY_DISABLE 2


/* verify_x509_setParameter options */
#define  VERIFY_X509_CA_PATH                  (unsigned int)  11000
#define  VERIFY_X509_CERTIFICATE_FILEPATH     (unsigned int)  11001
#define  VERIFY_X509_CERTIFICATE_F_HANDLE     (unsigned int)  11002
#define  VERIFY_X509_CERTIFICATE_PEM          (unsigned int)  11003
#define  VERIFY_X509_PRIVATEKEY_FILE          (unsigned int)  11004
#define  VERIFY_X509_PRIVATEKEY_PEM           (unsigned int)  11005
#define  VERIFY_X509_CRL_PATH                 (unsigned int)  11006
#define  VERIFY_X509_OCSP_RESPONDER_URI       (unsigned int)  11007
#define  VERIFY_X509_STACK_OF_X509            (unsigned int)  12101
#define  VERIFY_X509_EVP_PKEY                 (unsigned int)  12102

#define  VERIFY_X509_OPTIONS_NO_CRL_CHECK             (unsigned int)  30001
#define  VERIFY_X509_OPTIONS_ALLOW_LIMITED_PROXY      (unsigned int)  30002
#define  VERIFY_X509_OPTIONS_MUST_HAVE_PRIV_KEY       (unsigned int)  30004
#define  VERIFY_X509_OPTIONS_REQUIRE_LIMITED_PROXY    (unsigned int)  30005


/* Errors */
#define  ERR_VERIFY_X509_PARAMS_OK                   (unsigned int)      0
#define  ERR_VERIFY_X509_PARAMS_ALREADY_SET          (unsigned int)  20001
#define  ERR_VERIFY_X509_PARAMS_UNSUPPORTED_DATATYPE (unsigned int)  20002
#define  ERR_VERIFY_X509_PARAMS_ACCESS_FAILURE       (unsigned int)  20003
#define  ERR_VERIFY_X509_PARAMS_DATA_INCORRECT       (unsigned int)  20004
#define  ERR_VERIFY_X509_PARAMS_DATA_EMPTY           (unsigned int)  20005
#define  ERR_VERIFY_X509_PARAMS_CONTAINER_FAILURE    (unsigned int)  20006


#define L_ERROR  0  /* errors */
#define L_WARN   1  /* all unusual */
#define L_INFO   2  /* all status changes etc. */
#define L_DEBUG  3  /* all, including trace */


typedef struct intern_discovered_verify_x509_data_s
{
    /* Derived data */
    char * certificate_pem_str;
    char * private_key_pem;
    char * ocsp_responder_uri;

    STACK_OF (X509) * stack_of_x509;
    EVP_PKEY        * evp_pkey;
} internal_discovered_verify_x509_data_t;

typedef struct internal_verify_x509_data_s
{
    /* Assigned data */
    char * capath;
    char * certificate_filepath;
    FILE * certificate_f_handle;
    char * certificate_pem_str;
    char * private_key_filepath;
    char * private_key_pem;
    char * crl_path;
    char * ocsp_responder_uri;
    unsigned short no_crl_check;
    unsigned short allow_limited_proxy;
    unsigned short require_limited_proxy;
    unsigned short must_have_priv_key;

    STACK_OF (X509) * stack_of_x509;
    EVP_PKEY        * evp_pkey;

    unsigned short is_initialized;
    unsigned short is_derived;

    internal_discovered_verify_x509_data_t derived_data;
} internal_verify_x509_data_t;


#define PROXYCERTINFO_OID      "1.3.6.1.5.5.7.1.14"
#define OLD_PROXYCERTINFO_OID  "1.3.6.1.4.1.3536.1.222"

#define OID_GLOBUS_PROXY_V2    PROXYCERTINFO_OID
#define OID_GLOBUS_PROXY_V3    OLD_PROXYCERTINFO_OID
#define OID_RFC_PROXY          "1.3.6.1.5.5.7.1.14"



typedef enum
        {
          NONE              = 0,
          CA                = 1,
          EEC               = 2,
          GT2_PROXY         = 4,
          RFC_PROXY         = 8,
          GT2_LIMITED_PROXY = 16,
          RFC_LIMITED_PROXY = 32,
          GT3_PROXY         = 64,
          GT3_LIMITED_PROXY = 128
        } lcmaps_proxy_type_t;


#include <openssl/x509.h>
#include <openssl/x509v3.h>
#include <string.h>

#define ANY_LANGUAGE_OID         "1.3.6.1.5.5.7.21.0"
#define ANY_LANGUAGE_SN          "ANY_LANGUAGE"
#define ANY_LANGUAGE_LN          "Any Language"

#define IMPERSONATION_PROXY_OID         "1.3.6.1.5.5.7.21.1"
#define IMPERSONATION_PROXY_SN          "IMPERSONATION_PROXY"
#define IMPERSONATION_PROXY_LN          "GSI impersonation proxy"

#define INDEPENDENT_PROXY_OID           "1.3.6.1.5.5.7.21.2"
#define INDEPENDENT_PROXY_SN            "INDEPENDENT_PROXY"
#define INDEPENDENT_PROXY_LN            "GSI independent proxy"

#define LIMITED_PROXY_OID               "1.3.6.1.4.1.3536.1.1.1.9"
#define LIMITED_PROXY_SN                "LIMITED_PROXY"
#define LIMITED_PROXY_LN                "GSI limited proxy"

/* Used for error handling */
#define ASN1_F_PROXYPOLICY_NEW          450
#define ASN1_F_D2I_PROXYPOLICY          451

/**
 * @ingroup proxypolicy
 *
 * @note NOTE: The API provides functions to manipulate
 * the fields of a PROXYPOLICY.  Accessing the fields
 * directly will not work.
 *
 * This typedef maintains information about the policies
 * that have been placed on a proxy certificate
 *
 * @param policy_language defines which policy language
 * is to be used to define the policies
 * @param policy the policy that determines the policies
 * on a certificate
 */
struct PROXYPOLICY_st
{
    ASN1_OBJECT *                       policy_language;
    ASN1_OCTET_STRING *                 policy;
};

typedef struct PROXYPOLICY_st PROXYPOLICY;

DECLARE_STACK_OF(PROXYPOLICY)
DECLARE_ASN1_SET_OF(PROXYPOLICY)

/*
 * This typedef maintains information about a proxy
 * certificate.
 *
 * @note NOTE: The API provides functions to manipulate
 * the fields of a PROXYCERTINFO.  Accessing the fields
 * directly is not a good idea.
 *
 *
 * @param path_length an optional field in the ANS.1 DER encoding,
 * it specifies the maximum depth of the path of Proxy Certificates
 * that can be signed by this End Entity Certificate or Proxy Certificate.
 * @param policy a non-optional field in the ANS.1 DER encoding,
 * specifies policies on the use of this certificate.
 */
struct PROXYCERTINFO_st
{
    ASN1_INTEGER *                      path_length;       /* [ OPTIONAL ] */
    PROXYPOLICY *                       policy;
};

typedef struct PROXYCERTINFO_st PROXYCERTINFO;

DECLARE_STACK_OF(PROXYCERTINFO)
DECLARE_ASN1_SET_OF(PROXYCERTINFO)

#endif /* PUBLIC_VERIFY_X509_DATATYPES */
