; C Library
;
; Copyright (C) 2025 Kestrel Institute (http://www.kestrel.edu)
;
; License: A 3-clause BSD license. See the LICENSE file distributed with ACL2.
;
; Author: Alessandro Coglio (www.alessandrocoglio.info)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package "C$")

(include-book "unambiguity")

(include-book "std/util/error-value-tuples" :dir :system)

(local (include-book "std/alists/top" :dir :system))

(local (in-theory (enable* abstract-syntax-unambp-rules)))

(local (include-book "kestrel/built-ins/disable" :dir :system))
(local (acl2::disable-most-builtin-logic-defuns))
(local (acl2::disable-builtin-rewrite-rules-for-defaults))
(set-induction-depth-limit 0)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defxdoc+ disambiguator
  :parents (syntax-for-tools)
  :short "Disambiguator of the C abstract syntax for tools."
  :long
  (xdoc::topstring
   (xdoc::p
    "As discussed in the "
    (xdoc::seetopic "abstract-syntax" "abstract syntax")
    ", the syntax of C is inherently syntactically ambiguous,
     in ways that can be disambiguated only via a (static) semantic analysis.
     Here we define such semantic analysis, which we call `disambiguator'.
     The disambiguator does not check the full static validity of the code,
     but it performs sufficient checks to disambiguate
     the abstract syntax trees generated by the @(see parser).
     The disambiguator transforms the abstract syntax trees,
     generating unambiguous ones.")
   (xdoc::p
    "The abstract syntax trees generated by the parser
     contains the following ambiguities,
     which the disambiguator resolves as indicated:")
   (xdoc::ul
    (xdoc::li
     "Identifiers used as expressions
      are always classified as identifier expressions
      (i.e. the @(':ident') case of @(tsee expr))
      by the parser,
      but some of them may be enumeration constants.
      The disambiguator re-classifies the latter as required.")
    (xdoc::li
     "Some constructs include ambiguous expressions or type names,
      represented by the type @(tsee amb-expr/tyname).
      The disambiguator turns those constructs into unambiguous ones,
      by choosing either an expression or a type name.")
    (xdoc::li
     "Some constructs include ambiguous declarators or abstract declarators,
      represented by the type @(tsee amb-declor/absdeclor).
      The disambiguator turns those constructs into unambiguous ones,
      by choosing either a declarator or an abstract declarator.")
    (xdoc::li
     "Some constructs include ambiguous declarations or statements,
      represented by the type @(tsee amb-decl/stmt).
      The disambiguator turns those constructs into unambiguous ones,
      by choosing either a declaration or a statement.")
    (xdoc::li
     "Function declarators whose parameters are all identifiers
      are always classified as parameter type lists by the parser,
      but some of them may be identifier lists.
      The disambiguator re-classifies the latter as required.")
    (xdoc::li
     "Some expressions may be cast expressions or binary expressions,
      represented by the @(':cast/...-ambig') cases of @(tsee expr).
      The disambiguator turns these into
      unambiguous case or binary expressions.")
    (xdoc::li
     "The initializing part of a @('for') loop may be ambiguously
      a declaration or an expression followed by a semicolon,
      represented by the @(':for-ambig') case of @(tsee stmt).
      The disambiguator turns these ambiguous @('for') loops
      into unambiguous ones."))
   (xdoc::p
    "The disambiguator does not perform a full (static) semantic analysis,
     but only a light-weight one, enough for disambiguation.
     If the code is statically valid, the disambiguation must work.
     If the code is not statically valid, the disambiguation may fail;
     in some cases, the disambiguator reports the cause of invalidity,
     but in other cases it may not have enough information.")
   (xdoc::p
    "The disambiguation is performed by scanning the code,
     transforming it as applicable (re-classifying or choosing constructs).
     This needs certain information about the identifiers in the code,
     which we build and use.
     For instance, when we encounter an enumeration specifier,
     we need to record the information that the identifiers of the enumerators
     are enumeration constants,
     so that, if we later encounter an expression with that identifier,
     we can re-classify from an identifier expression to an enumeration constant
     (this is the first of the ambiguities listed above).
     In essence, we need a symbol table of identifiers;
     not a full one that would be needed to check the full validity of the code,
     but one with sufficient information to disambiguate.
     We need to take into account the scoping rules of C of course,
     since the same identifier
     may have different meaning in different scopes.
     We call these symbol tables `disambiguation tables'.")
   (xdoc::p
    "We use "
    (xdoc::seetopic "acl2::error-value-tuples" "error-value tuples")
    " to handle errors in the disambiguator.")
   (xdoc::p
    "The fixtype and functions in the implementation of the disambiguator
     are prefixed by @('dimb'), which stands for `DIsaMBiguator'.")
   (xdoc::p
    "For now we leave the GCC extensions unchanged,
     i.e. we do not apply the disambiguation to them,
     even though they may contain expression.
     Some initial experiments reveal that we will need to
     treat these GCC extensions in a more dedicated way.
     For instance, the @('access') attribute may include
     the arguments @('read_only'), @('write_only'), and @('read_write').
     Grammatically these are expressions,
     but it seems that they have just a special meaning, like keywords,
     in the context of the @('access') attribute,
     but they are likely not keywords elsewhere.
     A naive treatment would attempt to resolve those arguments,
     which are not in the disambiguation table."))
  :order-subtopics t
  :default-parent t)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(fty::deftagsum dimb-kind
  :short "Fixtype of kinds of identifiers in disambiguation tables."
  :long
  (xdoc::topstring
   (xdoc::p
    "For each identifier in the disambiguation table,
     we track whether the identifier
     (i) a @('typedef') name, or
     (ii) an object or function, or
     (iii) an enumeration constant.
     Not all identifiers fit these categories;
     for instance, structure and union tags correspond to none of the above.
     However, for disambiguation purposes,
     we do not need to track all identifiers,
     but just the kinds just listed.
     Note also that we do not need to distinguish between objects and functions,
     as they are treated the same for disambiguation purposes."))
  (:typedef ())
  (:objfun ())
  (:enumconst ())
  :pred dimb-kindp)

;;;;;;;;;;

(defirrelevant irr-dimb-kind
  :short "An irrelevant kind of identifiers in disambiguation tables."
  :type dimb-kindp
  :body (dimb-kind-typedef))

;;;;;;;;;;;;;;;;;;;;

(fty::defoption dimb-kind-option
  dimb-kind
  :short "Fixtype of optional kinds of identifiers in disambiguation tables."
  :long
  (xdoc::topstring
   (xdoc::p
    "Kinds of identifiers in disambiguation tables
     are defined in @(tsee dimb-kind)."))
  :pred dimb-kind-optionp)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(fty::defalist dimb-scope
  :short "Fixtype of scopes in disambiguation tables."
  :long
  (xdoc::topstring
   (xdoc::p
    "An identifier may have different meanings in different scopes,
     but it must have one meaning within the same scope.
     Thus, we represent scopes as alists from identifiers to their kinds."))
  :key-type ident
  :val-type dimb-kind
  :true-listp t
  :keyp-of-nil nil
  :valp-of-nil nil
  :pred dimb-scopep
  :prepwork ((set-induction-depth-limit 1))

  ///

  (defrule dimb-kindp-of-cdr-of-assoc-equal-when-dimb-scopep
    (implies (dimb-scopep scope)
             (iff (dimb-kindp (cdr (assoc-equal ident scope)))
                  (assoc-equal ident scope)))
    :induct t
    :enable (dimb-scopep assoc-equal)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(fty::deflist dimb-table
  :short "Fixtype of disambiguation tables."
  :long
  (xdoc::topstring
   (xdoc::p
    "A disambiguation table associates kinds to identifiers,
     with the mapping organized into scopes.
     Starting from the outer scope,
     i.e. the file scope [C17:6.2.1/4],
     traversing the code enters and exits
     block scopes and prototype scopes [C17:6.2.1/4],
     in a stack-like fashion.
     So we represent a disambiguation table as
     a list of disambiguation scopes.")
   (xdoc::p
    "The list should be never empty,
     but using @(':non-emptyp t') in this fixtype
     generates a false subgoal.
     It should be possible to improve @(tsee fty::deflist)
     to work with this option in this fixtype,
     but for now we just allow empty lists in the fixtype."))
  :elt-type dimb-scope
  :true-listp t
  :elementp-of-nil t
  :pred dimb-tablep
  :prepwork ((local (in-theory (enable nfix)))))

;;;;;;;;;;;;;;;;;;;;

(defirrelevant irr-dimb-table
  :short "An irrelevant disambiguation table."
  :type dimb-tablep
  :body nil)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define dimb-init-table ()
  :returns (table dimb-tablep)
  :short "Initial disambiguation table."
  :long
  (xdoc::topstring
   (xdoc::p
    "This consists of a single empty scope,
     which is the file scope.
     We use one disambiguation table for each translation unit."))
  (list nil))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define dimb-push-scope ((table dimb-tablep))
  :returns (new-table dimb-tablep)
  :short "Push a scope into the disambiguation table."
  :long
  (xdoc::topstring
   (xdoc::p
    "We add an empty scope.
     The top of the stack is on the left,
     so we push via @(tsee cons).
     Also see @(tsee dimb-pop-scope)."))
  (cons nil (dimb-table-fix table))
  :hooks (:fix))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define dimb-pop-scope ((table dimb-tablep))
  :returns (new-table dimb-tablep)
  :short "Pop a scope from the disambiguation table."
  :long
  (xdoc::topstring
   (xdoc::p
    "It is an internal error if the table is empty;
     it should never be empty.
     We should replace this with guards and proofs.")
   (xdoc::p
    "We remove the top scope, via @(tsee cdr).
     Recall that the stack top is on the left;
     see @(tsee dimb-push-scope)."))
  (if (consp table)
      (dimb-table-fix (cdr table))
    (raise "Internal error: empty disambiguation table."))
  :hooks (:fix))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define dimb-lookup-ident ((ident identp) (table dimb-tablep))
  :returns (kind? dimb-kind-optionp)
  :short "Look up an identifier in the disambiguation table."
  :long
  (xdoc::topstring
   (xdoc::p
    "According the visibility and hiding rules [C17:6.2.1/2],
     we look up the identifier starting from the innermost scope.
     We stop as soon as we find a match.
     We return @('nil') if we reach the outermost scope
     without finding a match."))
  (b* (((when (endp table)) nil)
       (scope (dimb-scope-fix (car table)))
       (ident+kind (assoc-equal (ident-fix ident) scope))
       ((when ident+kind) (dimb-kind-fix (cdr ident+kind))))
    (dimb-lookup-ident ident (cdr table)))
  :hooks (:fix))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define dimb-add-ident ((ident identp) (kind dimb-kindp) (table dimb-tablep))
  :returns (new-table dimb-tablep)
  :short "Add an identifier and its kind to the disambiguation table."
  :long
  (xdoc::topstring
   (xdoc::p
    "It is an internal error if the table is empty;
     it should never be empty.
     We should replace this with guards and proofs.")
   (xdoc::p
    "We add the identifier to the innermost (i.e. top) scope.
     If the identifier is already in the innermost scope,
     we override its mapping with the new one.
     This is necessary to handle re-declarations of the same identifier,
     which are allowed under suitable conditions;
     in particular, the kind of the identifier should not change.
     But we override the mapping unconditionally,
     even if the new kind differs from the old kind:
     this situation should only happen with invalid code,
     in which case it does not matter how we disambiguate it exactly."))
  (b* (((when (endp table))
        (raise "Internal error: empty disambiguation table."))
       (scope (dimb-scope-fix (car table)))
       (new-scope (acons (ident-fix ident) (dimb-kind-fix kind) scope))
       (new-table (cons new-scope (cdr table))))
    (dimb-table-fix new-table))
  :guard-hints (("Goal" :in-theory (enable acons)))
  :hooks (:fix))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define dimb-add-ident-objfun ((ident identp) (table dimb-tablep))
  :returns (new-table dimb-tablep)
  :short "Add an identifier to the disambiguation table,
          with object or function kind."
  (dimb-add-ident ident (dimb-kind-objfun) table)
  :hooks (:fix))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define dimb-add-idents-objfun ((idents ident-listp) (table dimb-tablep))
  :returns (new-table dimb-tablep)
  :short "Add all the identifiers in a list to the disambiguation table,
          with object or function kind."
  (cond ((endp idents) (dimb-table-fix table))
        (t (dimb-add-idents-objfun (cdr idents)
                                   (dimb-add-ident-objfun (car idents)
                                                          table))))
  :hooks (:fix))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define dimb-make/adjust-expr-cast ((type tynamep) (arg exprp))
  :guard (and (tyname-unambp type)
              (expr-unambp arg))
  :returns (expr exprp)
  :short "Build, and adjust if needed, a cast expression."
  :long
  (xdoc::topstring
   (xdoc::p
    "This is used to build or adjust cast expression,
     in @(tsee dimb-expr) and other functions.
     When @(tsee dimb-expr) encounters a cast expression,
     it recursively disambiguates
     the type name @('T0') and the argument expression @('E0'),
     obtaining a type name @('T') and an argument expression @('E').
     Then it should generally return the cast expression @('(T) E'),
     but there are cases in which an adjustment is needed.
     Suppose that @('E0') was an ambiguous cast/binary expression,
     which gets disambiguated to a binary expression @('A op B').
     Then it means that the original code was @('(T0) A op B'),
     which is a binary expression
     whose first operand is the cast expression @('(T0) A')
     and whose second operand is the expression @('B').
     Thus, it would be incorrect for @(tsee dimb-expr)
     to return a cast expression, which would be @('(T) (A op B)').
     Therefore, @(tsee dimb-expr) uses this function defined here
     to combine @('T') and @('E') properly, and return the result.")
   (xdoc::p
    "Note that @('A') may be itself a binary expression, e.g. @('A1 op\' A2').
     Thus, this function must be called recursively
     when attempting to make a cast out of @('T') and @('A'),
     because it may need to be adjusted to
     a binary expression with @'op\''),
     pushing the cast down.")
   (xdoc::p
    "The same kind of adjustment may be needed, besides in @(tsee dimb-expr),
     also in functions like @(tsee dimb-cast/call-to-cast),
     which also normally build cast expressions.")
   (xdoc::p
    "This function takes @('T') and @('E') as inputs.
     If @('E') is not a binary expression,
     we return the cast expression @('(T) E').
     If @('E') is a binary expression, let that be @('A op B'),
     then we return @('[ TA ] op B'),
     where TA is the result of recursively calling this function
     on @('T') and @('A'),
     and where the square brackets show how things are grouped.")
   (xdoc::p
    "In other words, this builds and adjusts the expression
     so that the sub-expressions have priorities greater than or equal to
     the ones expected by the super-expressions at those place.
     Another way to express this condition on priorities
     is that the expression prints without any added parentheses.
     But this is not a function to adjust all kinds of priority mismatches:
     it only works on the ones that may arise during disambiguation."))
  (if (expr-case arg :binary)
      (make-expr-binary :op (expr-binary->op arg)
                        :arg1 (dimb-make/adjust-expr-cast
                               type (expr-binary->arg1 arg))
                        :arg2 (expr-binary->arg2 arg))
    (make-expr-cast :type type :arg arg))
  :measure (expr-count arg)
  :hints (("Goal" :in-theory (enable o< o-finp)))
  :verify-guards :after-returns
  :hooks (:fix)

  ///

  (defret expr-unambp-of-dimb-make/adjust-expr-cast
    (expr-unambp expr)
    :hyp (and (tyname-unambp type)
              (expr-unambp arg))
    :hints (("Goal" :induct t))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define dimb-make/adjust-expr-binary ((op binopp) (arg1 exprp) (arg2 exprp))
  :guard (and (expr-unambp arg1)
              (expr-unambp arg2))
  :returns (expr exprp)
  :short "Build, and adjust if needed, a binary expression."
  :long
  (xdoc::topstring
   (xdoc::p
    "This has a similar purpose to @(tsee dimb-make/adjust-expr-cast):
     see that function's documentation first.
     The same kind of incorrect grouping
     (prevented by the use of @(tsee dimb-make/adjust-expr-cast))
     may arise with a binary expression @('A0 op B0').
     When encountering such a binary expression,
     @(tsee dimb-expr) first disambiguates
     @('A0') to @('A') and @('B0') to @('B').
     But if @('A0') and/or @('B0') were ambiguous cast/binary expressions,
     and one or both get disambiguated to binary expressions,
     returning the expression @('A op B') could be incorrect.
     For instance,
     @('op') could be @('*'),
     @('A') could be a variable @('x'),
     and @('B') could be the expression @('(y) & z')
     where @('y') and @('z') are variables.
     That is, @('A op B') is @('x * (y) & z'),
     which must be grouped as @('[ x * (y) ] & z')
     (where the square brackets indicate the grouping),
     and not as @('x * [ (y) & z ]').
     The cue is that the priority of @('(y) & z') is
     lower than the expected priority of the right operand of @('*'):
     since our parser preserves parenthesized expressions,
     it should be the case that, in the AST,
     each sub-expression has priority greater than or equal to
     the expected priority by the super-expression at that place.
     So the adjustment performed by this function,
     like the one performed by @(tsee dimb-make/adjust-expr-cast),
     can be seen as restoring that property of ASTs.")
   (xdoc::p
    "This function assumes that @('arg1') and @('arg2')
     already satisfy the property about priorities mentioned above,
     which is in fact the case when this function is called.
     This function attempts to construct the binary expression,
     but adjusts it so that the prioperty on priorities will hold.
     We compare the expected priorities
     for the left and right operand of @('op')
     with the actual priorities of @('arg1') and @('arg2').
     Since each may match or mismatch, there are four possibilities.
     This may be a bit more general than needed in the disambiguator,
     because for instance the expression @('(x) + y * (z) & w')
     could not lead, without adjustment, to @('[ (x) + y ] * [ (z) & w ]'),
     because the parser would parse all of @('y * (z) & w') after the @('+'),
     but the generality is easies to handle,
     compared to establishing restrictions on what the parser can produce.
     So we consider a case like @('[ (x) + y ] * [ (z) & w ]') possible,
     with both sub-expressions mismatching,
     in the sense of having priority lower than expected by @('op').")
   (xdoc::p
    "A mismatch should be possible only if the sub-expression is a binary one,
     because the only decrease in priority during disambiguation
     can happen when the lower-priority expression is a binary one.
     We throw a hard error if that is not the case,
     which we expect to never happen.")
   (xdoc::p
    "If there is no mismatch, we just build the binary expression.")
   (xdoc::p
    "If only one sub-expression mismatches,
     we perform the adjustment by moving up the sub-expression's operator
     and redirecting one of its operands to the lowered operator.
     For instance, consider the adjustment
     from @('A * [ B + C ]') to @('[ A * B ] + C'):")
   (xdoc::codeblock
    "   *               +    "
    "  / \             / \   "
    " A   +    --->   *   C  "
    "    / \         / \     "
    "   B   C       A   B    ")
   (xdoc::p
    "where the @('+') is moved up and
     its operand @('B') is redirected under the lowered @('*').
     The situation is symmetric if the expression is flipped,
     i.e. if the mismatch is in the left operand instead of the right one.")
   (xdoc::p
    "But note that, in the example above,
     @('B') itself could be a @('+') expression,
     which is now under @('*'), and therefore needs to be adjusted.
     So the adjustment may need to be recursive:
     we recursively call this function to build the new binary sub-expression.")
   (xdoc::p
    "If both sub-expressions mismatch,
     we compare the priorities of the two sub-expressions,
     and we perform the adjustment as if only the lower-priority one mismatches.
     For instance, consider @('[ A & B ] * [ C + D ]'):
     since @('&') has lower priority than @('+'),
     we move the @('&') up, not the @('+').
     This results in @('A & [ B * [ C + D ] ]'),
     where the @('[ B * [ C + D ] ]') is recursively adjusted:
     this further adjustment moves up the @('+'),
     but it does not need to go above the @('&'),
     which has lower priority;
     the result is then @('A & [ [ B * C] + D ]').")
   (xdoc::p
    "The case in which the two mismatching sub-expressions
     have the same priority is handled in the same way as
     when the right one has lower priority.
     The reason for this asymmetry is that
     all the binary operations involved are left-associative.
     For instance, consider @('[ A + B ] * [ C + D ]').
     The correct adjustment is @('[ A + [ B * C ] ] + D'),
     not @('[A + [ [ B * C ] + D ]')."))
  (b* (((mv arg1-expected arg2-expected) (binop-expected-priorities op))
       (arg1-actual (expr->priority arg1))
       (arg2-actual (expr->priority arg2))
       (arg1-mismatch (expr-priority-< arg1-actual arg1-expected))
       (arg2-mismatch (expr-priority-< arg2-actual arg2-expected))
       ((when (and arg1-mismatch
                   (not (expr-case arg1 :binary))))
        (raise "Internal error: ~
                non-binary expression ~x0 ~
                used as left argument of binary operator ~x1."
               (expr-fix arg1) (binop-fix op))
        (expr-binary op arg1 arg2))
       ((when (and arg2-mismatch
                   (not (expr-case arg2 :binary))))
        (raise "Internal error: ~
                non-binary expression ~x0 ~
                used as right argument of binary operator ~x1."
               (expr-fix arg2) (binop-fix op))
        (expr-binary op arg1 arg2)))
    (cond
     ((and arg1-mismatch
           (or (not arg2-mismatch)
               (expr-priority-> arg1-actual arg2-actual)))
      (b* ((new-op (expr-binary->op arg1))
           (new-arg1 (expr-binary->arg1 arg1))
           (new-arg2 (dimb-make/adjust-expr-binary op
                                                   (expr-binary->arg2 arg1)
                                                   arg2)))
        (make-expr-binary :op new-op :arg1 new-arg1 :arg2 new-arg2)))
     ((and arg2-mismatch
           (or (not arg1-mismatch)
               (expr-priority-<= arg1-actual arg2-actual)))
      (b* ((new-op (expr-binary->op arg2))
           (new-arg1 (dimb-make/adjust-expr-binary op
                                                   arg1
                                                   (expr-binary->arg1 arg2)))
           (new-arg2 (expr-binary->arg2 arg2)))
        (make-expr-binary :op new-op :arg1 new-arg1 :arg2 new-arg2)))
     (t (make-expr-binary :op op :arg1 arg1 :arg2 arg2))))
  :measure (+ (expr-count arg1) (expr-count arg2))
  :hints (("Goal" :in-theory (enable o-p o< o-finp)))
  :verify-guards :after-returns
  :hooks (:fix)

  ///

  (defret expr-unambp-of-dimb-make/adjust-expr-binary
    (expr-unambp expr)
    :hyp (and (expr-unambp arg1)
              (expr-unambp arg2))
    :hints (("Goal" :induct t))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define dimb-make/adjust-expr-unary ((op unopp) (arg exprp))
  :guard (expr-unambp arg)
  :returns (expr exprp)
  :short "Build, and adjust if needed, a unary expression."
  :long
  (xdoc::topstring
   (xdoc::p
    "This is similar to @(tsee dimb-make/adjust-expr-cast)
     and @(tsee dimb-make/adjust-expr-binary).
     Since some of the unary operators expect a cast expression as argument
     (as well as an expression with priority higher than a cast),
     the argument of a unary operators, as produced by the parser,
     may be an ambiguous cast expression that becomes a binary expression,
     which has therefore lower priority than a cast expression.
     This makes it necessary to adjust the expression
     so that the actual priorities match the expected priorities
     (where `matching' includes being higher, besides being equal).")
   (xdoc::p
    "For instance, @('~ (a) + b') is parsed as
     the unary operator @('~') applied to the ambiguous @('(a) + b'),
     which may be disambiguated to a binary expression,
     thus resulting (without adjustment) in @('~ [ (a) + b ]'),
     where the square brackets indicate grouping.
     Instead, the correct expression is @('[ ~ (a) ] + b').")
   (xdoc::p
    "So this function builds, and adjusts if needed,
     a unary expression out of the operator @('op') and operand @('A').
     If @('A') has priority greater than or equal to
     the expected operand priority of @('op'),
     we just build and return the unary expression.
     Otherwise, @('A') must be a binary expression, say @('A1 op\' A2').
     because this is the only kind that may arise from disambiguation
     with a lower priority than a cast expression;
     we throw a hard error if that is not the case,
     which we never expect to happen.
     To adjust the expression,
     we push the unary operator into the left operand,
     i.e. @('[ op A1 ] op\' A2').")
   (xdoc::p
    "Note that @('A1') may be itself a binary expression,
     so we need to call this function recursively."))
  (b* ((arg-expected (if (member-eq (unop-kind op)
                                    '(:predec :preinc
                                      :postdec :postinc
                                      :sizeof))
                         (expr-priority-unary)
                       (expr-priority-cast)))
       (arg-actual (expr->priority arg))
       ((when (expr-priority->= arg-actual arg-expected))
        (make-expr-unary :op op :arg arg))
       ((unless (expr-case arg :binary))
        (raise "Internal error: ~
                non-binary expression ~x0 ~
                used as argument of unary operator ~x1."
               (expr-fix arg) (unop-fix op))
        (expr-unary op arg)))
    (make-expr-binary :op (expr-binary->op arg)
                      :arg1 (dimb-make/adjust-expr-unary
                             op (expr-binary->arg1 arg))
                      :arg2 (expr-binary->arg2 arg)))
  :measure (expr-count arg)
  :hints (("Goal" :in-theory (enable o< o-finp)))
  :verify-guards :after-returns
  :hooks (:fix)

  ///

  (defret expr-unambp-of-dimb-make/adjust-expr-unary
    (expr-unambp expr)
    :hyp (expr-unambp arg)
    :hints (("Goal" :induct t))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define dimb-cast/call-to-cast ((tyname tynamep)
                                (inc/dec inc/dec-op-listp)
                                (arg exprp))
  :guard (and (tyname-unambp tyname)
              (expr-unambp arg))
  :returns (cast-expr exprp)
  :short "Disambiguate an ambiguous cast or call expression
          to be a cast expression."
  :long
  (xdoc::topstring
   (xdoc::p
    "The form @('(X) IncDec (E)Pr') of an ambiguous call or cast expression
     is described in detail in @(tsee expr): refer to that description.
     This ambiguous expression is disambiguated in @(tsee dimb-expr),
     by first disambiguating whether @('X') is a type name or an expression.
     Based on that, the constituents of that ambiguous expression
     must be re-arranged into an unambiguous expression:
     this is done by this function,
     for the case in which @('X') is a type name.
     The case in which @('X') is an expression
     is handled in @(tsee dimb-cast/call-to-call).")
   (xdoc::p
    "In this function,
     @('tyname') is @('X'),
     @('inc/dec') is @('IncDec'), and
     @('arg') is @('(E)Pr').
     If @('X') is a type name,
     the increment and decrement operators, if any,
     are pre-increment and pre-decrement operators
     applied to the expression @('(E)Pr').
     We apply them and we form a cast expression."))
  (dimb-make/adjust-expr-cast tyname
                              (apply-pre-inc/dec-ops inc/dec arg))
  :hooks (:fix)

  ///

  (defret expr-unambp-of-dimb-cast/call-to-cast
    (expr-unambp cast-expr)
    :hyp (and (tyname-unambp tyname)
              (expr-unambp arg))))

;;;;;;;;;;;;;;;;;;;;

(define dimb-cast/call-to-call ((fun exprp)
                                (inc/dec inc/dec-op-listp)
                                (rest exprp))
  :guard (and (expr-unambp fun)
              (expr-unambp rest))
  :returns (call-expr exprp)
  :short "Disambiguate an ambiguous cast or call expression
          to be a call expression."
  :long
  (xdoc::topstring
   (xdoc::p
    "The form @('(X) IncDec (E)Pr') of an ambiguous call or cast expression
     is described in detail in @(tsee expr): refer to that description.
     This ambiguous expression is disambiguated in @(tsee dimb-expr),
     by first disambiguating whether @('X') is a type name or an expression.
     Based on that, the constituents of that ambiguous expression
     must be re-arranged into an unambiguous expression:
     this is done by this function,
     for the case in which @('X') is an expression.
     The case in which @('X') is a type name
     is handled in @(tsee dimb-cast/call-to-call).")
   (xdoc::p
    "In this function,
     @('fun') is @('X'),
     @('inc/dec') is @('IncDec'), and
     @('rest') is @('(E)Pr').
     If @('X') is an expression,
     the increment and decrement operators, if any,
     are post-increment and post-decrement operators
     applied to the expression @('X').
     We apply them and we form a call expression.
     To do so, we need to separate @('(E)') and @('Pr'),
     turn @('(E)') into @('(E1,...,En)') according to the comma operators,
     and then apply the @('Pr') to the call @('(X)(E1,...,En)').")
   (xdoc::p
    "We perform the latter transformation via a recursion,
     because we need to go through the individual postfix constructs of @('Pr'),
     till we reach @('(E)'),
     and then we re-apply the postfix constructs of @('Pr')."))
  (b* ((fun (expr-paren fun))
       (fun (apply-post-inc/dec-ops fun inc/dec)))
    (dimb-cast/call-to-call-loop fun rest))
  :hooks (:fix)

  :prepwork

  ((define dimb-cast/call-to-call-loop ((fun exprp) (rest exprp))
     :guard (and (expr-unambp fun)
                 (expr-unambp rest))
     :returns (new-expr exprp)
     :parents nil
     (b* (((when (expr-case rest :paren))
           (b* ((args (expr-to-asg-expr-list (expr-paren->inner rest))))
             (make-expr-funcall :fun fun :args args)))
          ((when (expr-case rest :arrsub))
           (b* ((expr
                 (dimb-cast/call-to-call-loop fun (expr-arrsub->arg1 rest))))
             (make-expr-arrsub :arg1 expr :arg2 (expr-arrsub->arg2 rest))))
          ((when (expr-case rest :funcall))
           (b* ((expr
                 (dimb-cast/call-to-call-loop fun (expr-funcall->fun rest))))
             (make-expr-funcall :fun expr :args (expr-funcall->args rest))))
          ((when (expr-case rest :member))
           (b* ((expr
                 (dimb-cast/call-to-call-loop fun (expr-member->arg rest))))
             (make-expr-member :arg expr :name (expr-member->name rest))))
          ((when (expr-case rest :memberp))
           (b* ((expr
                 (dimb-cast/call-to-call-loop fun (expr-memberp->arg rest))))
             (make-expr-memberp :arg expr :name (expr-memberp->name rest)))))
       (prog2$
        (raise "Internal error: unexpected expression ~x0." (expr-fix fun))
        (irr-expr)))
     :measure (expr-count rest)
     :hints (("Goal" :in-theory (enable o< o-finp)))
     :verify-guards :after-returns
     :hooks (:fix)

     ///

     (defret expr-unambp-of-dimb-cast/call-to-call-loop
       (expr-unambp new-expr)
       :hyp (and (expr-unambp fun)
                 (expr-unambp rest))
       :hints (("Goal" :induct t :in-theory (enable irr-expr))))))

  ///

  (defret expr-unambp-of-dimb-cast/call-to-call
    (expr-unambp call-expr)
    :hyp (and (expr-unambp fun)
              (expr-unambp rest))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define dimb-cast/mul-to-cast ((tyname tynamep)
                               (inc/dec inc/dec-op-listp)
                               (arg exprp))
  :guard (and (tyname-unambp tyname)
              (expr-unambp arg))
  :returns (cast-expr exprp)
  :short "Disambiguate an ambiguous cast or multiplication expression
          to be a cast expression."
  :long
  (xdoc::topstring
   (xdoc::p
    "This is analogous in purpose to @(tsee dimb-cast/call-to-cast),
     but for a different kind of ambiguous expression.
     Note that the @('*'), which is unary in this disambiguation,
     is implicit in the abstract syntax of the ambiguous expression."))
  (dimb-make/adjust-expr-cast tyname
                              (dimb-make/adjust-expr-unary
                               (unop-indir)
                               (apply-pre-inc/dec-ops inc/dec arg)))
  :hooks (:fix)

  ///

  (defret expr-unambp-of-dimb-cast/mul-to-cast
    (expr-unambp cast-expr)
    :hyp (and (tyname-unambp tyname)
              (expr-unambp arg))))

;;;;;;;;;;;;;;;;;;;;

(define dimb-cast/mul-to-mul ((arg1 exprp)
                              (inc/dec inc/dec-op-listp)
                              (arg2 exprp))
  :guard (and (expr-unambp arg1)
              (expr-unambp arg2))
  :returns (mul-expr exprp)
  :short "Disambiguate an ambiguous cast or multiplication expression
          to be a multiplication expression."
  :long
  (xdoc::topstring
   (xdoc::p
    "This is analogous in purpose to @(tsee dimb-cast/call-to-call),
     but for a different kind of ambiguous expression.
     Note that the @('*'), which is binary in this disambiguation,
     is implicit in the abstract syntax of the ambiguous expression."))
  (dimb-make/adjust-expr-binary (binop-mul)
                                (apply-post-inc/dec-ops arg1 inc/dec)
                                arg2)
  :hooks (:fix)

  ///

  (defret expr-unambp-of-dimb-cast/mul-to-mul
    (expr-unambp mul-expr)
    :hyp (and (expr-unambp arg1)
              (expr-unambp arg2))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define dimb-cast/addsub-to-cast ((tyname tynamep)
                                  (inc/dec inc/dec-op-listp)
                                  (arg exprp)
                                  (plus/minus unopp))
  :guard (and (tyname-unambp tyname)
              (expr-unambp arg)
              (or (unop-case plus/minus :plus)
                  (unop-case plus/minus :minus)))
  :returns (expr exprp)
  :short "Disambiguate an ambiguous cast or addition/subtraction expression
          to be a cast expression."
  :long
  (xdoc::topstring
   (xdoc::p
    "This is analogous in purpose to @(tsee dimb-cast/call-to-cast),
     but for a different kind of ambiguous expression,
     actually two kinds, which are very similar and thus handled together;
     the two kinds are selected by the unary operator passed as input.
     Note that the @('+') or @('-'), which is unary in this disambiguation,
     is implicit in the abstract syntax of the ambiguous expression."))
  (dimb-make/adjust-expr-cast tyname
                              (dimb-make/adjust-expr-unary
                               plus/minus
                               (apply-pre-inc/dec-ops inc/dec arg)))
  :hooks (:fix)

  ///

  (defret expr-unambp-of-dimb-cast/addsub-to-cast
    (expr-unambp expr)
    :hyp (and (tyname-unambp tyname)
              (expr-unambp arg))))

;;;;;;;;;;;;;;;;;;;;

(define dimb-cast/addsub-to-addsub ((arg1 exprp)
                                    (inc/dec inc/dec-op-listp)
                                    (arg2 exprp)
                                    (add/sub binopp))
  :guard (and (expr-unambp arg1)
              (expr-unambp arg2)
              (or (binop-case add/sub :add)
                  (binop-case add/sub :sub)))
  :returns (expr exprp)
  :short "Disambiguate an ambiguous cast or addition/subtraction expression
          to be an addition or subtraction expression."
  :long
  (xdoc::topstring
   (xdoc::p
    "This is analogous in purpose to @(tsee dimb-cast/call-to-call),
     but for a different kind of ambiguous expression,
     actually two kinds, which are very similar and thus handled together;
     the two kinds are selected by the binary operator passed as input.
     Note that the @('+') or @('-'), which is binary in this disambiguation,
     is implicit in the abstract syntax of the ambiguous expression."))
  (dimb-make/adjust-expr-binary add/sub
                                (apply-post-inc/dec-ops arg1 inc/dec)
                                arg2)
  :hooks (:fix)

  ///

  (defret expr-unambp-of-dimb-cast/addsub-to-addsub
    (expr-unambp expr)
    :hyp (and (expr-unambp arg1)
              (expr-unambp arg2))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define dimb-cast/and-to-cast ((tyname tynamep)
                               (inc/dec inc/dec-op-listp)
                               (arg exprp))
  :guard (and (tyname-unambp tyname)
              (expr-unambp arg))
  :returns (expr exprp)
  :short "Disambiguate an ambiguous cast or conjunction expression
          to be a cast expression."
  :long
  (xdoc::topstring
   (xdoc::p
    "This is analogous in purpose to @(tsee dimb-cast/call-to-cast),
     but for a different kind of ambiguous expression.
     Note that the @('&'), which is unary in this disambiguation,
     is implicit in the abstract syntax of the ambiguous expression."))
  (dimb-make/adjust-expr-cast tyname
                              (dimb-make/adjust-expr-unary
                               (unop-address)
                               (apply-pre-inc/dec-ops inc/dec arg)))
  :hooks (:fix)

  ///

  (defret expr-unambp-of-dimb-cast/and-to-cast
    (expr-unambp expr)
    :hyp (and (tyname-unambp tyname)
              (expr-unambp arg))))

;;;;;;;;;;;;;;;;;;;;

(define dimb-cast/and-to-and ((arg1 exprp)
                              (inc/dec inc/dec-op-listp)
                              (arg2 exprp))
  :guard (and (expr-unambp arg1)
              (expr-unambp arg2))
  :returns (expr exprp)
  :short "Disambiguate an ambiguous cast or conjunction expression
          to be a conjunction expression."
  :long
  (xdoc::topstring
   (xdoc::p
    "This is analogous in purpose to @(tsee dimb-cast/call-to-call),
     but for a different kind of ambiguous expression.
     Note that the @('&'), which is binary in this disambiguation,
     is implicit in the abstract syntax of the ambiguous expression."))
  (dimb-make/adjust-expr-binary (binop-bitand)
                                (apply-post-inc/dec-ops arg1 inc/dec)
                                arg2)
  :hooks (:fix)

  ///

  (defret expr-unambp-of-dimb-cast/and-to-and
    (expr-unambp expr)
    :hyp (and (expr-unambp arg1)
              (expr-unambp arg2))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define dimb-params-to-names ((params paramdecl-listp)
                              (fundefp booleanp)
                              (table dimb-tablep))
  :returns (mv (yes/no booleanp) (names ident-listp))
  :short "Disambiguate a list of parameter declarations to a list of names,
          if appropriate."
  :long
  (xdoc::topstring
   (xdoc::p
    "There are two kinds of direct function declarators,
     both in the grammar and in the abstract syntax:
     one has a (non-empty) list of parameter declarations
     optionally followed by ellipsis;
     the other has a possibly empty list of names.")
   (xdoc::p
    "The second kind is allowed to be non-empty only if
     the function declarator is part of a function definition
     [C17:6.7.6.3/3].
     This is indicated by the flag @('fundefp') passed to this ACL2 function.")
   (xdoc::p
    "The parser always creates the first kind,
     because a name, which is an identifier, is syntactically ambiguous:
     it could be a parameter name, or it could be a @('typedef') name.
     Also, if there are no parameters,
     the parser does not bother creating the second kind of declarator:
     it creates an empty list of parameter declarations,
     because this needs to be disambiguated anyhow.")
   (xdoc::p
    "This ACL2 function checks whether
     a possibly empty list of parameter declarations
     should in fact be a list of names.
     This is the case when either the list is empty,
     or the @('fundefp') flag is @('t') and
     every parameter declaration consists of
     a single type specifier consisting of a @('typedef') name,
     but that identifier does not identify a @('typedef') name in scope.
     This means that, for example, if we have two identifiers @('x') and @('y'),
     one of which is a @('typedef') name but the other one is not,
     the re-classification to names fails;
     one @('typedef') name suffices to re-classify the parameters to names.
     [C17:6.7.6.3/11] says that @('typedef') names have priority,
     but strictly speaking it mentions only parameter declarations,
     not also identifier lists;
     nonetheless, some simple experiments with GCC show that
     this priority of @('typedef') names also applies to
     the choice between parameter declarations and identifier lists,
     and not just withing parameter declarations
     (this aspect is dealt with elsewhere,
     in the code to disambiguate parameter declarations).
     So, in the example above with @('x') and @('y'),
     the code is in fact invalid.")
   (xdoc::p
    "This ACL2 function returns a boolean saying whether
     the parameter declarations are re-classified into names,
     and in this case it also returns the list of names, whieh may be empty.
     If the check for any element of the list,
     the re-classification fails,
     and the caller will do its own processing and disamguation
     of the (non-empty) list of parameter declarations,
     which will then remain parameter declarations (not names)
     after that processing and disambiguation."))
  (b* (((when (endp params)) (mv t nil))
       ((unless fundefp) (mv nil nil)))
    (dimb-params-to-names-loop params table))
  :hooks (:fix)

  :prepwork
  ((define dimb-params-to-names-loop ((params paramdecl-listp)
                                      (table dimb-tablep))
     :returns (mv (yes/no booleanp) (names ident-listp))
     :parents nil
     (b* (((when (endp params)) (mv t nil))
          (param (car params))
          ((unless (paramdeclor-case (paramdecl->decl param) :none))
           (mv nil nil))
          (declspecs (paramdecl->spec param))
          ((unless (and (consp declspecs) (endp (cdr declspecs))))
           (mv nil nil))
          (declspec (car declspecs))
          ((unless (decl-spec-case declspec :typespec)) (mv nil nil))
          (tyspec (decl-spec-typespec->spec declspec))
          ((unless (type-spec-case tyspec :typedef)) (mv nil nil))
          (ident (type-spec-typedef->name tyspec))
          (kind? (dimb-lookup-ident ident table))
          ((when (equal kind? (dimb-kind-typedef))) (mv nil nil))
          ((mv yes/no names) (dimb-params-to-names-loop (cdr params) table))
          ((unless yes/no) (mv nil nil)))
       (mv t (cons ident names)))
     :hooks (:fix))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defines dimb-exprs/decls/stmts
  :short "Disambiguate expressions, declarations, statements,
          and related artifacts."
  :long
  (xdoc::topstring
   (xdoc::p
    "In general, to disambiguate a construct,
     first we recursively disambiguate its sub-constructs,
     then we either join them into an updated construct,
     or perform a disambiguation of the construct itself if needed.")
   (xdoc::p
    "In general, the disambiguation of a construct
     may involve the extension of the disambiguation table.
     For instance, the mere occurrence of an enumeration specifier
     in the type name that is part of a cast expression
     extends the disambiguation table with enumeration constants."))

  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  (define dimb-expr ((expr exprp) (table dimb-tablep))
    :returns (mv erp (new-expr exprp) (new-table dimb-tablep))
    :parents (disambiguator dimb-exprs/decls/stmts)
    :short "Disambiguate an expression."
    :long
    (xdoc::topstring
     (xdoc::p
      "If an expression is an identifier, we look it up in the table
       to see whether it should be re-classified as an enumeration constant.")
     (xdoc::p
      "A constant or string literal is left unchanged;
       there is nothing to disambiguate.
       If the constant is an enumeration constant,
       we could check that the identifier is in the table
       and has the kind of enumeration constant.
       However, the parser never generates enumeration constants,
       so there is no need to do that.")
     (xdoc::p
      "We recursively disambiguate sub-expressions,
       and other sub-entities (e.g. generic associations, type names),
       following the recursive structure of the types.")
     (xdoc::p
      "We call a separate function to disambiguate
       an ambiguous @('sizeof') expression.
       Depending on whether the result is an expression or a type name,
       we re-classify the expression into an unambiguous one.")
     (xdoc::p
      "An ambiguous cast or call is described in detail in @(tsee expr):
       refer to that documentation to understand how it is disambiguated here;
       recall that it has the form @('(X) IncDec (E)Pr').
       We call a separate function to disambiguate
       the initial expression or type name, @('X').
       Based on the result, we re-classify the expression as a cast or call,
       using separate ACL2 functions.")
     (xdoc::p
      "The other kinds of ambiguous cast or binary expressions
       are described in detail in @(tsee expr):
       refer to that documentation to understand how it is disambiguated here;
       recall that is has the form @('(X) IncDec O E'),
       where @('O') is an ambiguous unary or binary operator.
       We call a separate function to disambiguate
       the initial expression or type name, @('X').
       Based on the result,
       we re-classify the expression as a cast or a binary one,
       using separate ACL2 functions.")
     (xdoc::p
      "For cast and binary expressions,
       after disambiguating their sub-constructs,
       we use @(tsee dimb-make/adjust-expr-cast)
       and @(tsee dimb-make/adjust-expr-binary)
       for the reasons explained in the documentation of those functions.
       These two kinds of expressions are the only ones needing this treatment,
       because the only change in priorities
       that may be caused by disambiguation
       is for cast and binary expressions."))
    (b* (((reterr) (irr-expr) (irr-dimb-table)))
      (expr-case
       expr
       :ident
       (b* ((kind (dimb-lookup-ident expr.ident table))
            ((unless kind)
             (reterr (msg "The identifier ~x0 is used as an expression ~
                           but is not in scope."
                          (ident->unwrap expr.ident)))))
         (dimb-kind-case
          kind
          :typedef (reterr (msg "The identifier ~x0 denotes a typedef ~
                                 but it is used as an expression."
                                (ident->unwrap expr.ident)))
          :objfun (retok (expr-fix expr)
                         (dimb-table-fix table))
          :enumconst (retok (expr-const (const-enum expr.ident))
                            (dimb-table-fix table))))
       :const
       (retok (expr-fix expr) (dimb-table-fix table))
       :string
       (retok (expr-fix expr) (dimb-table-fix table))
       :paren
       (b* (((erp new-expr table) (dimb-expr expr.inner table)))
         (retok (expr-paren new-expr) table))
       :gensel
       (b* (((erp new-control table) (dimb-expr expr.control table))
            ((erp new-assocs table) (dimb-genassoc-list expr.assocs table)))
         (retok (make-expr-gensel :control new-control
                                  :assocs new-assocs)
                table))
       :arrsub
       (b* (((erp new-arg1 table) (dimb-expr expr.arg1 table))
            ((erp new-arg2 table) (dimb-expr expr.arg2 table)))
         (retok (make-expr-arrsub :arg1 new-arg1
                                  :arg2 new-arg2)
                table))
       :funcall
       (b* (((erp new-fun table) (dimb-expr expr.fun table))
            ((erp new-args table) (dimb-expr-list expr.args table)))
         (retok (make-expr-funcall :fun new-fun :args new-args)
                table))
       :member
       (b* (((erp new-arg table) (dimb-expr expr.arg table)))
         (retok (make-expr-member :arg new-arg :name expr.name)
                table))
       :memberp
       (b* (((erp new-arg table) (dimb-expr expr.arg table)))
         (retok (make-expr-memberp :arg new-arg :name expr.name)
                (dimb-table-fix table)))
       :complit
       (b* (((erp new-type table) (dimb-tyname expr.type table))
            ((erp new-elems table) (dimb-desiniter-list expr.elems table)))
         (retok (make-expr-complit :type new-type
                                   :elems new-elems
                                   :final-comma expr.final-comma)
                table))
       :unary
       (b* (((erp new-arg table) (dimb-expr expr.arg table)))
         (retok (dimb-make/adjust-expr-unary expr.op
                                             new-arg)
                table))
       :sizeof
       (b* (((erp new-tyname table) (dimb-tyname expr.type table)))
         (retok (expr-sizeof new-tyname) table))
       :sizeof-ambig
       (b* (((erp expr-or-tyname table)
             (dimb-amb-expr/tyname expr.expr/tyname t table)))
         (expr/tyname-case
          expr-or-tyname
          :expr (retok (dimb-make/adjust-expr-unary (unop-sizeof)
                                                    expr-or-tyname.unwrap)
                       table)
          :tyname (retok (expr-sizeof expr-or-tyname.unwrap)
                         table)))
       :alignof
       (b* (((erp new-tyname table) (dimb-tyname expr.type table)))
         (retok (make-expr-alignof :type new-tyname :uscores expr.uscores)
                table))
       :cast
       (b* (((erp new-type table) (dimb-tyname expr.type table))
            ((erp new-arg table) (dimb-expr expr.arg table)))
         (retok (dimb-make/adjust-expr-cast new-type new-arg)
                table))
       :binary
       (b* (((erp new-arg1 table) (dimb-expr expr.arg1 table))
            ((erp new-arg2 table) (dimb-expr expr.arg2 table)))
         (retok (dimb-make/adjust-expr-binary expr.op new-arg1 new-arg2)
                table))
       :cond
       (b* (((erp new-test table) (dimb-expr expr.test table))
            ((erp new-then table) (dimb-expr-option expr.then table))
            ((erp new-else table) (dimb-expr expr.else table)))
         (retok (make-expr-cond :test new-test
                                :then new-then
                                :else new-else)
                table))
       :comma
       (b* (((erp new-first table) (dimb-expr expr.first table))
            ((erp new-next table) (dimb-expr expr.next table)))
         (retok (make-expr-comma :first new-first
                                 :next new-next)
                table))
       :cast/call-ambig
       (b* (((erp expr/tyname table)
             (dimb-amb-expr/tyname expr.type/fun t table))
            ((erp new-arg/rest table) (dimb-expr expr.arg/rest table)))
         (expr/tyname-case
          expr/tyname
          :tyname
          (retok
           (dimb-cast/call-to-cast (expr/tyname-tyname->unwrap expr/tyname)
                                   expr.inc/dec
                                   new-arg/rest)
           table)
          :expr
          (retok
           (dimb-cast/call-to-call (expr/tyname-expr->unwrap expr/tyname)
                                   expr.inc/dec
                                   new-arg/rest)
           table)))
       :cast/mul-ambig
       (b* (((erp expr/tyname table)
             (dimb-amb-expr/tyname expr.type/arg1 t table))
            ((erp new-arg/arg2 table) (dimb-expr expr.arg/arg2 table)))
         (expr/tyname-case
          expr/tyname
          :tyname
          (retok
           (dimb-cast/mul-to-cast (expr/tyname-tyname->unwrap expr/tyname)
                                  expr.inc/dec
                                  new-arg/arg2)
           table)
          :expr
          (retok
           (dimb-cast/mul-to-mul (expr/tyname-expr->unwrap expr/tyname)
                                 expr.inc/dec
                                 new-arg/arg2)
           table)))
       :cast/add-ambig
       (b* (((erp expr/tyname table)
             (dimb-amb-expr/tyname expr.type/arg1 t table))
            ((erp new-arg/arg2 table) (dimb-expr expr.arg/arg2 table)))
         (expr/tyname-case
          expr/tyname
          :tyname
          (retok
           (dimb-cast/addsub-to-cast (expr/tyname-tyname->unwrap expr/tyname)
                                     expr.inc/dec
                                     new-arg/arg2
                                     (unop-plus))
           table)
          :expr
          (retok
           (dimb-cast/addsub-to-addsub (expr/tyname-expr->unwrap expr/tyname)
                                       expr.inc/dec
                                       new-arg/arg2
                                       (binop-add))
           table)))
       :cast/sub-ambig
       (b* (((erp expr/tyname table)
             (dimb-amb-expr/tyname expr.type/arg1 t table))
            ((erp new-arg/arg2 table) (dimb-expr expr.arg/arg2 table)))
         (expr/tyname-case
          expr/tyname
          :tyname
          (retok
           (dimb-cast/addsub-to-cast (expr/tyname-tyname->unwrap expr/tyname)
                                     expr.inc/dec
                                     new-arg/arg2
                                     (unop-minus))
           table)
          :expr
          (retok
           (dimb-cast/addsub-to-addsub (expr/tyname-expr->unwrap expr/tyname)
                                       expr.inc/dec
                                       new-arg/arg2
                                       (binop-sub))
           table)))
       :cast/and-ambig
       (b* (((erp expr/tyname table)
             (dimb-amb-expr/tyname expr.type/arg1 t table))
            ((erp new-arg/arg2 table) (dimb-expr expr.arg/arg2 table)))
         (expr/tyname-case
          expr/tyname
          :tyname
          (retok
           (dimb-cast/and-to-cast (expr/tyname-tyname->unwrap expr/tyname)
                                  expr.inc/dec
                                  new-arg/arg2)
           table)
          :expr
          (retok
           (dimb-cast/and-to-and (expr/tyname-expr->unwrap expr/tyname)
                                 expr.inc/dec
                                 new-arg/arg2)
           table)))
       :stmt
       (b* (((erp items table) (dimb-block-item-list expr.items table)))
         (retok (expr-stmt items) table))
       :tycompat
       (b* (((erp type1 table) (dimb-tyname expr.type1 table))
            ((erp type2 table) (dimb-tyname expr.type2 table)))
         (retok (make-expr-tycompat :type1 type1 :type2 type2) table))
       :offsetof
       (b* (((erp type table) (dimb-tyname expr.type table))
            ((erp memdes table) (dimb-member-designor expr.member table)))
         (retok (make-expr-offsetof :type type :member memdes) table))
       :va-arg
       (b* (((erp list table) (dimb-expr expr.list table))
            ((erp type table) (dimb-tyname expr.type table)))
         (retok (make-expr-va-arg :list list :type type) table))
       :extension
       (b* (((erp expr table) (dimb-expr expr.expr table)))
         (retok (expr-extension expr) table))))
    :measure (expr-count expr))

  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  (define dimb-expr-list ((exprs expr-listp) (table dimb-tablep))
    :returns (mv erp (new-exprs expr-listp) (new-table dimb-tablep))
    :parents (disambiguator dimb-exprs/decls/stmts)
    :short "Disambiguate a list of expressions."
    (b* (((reterr) nil (irr-dimb-table))
         ((when (endp exprs)) (retok nil (dimb-table-fix table)))
         ((erp new-expr table) (dimb-expr (car exprs) table))
         ((erp new-exprs table) (dimb-expr-list (cdr exprs) table)))
      (retok (cons new-expr new-exprs) table))
    :measure (expr-list-count exprs))

  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  (define dimb-expr-option ((expr? expr-optionp) (table dimb-tablep))
    :returns (mv erp (new-expr? expr-optionp) (new-table dimb-tablep))
    :parents (disambiguator dimb-exprs/decls/stmts)
    :short "Disambiguate an optional expression."
    (b* (((reterr) nil (irr-dimb-table)))
      (expr-option-case
       expr?
       :some (dimb-expr expr?.val table)
       :none (retok nil (dimb-table-fix table))))
    :measure (expr-option-count expr?))

  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  (define dimb-const-expr ((cexpr const-exprp) (table dimb-tablep))
    :returns (mv erp (new-cexpr const-exprp) (new-table dimb-tablep))
    :parents (disambiguator dimb-exprs/decls/stmts)
    :short "Disambiguate a constant expression."
    (b* (((reterr) (irr-const-expr) (irr-dimb-table))
         ((erp new-expr table) (dimb-expr (const-expr->expr cexpr) table)))
      (retok (const-expr new-expr) table))
    :measure (const-expr-count cexpr))

  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  (define dimb-const-expr-option ((cexpr? const-expr-optionp)
                                  (table dimb-tablep))
    :returns (mv erp (new-cexpr? const-expr-optionp) (new-table dimb-tablep))
    :parents (disambiguator dimb-exprs/decls/stmts)
    :short "Disambiguate an optional constant expression."
    (b* (((reterr) nil (irr-dimb-table)))
      (const-expr-option-case
       cexpr?
       :some (dimb-const-expr cexpr?.val table)
       :none (retok nil (dimb-table-fix table))))
    :measure (const-expr-option-count cexpr?))

  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  (define dimb-genassoc ((assoc genassocp) (table dimb-tablep))
    :returns (mv erp (new-assoc genassocp) (new-table dimb-tablep))
    :parents (disambiguator dimb-exprs/decls/stmts)
    :short "Disambiguate a generic association."
    (b* (((reterr) (irr-genassoc) (irr-dimb-table)))
      (genassoc-case
       assoc
       :type (b* (((erp new-tyname table) (dimb-tyname assoc.type table))
                  ((erp new-expr table) (dimb-expr assoc.expr table)))
               (retok (make-genassoc-type :type new-tyname
                                          :expr new-expr)
                      table))
       :default (b* (((erp new-expr table) (dimb-expr assoc.expr table)))
                  (retok (genassoc-default new-expr) table))))
    :measure (genassoc-count assoc))

  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  (define dimb-genassoc-list ((assocs genassoc-listp) (table dimb-tablep))
    :returns (mv erp (new-assocs genassoc-listp) (new-table dimb-tablep))
    :parents (disambiguator dimb-exprs/decls/stmts)
    :short "Disambiguate a list of generic associations."
    (b* (((reterr) nil (irr-dimb-table))
         ((when (endp assocs)) (retok nil (dimb-table-fix table)))
         ((erp new-assoc table) (dimb-genassoc (car assocs) table))
         ((erp new-assocs table) (dimb-genassoc-list (cdr assocs) table)))
      (retok (cons new-assoc new-assocs) table))
    :measure (genassoc-list-count assocs))

  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  (define dimb-member-designor ((memdes member-designorp) (table dimb-tablep))
    :returns (mv erp (new-memdes member-designorp) (new-table dimb-tablep))
    :parents (disambiguator dimb-exprs/decls/stmts)
    :short "Disambiguate a member designator."
    (b* (((reterr) (irr-member-designor) (irr-dimb-table)))
      (member-designor-case
       memdes
       :ident (retok (member-designor-fix memdes) (dimb-table-fix table))
       :dot (b* (((erp new-memdes table)
                  (dimb-member-designor memdes.member table)))
              (retok (make-member-designor-dot :member new-memdes
                                               :name memdes.name)
                     table))
       :sub (b* (((erp new-memdes table)
                  (dimb-member-designor memdes.member table))
                 ((erp new-index table)
                  (dimb-expr memdes.index table)))
              (retok (make-member-designor-sub :member new-memdes
                                               :index new-index)
                     table))))
    :measure (member-designor-count memdes))

  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  (define dimb-type-spec ((tyspec type-specp) (table dimb-tablep))
    :returns (mv erp (new-tyspec type-specp) (new-table dimb-tablep))
    :parents (disambiguator dimb-exprs/decls/stmts)
    :short "Disambiguate a type specifier."
    :long
    (xdoc::topstring
     (xdoc::p
      "A type specifier may affect the disambiguation table,
       by adding identifiers to it.
       Thus, this function returns
       not only a disambiguated type specifier,
       but also a possibly updated table.")
     (xdoc::p
      "The only type specifier that affect the disambiguation table
       is an enumeration specifiers,
       which extends the table with information about
       the enumeration constants it introduces.
       This is actually done by a separate function,
       the one that processes the enumerators.")
     (xdoc::p
      "Note that the disambiguation table contains no information
       about the structure, union, and enumeration tags:
       so we do need to extend the table when we encounter
       the corresponding type specifiers.
       Members or structures and unions also do not affect
       the disambiguation table,
       since the table contains no information about any of them.")
     (xdoc::p
      "If we encounter a @('typedef') name,
       we need to check that it is in the disambiguation table,
       with the right kind."))
    (b* (((reterr) (irr-type-spec) (irr-dimb-table)))
      (type-spec-case
       tyspec
       :void (retok (type-spec-void) (dimb-table-fix table))
       :char (retok (type-spec-char) (dimb-table-fix table))
       :short (retok (type-spec-short) (dimb-table-fix table))
       :int (retok (type-spec-int) (dimb-table-fix table))
       :long (retok (type-spec-long) (dimb-table-fix table))
       :float (retok (type-spec-float) (dimb-table-fix table))
       :double (retok (type-spec-double) (dimb-table-fix table))
       :signed (retok (type-spec-signed tyspec.uscores) (dimb-table-fix table))
       :unsigned (retok (type-spec-unsigned) (dimb-table-fix table))
       :bool (retok (type-spec-bool) (dimb-table-fix table))
       :complex (retok (type-spec-complex) (dimb-table-fix table))
       :atomic (b* (((erp new-type table) (dimb-tyname tyspec.type table)))
                 (retok (type-spec-atomic new-type) table))
       :struct (b* (((erp new-strunispec table)
                     (dimb-strunispec tyspec.spec table)))
                 (retok (type-spec-struct new-strunispec)
                        table))
       :union (b* (((erp new-strunispec table)
                    (dimb-strunispec tyspec.spec table)))
                (retok (type-spec-union new-strunispec)
                       table))
       :enum (b* (((erp new-enumspec table)
                   (dimb-enumspec tyspec.spec table)))
               (retok (type-spec-enum new-enumspec) table))
       :typedef (b* ((kind (dimb-lookup-ident tyspec.name table))
                     ((unless kind)
                      (reterr
                       (msg "The identifier ~x0 is used as a type specifier ~
                             but it is not in scope."
                            (ident->unwrap tyspec.name)))))
                  (dimb-kind-case
                   kind
                   :typedef (retok (type-spec-typedef tyspec.name)
                                   (dimb-table-fix table))
                   :objfun (reterr
                            (msg "The identifier ~x0 denotes ~
                                  an object or function ~
                                  but it is used as a typedef name."
                                 (ident->unwrap tyspec.name)))
                   :enumconst (reterr
                               (msg "The identifier ~x0 denotes ~
                                     an enumeration constant ~
                                     but it is used as a typedef name."
                                    (ident->unwrap tyspec.name)))))
       :int128 (retok (type-spec-int128) (dimb-table-fix table))
       :float32 (retok (type-spec-float32) (dimb-table-fix table))
       :float32x (retok (type-spec-float32x) (dimb-table-fix table))
       :float64 (retok (type-spec-float64) (dimb-table-fix table))
       :float64x (retok (type-spec-float64x) (dimb-table-fix table))
       :float128 (retok (type-spec-float128) (dimb-table-fix table))
       :float128x (retok (type-spec-float128x) (dimb-table-fix table))
       :builtin-va-list (retok (type-spec-builtin-va-list)
                               (dimb-table-fix table))
       :struct-empty (retok (type-spec-fix tyspec)
                            (dimb-table-fix table))
       :typeof-expr
       (b* (((erp new-expr table) (dimb-expr tyspec.expr table)))
         (retok (make-type-spec-typeof-expr :expr new-expr
                                            :uscores tyspec.uscores)
                table))
       :typeof-type
       (b* (((erp new-tyname table) (dimb-tyname tyspec.type table)))
         (retok (make-type-spec-typeof-type :type new-tyname
                                            :uscores tyspec.uscores)
                table))
       :typeof-ambig
       (b* (((erp expr/tyname table)
             (dimb-amb-expr/tyname tyspec.expr/type nil table)))
         (expr/tyname-case
          expr/tyname
          :expr (retok (make-type-spec-typeof-expr :expr expr/tyname.unwrap
                                                   :uscores tyspec.uscores)
                       table)
          :tyname (retok (make-type-spec-typeof-type :type expr/tyname.unwrap
                                                     :uscores tyspec.uscores)
                         table)))
       :auto-type (retok (type-spec-auto-type) (dimb-table-fix table))))
    :measure (type-spec-count tyspec))

  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  (define dimb-spec/qual ((specqual spec/qual-p) (table dimb-tablep))
    :returns (mv erp (new-specqual spec/qual-p) (new-table dimb-tablep))
    :parents (disambiguator dimb-exprs/decls/stmts)
    :short "Disambiguate a specifier or qualifier."
    :long
    (xdoc::topstring
     (xdoc::p
      "Type qualifiers are left unchanged.
       Type specifiers may extend the disambiguation table."))
    (b* (((reterr) (irr-spec/qual) (irr-dimb-table)))
      (spec/qual-case
       specqual
       :typespec (b* (((erp new-tyspec table)
                       (dimb-type-spec specqual.spec table)))
                   (retok (spec/qual-typespec new-tyspec)
                          table))
       :typequal (retok (spec/qual-typequal specqual.qual)
                        (dimb-table-fix table))
       :align (b* (((erp new-alignspec table)
                    (dimb-align-spec specqual.spec table)))
                (retok (spec/qual-align new-alignspec)
                       table))
       :attrib (retok (spec/qual-attrib specqual.spec)
                      (dimb-table-fix table))))
    :measure (spec/qual-count specqual))

  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  (define dimb-spec/qual-list ((specquals spec/qual-listp) (table dimb-tablep))
    :returns (mv erp (new-specquals spec/qual-listp) (new-table dimb-tablep))
    :parents (disambiguator dimb-exprs/decls/stmts)
    :short "Disambiguate a list of specifiers and qualifiers."
    :long
    (xdoc::topstring
     (xdoc::p
      "We process them in order, threading through the table,
       which may be updated by each."))
    (b* (((reterr) nil (irr-dimb-table))
         ((when (endp specquals)) (retok nil (dimb-table-fix table)))
         ((erp new-specqual table) (dimb-spec/qual (car specquals) table))
         ((erp new-specquals table) (dimb-spec/qual-list (cdr specquals) table)))
      (retok (cons new-specqual new-specquals) table))
    :measure (spec/qual-list-count specquals))

  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  (define dimb-align-spec ((alignspec align-specp) (table dimb-tablep))
    :returns (mv erp (new-alignspec align-specp) (new-table dimb-tablep))
    :parents (disambiguator dimb-exprs/decls/stmts)
    :short "Disambiguate an alignment specifier."
    :long
    (xdoc::topstring
     (xdoc::p
      "For an ambiguous alignment specifier,
       we disambiguate the underlying type name or expression,
       and then return one of the two kinds of
       unambiguous alignment specifiers."))
    (b* (((reterr) (irr-align-spec) (irr-dimb-table)))
      (align-spec-case
       alignspec
       :alignas-type
       (b* (((erp new-type table) (dimb-tyname alignspec.type table)))
         (retok (align-spec-alignas-type new-type) table))
       :alignas-expr
       (b* (((erp new-expr table) (dimb-const-expr alignspec.expr table)))
         (retok (align-spec-alignas-expr new-expr) table))
       :alignas-ambig
       (b* (((erp expr/tyname table)
             (dimb-amb-expr/tyname alignspec.expr/type nil table)))
         (expr/tyname-case
          expr/tyname
          :expr (retok (align-spec-alignas-expr (const-expr expr/tyname.unwrap))
                       table)
          :tyname (retok (align-spec-alignas-type expr/tyname.unwrap)
                         table)))))
    :measure (align-spec-count alignspec))

  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  (define dimb-decl-spec ((declspec decl-specp)
                          (kind dimb-kindp)
                          (table dimb-tablep))
    :returns (mv erp
                 (new-declspec decl-specp)
                 (new-kind dimb-kindp)
                 (new-table dimb-tablep))
    :parents (disambiguator dimb-exprs/decls/stmts)
    :short "Disambiguate a declaration specifier."
    :long
    (xdoc::topstring
     (xdoc::p
      "Declaration specifiers (may) precede declarators,
       which add identifiers to the current scope.
       For our disambiguation purposes,
       we need to determine which @(tsee dimb-kind)
       those identifiers denote, which, for declarators,
       is either @(':objfun') or @(':typedef').
       It is @(':typedef') if the list of declaration specifiers
       includes the @('typedef') storage class specifier;
       otherwise, it is @(':objfun').
       Thus, when going through the declaration specifiers,
       we initialize the kind to @(':objfun'),
       and change it to @(':typedef') if we encounter a @('typedef').
       This is why this ACL2 function takes and returns
       a disambiguation kind, i.e. a value of type @(tsee dimb-kind)."))
    (b* (((reterr) (irr-decl-spec) (irr-dimb-kind) (irr-dimb-table)))
      (decl-spec-case
       declspec
       :stoclass (if (stor-spec-case declspec.spec :typedef)
                     (retok (decl-spec-fix declspec)
                            (dimb-kind-typedef)
                            (dimb-table-fix table))
                   (retok (decl-spec-fix declspec)
                          (dimb-kind-fix kind)
                          (dimb-table-fix table)))
       :typespec (b* (((erp new-tyspec table)
                       (dimb-type-spec declspec.spec table)))
                   (retok (decl-spec-typespec new-tyspec)
                          (dimb-kind-fix kind)
                          (dimb-table-fix table)))
       :typequal (retok (decl-spec-fix declspec)
                        (dimb-kind-fix kind)
                        (dimb-table-fix table))
       :function (retok (decl-spec-fix declspec)
                        (dimb-kind-fix kind)
                        (dimb-table-fix table))
       :align (b* (((erp new-alignspec table)
                    (dimb-align-spec declspec.spec table)))
                (retok (decl-spec-align new-alignspec)
                       (dimb-kind-fix kind)
                       table))
       :attrib (retok (decl-spec-fix declspec)
                      (dimb-kind-fix kind)
                      (dimb-table-fix table))
       :stdcall (retok (decl-spec-fix declspec)
                       (dimb-kind-fix kind)
                       (dimb-table-fix table))
       :declspec (retok (decl-spec-fix declspec)
                        (dimb-kind-fix kind)
                        (dimb-table-fix table))))
    :measure (decl-spec-count declspec))

  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  (define dimb-decl-spec-list ((declspecs decl-spec-listp)
                               (kind dimb-kindp)
                               (table dimb-tablep))
    :returns (mv erp
                 (new-declspecs decl-spec-listp)
                 (new-kind dimb-kindp)
                 (new-table dimb-tablep))
    :parents (disambiguator dimb-exprs/decls/stmts)
    :short "Disambiguate a list of declaration specifiers."
    :long
    (xdoc::topstring
     (xdoc::p
      "See @(tsee dimb-decl-spec) for an explanation of
       the disambiguation kind passed as input and returned as output."))
    (b* (((reterr) nil (irr-dimb-kind) (irr-dimb-table))
         ((when (endp declspecs)) (retok nil
                                         (dimb-kind-fix kind)
                                         (dimb-table-fix table)))
         ((erp new-declspec kind table)
          (dimb-decl-spec (car declspecs) kind table))
         ((erp new-declspecs kind table)
          (dimb-decl-spec-list (cdr declspecs) kind table)))
      (retok (cons new-declspec new-declspecs) kind table))
    :measure (decl-spec-list-count declspecs))

  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  (define dimb-initer ((initer initerp) (table dimb-tablep))
    :returns (mv erp (new-initer initerp) (new-table dimb-tablep))
    :parents (disambiguator dimb-exprs/decls/stmts)
    :short "Disambiguate an initializer."
    (b* (((reterr) (irr-initer) (irr-dimb-table)))
      (initer-case
       initer
       :single (b* (((erp new-expr table) (dimb-expr initer.expr table)))
                 (retok (initer-single new-expr) table))
       :list (b* (((erp new-elems table)
                   (dimb-desiniter-list initer.elems table)))
               (retok (make-initer-list :elems new-elems
                                        :final-comma initer.final-comma)
                      table))))
    :measure (initer-count initer))

  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  (define dimb-initer-option ((initer? initer-optionp) (table dimb-tablep))
    :returns (mv erp (new-initer? initer-optionp) (new-table dimb-tablep))
    :parents (disambiguator dimb-exprs/decls/stmts)
    :short "Disambiguate an optional initializer."
    (b* (((reterr) nil (irr-dimb-table)))
      (initer-option-case
       initer?
       :some (dimb-initer initer?.val table)
       :none (retok nil (dimb-table-fix table))))
    :measure (initer-option-count initer?))

  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  (define dimb-desiniter ((desiniter desiniterp) (table dimb-tablep))
    :returns (mv erp (new-desiniter desiniterp) (new-table dimb-tablep))
    :parents (disambiguator dimb-exprs/decls/stmts)
    :short "Disambiguate an initializer with optional designations."
    (b* (((reterr) (irr-desiniter) (irr-dimb-table))
         ((desiniter desiniter) desiniter)
         ((erp new-designors table)
          (dimb-designor-list desiniter.designors table))
         ((erp new-initer table) (dimb-initer desiniter.initer table)))
      (retok (make-desiniter :designors new-designors :initer new-initer)
             table))
    :measure (desiniter-count desiniter))

  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  (define dimb-desiniter-list ((desiniters desiniter-listp) (table dimb-tablep))
    :returns (mv erp (new-desiniters desiniter-listp) (new-table dimb-tablep))
    :parents (disambiguator dimb-exprs/decls/stmts)
    :short "Disambiguate a list of initializers with optional designations."
    (b* (((reterr) nil (irr-dimb-table))
         ((when (endp desiniters)) (retok nil (dimb-table-fix table)))
         ((erp new-desiniter table) (dimb-desiniter (car desiniters) table))
         ((erp new-desiniters table)
          (dimb-desiniter-list (cdr desiniters) table)))
      (retok (cons new-desiniter new-desiniters) table))
    :measure (desiniter-list-count desiniters))

  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  (define dimb-designor ((design designorp) (table dimb-tablep))
    :returns (mv erp (new-design designorp) (new-table dimb-tablep))
    :parents (disambiguator dimb-exprs/decls/stmts)
    :short "Disambiguate a designator."
    (b* (((reterr) (irr-designor) (irr-dimb-table)))
      (designor-case
       design
       :sub (b* (((erp new-index table) (dimb-const-expr design.index table)))
              (retok (designor-sub new-index) table))
       :dot (retok (designor-dot design.name) (dimb-table-fix table))))
    :measure (designor-count design))

  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  (define dimb-designor-list ((designs designor-listp) (table dimb-tablep))
    :returns (mv erp (new-designs designor-listp) (new-table dimb-tablep))
    :parents (disambiguator dimb-exprs/decls/stmts)
    :short "Disambiguate a list of designators."
    (b* (((reterr) nil (irr-dimb-table))
         ((when (endp designs)) (retok nil (dimb-table-fix table)))
         ((erp new-design table) (dimb-designor (car designs) table))
         ((erp new-designs table) (dimb-designor-list (cdr designs) table)))
      (retok (cons new-design new-designs) table))
    :measure (designor-list-count designs))

  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  (define dimb-declor ((declor declorp) (fundefp booleanp) (table dimb-tablep))
    :returns (mv erp (new-declor declorp) (ident identp) (table dimb-tablep))
    :parents (disambiguator dimb-exprs/decls/stmts)
    :short "Disambiguate a declarator."
    :long
    (xdoc::topstring
     (xdoc::p
      "A declarator adds an identifier to the scope.
       This function returns the identifier.
       Its addition to the disambiguation table is performed
       outside of this function,
       after processing the top-level declarator
       in the construct of interest.")
     (xdoc::p
      "The pointer part of a declarator does not contribute to the table
       and does not need to be disambiguated.
       So we recursively disambiguate the direct declarator,
       which also gives us the identifier,
       and then we re-add the pointer part.")
     (xdoc::p
      "The @('fundefp') flag passed to this function
       says whether we are disambiguating
       the declarator of a function definition or not.
       It is used to determine whether,
       when disambiguating the parameters of a function declarator,
       the new scope pushed for the function prototype
       should be popped at the end of the function declarator or not.
       If the declarator is not part of a function definition,
       then that scope must be popped;
       but if instead the declarator is part of a function definition,
       that scope is the block scope of the definition,
       and must not be popped.
       Here by `part of a function definition' of course we mean
       the one that introduces the name of the function being defined."))
    (b* (((reterr) (irr-declor) (irr-ident) (irr-dimb-table))
         ((declor declor) declor)
         ((erp new-dirdeclor ident table)
          (dimb-dirdeclor declor.direct fundefp table)))
      (retok (make-declor :pointers declor.pointers
                          :direct new-dirdeclor)
             ident
             table))
    :measure (declor-count declor))

  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  (define dimb-declor-option ((declor? declor-optionp) (table dimb-tablep))
    :returns (mv erp
                 (new-declor? declor-optionp)
                 (ident? ident-optionp)
                 (new-table dimb-tablep))
    :parents (disambiguator dimb-exprs/decls/stmts)
    :short "Disambiguate an optional declarator."
    :long
    (xdoc::topstring
     (xdoc::p
      "As with similar disambiguation functions,
       this lifts @(tsee dimb-declor) to optional declarators.
       Since the declarator may be absent,
       we also generalize the returned identifier to be an optional one.")
     (xdoc::p
      "Note that we call @(tsee dimb-declor) with
       @('nil') as the @('fundefp') flag,
       because if we are disambiguating an optional declarator
       we are not disambiguating the declarator of a defined function."))
    (b* (((reterr) nil nil (irr-dimb-table)))
      (declor-option-case
       declor?
       :some (dimb-declor declor?.val nil table)
       :none (retok nil nil (dimb-table-fix table))))
    :measure (declor-option-count declor?))

  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  (define dimb-dirdeclor ((dirdeclor dirdeclorp)
                          (fundefp booleanp)
                          (table dimb-tablep))
    :returns (mv erp
                 (new-dirdeclor dirdeclorp)
                 (ident identp)
                 (new-table dimb-tablep))
    :parents (disambiguator dimb-exprs/decls/stmts)
    :short "Disambiguate a direct declarator."
    :long
    (xdoc::topstring
     (xdoc::p
      "As explained in @(tsee dimb-declor),
       a (direct) declarator adds an identifier to the scope.
       So here we return the identifer,
       recursively extracted from the direct declarator.
       The actual addition to the disambiguation table
       is performed outside this function.")
     (xdoc::p
      "The purpose of the @('fundefp') flag is
       the same as in @(tsee dimb-declor),
       which in fact passes it to this function.
       Here we make use of it, as explained below.")
     (xdoc::p
      "We recursively disambiguate the inner declarator and direct declarator,
       from which we obtain the identifier.
       We also recursively disambiguate any expressions in array declarators.")
     (xdoc::p
      "For function declarators,
       the parser only produces @(':function-params'),
       never @(':function-names').
       However, here we also process @(':function-names'),
       so that the disambiguator is idempotent.
       We push a new scope, for uniformity with the treatment
       described in the next paragraph.")
     (xdoc::p
      "For a @(':function-params'),
       first we attempt to turn it into a @(':function-names'), if applicable.
       we also push a new scope if @('fundefp') is @('t'),
       for the reason explained below.
       If we cannot turn the @(':function-params') into @(':function-names'),
       we disambiguate it (into another @(':function-params')) as follows.
       We push a new scope for the function prototype [C17:6.2.1/2] [C17:6.2.1/4].
       We call a separate function to disambiguate each parameter declaration.
       Then, based on the @('fundefp') flag,
       we pop the scope (if the flag is @('nil')),
       or we leave the scope there (if the flag is @('t')):
       in the latter case, this will be the scope of the function definition.
       So that is the reason why we push a scope
       also in the case, described above,
       that we turn @(':function-params') into @(':function-names'):
       either way, we are pushing a scope for the function definition.
       If the code is valid, the function definition
       will indeed have parameter declarations,
       and so the disambiguator will do the right thing;
       if the code is invalid,
       it does not actually matter what the disambiguator does."))
    (b* (((reterr) (irr-dirdeclor) (irr-ident) (irr-dimb-table)))
      (dirdeclor-case
       dirdeclor
       :ident
       (retok (dirdeclor-fix dirdeclor) dirdeclor.unwrap (dimb-table-fix table))
       :paren
       (b* (((erp new-declor ident table)
             (dimb-declor dirdeclor.unwrap fundefp table)))
         (retok (dirdeclor-paren new-declor) ident table))
       :array
       (b* (((erp new-dirdeclor ident table)
             (dimb-dirdeclor dirdeclor.decl fundefp table))
            ((erp new-expr? table) (dimb-expr-option dirdeclor.expr? table)))
         (retok (make-dirdeclor-array :decl new-dirdeclor
                                      :tyquals dirdeclor.tyquals
                                      :expr? new-expr?)
                ident
                table))
       :array-static1
       (b* (((erp new-dirdeclor ident table)
             (dimb-dirdeclor dirdeclor.decl fundefp table))
            ((erp new-expr table) (dimb-expr dirdeclor.expr table)))
         (retok (make-dirdeclor-array-static1 :decl new-dirdeclor
                                              :tyquals dirdeclor.tyquals
                                              :expr new-expr)
                ident
                table))
       :array-static2
       (b* (((erp new-dirdeclor ident table)
             (dimb-dirdeclor dirdeclor.decl fundefp table))
            ((erp new-expr table) (dimb-expr dirdeclor.expr table)))
         (retok (make-dirdeclor-array-static2 :decl new-dirdeclor
                                              :tyquals dirdeclor.tyquals
                                              :expr new-expr)
                ident
                table))
       :array-star
       (b* (((erp new-dirdeclor ident table)
             (dimb-dirdeclor dirdeclor.decl fundefp table)))
         (retok (make-dirdeclor-array-star :decl new-dirdeclor
                                           :tyquals dirdeclor.tyquals)
                ident
                table))
       :function-params
       (b* (((erp new-dirdeclor ident table)
             (dimb-dirdeclor dirdeclor.decl fundefp table))
            ((mv yes/no names)
             (dimb-params-to-names dirdeclor.params fundefp table))
            ((when yes/no)
             (retok (make-dirdeclor-function-names :decl new-dirdeclor
                                                   :names names)
                    ident
                    (if fundefp
                        (dimb-push-scope table)
                      table)))
            (table (dimb-push-scope table))
            ((erp new-params table)
             (dimb-paramdecl-list dirdeclor.params table))
            (table (if fundefp
                       table
                     (dimb-pop-scope table))))
         (retok (make-dirdeclor-function-params :decl new-dirdeclor
                                                :params new-params
                                                :ellipsis dirdeclor.ellipsis)
                ident
                table))
       :function-names
       (b* (((erp new-dirdeclor ident table)
             (dimb-dirdeclor dirdeclor.decl fundefp table)))
         (retok (make-dirdeclor-function-names :decl new-dirdeclor
                                               :names dirdeclor.names)
                ident
                (if fundefp
                    (dimb-push-scope table)
                  table)))))
    :measure (dirdeclor-count dirdeclor))

  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  (define dimb-absdeclor ((absdeclor absdeclorp) (table dimb-tablep))
    :returns (mv erp (new-absdeclor absdeclorp) (new-table dimb-tablep))
    :parents (disambiguator dimb-exprs/decls/stmts)
    :short "Disambiguate an abstract declarator."
    :long
    (xdoc::topstring
     (xdoc::p
      "The disambiguation of an abstract declarator
       is similar to the one of a declarator,
       but an abstract declarator does not introduce an identifiers,
       and so there is no identifier to return here."))
    (b* (((reterr) (irr-absdeclor) (irr-dimb-table))
         ((absdeclor absdeclor) absdeclor)
         ((erp new-decl? table)
          (dimb-dirabsdeclor-option absdeclor.decl? table)))
      (retok (make-absdeclor :pointers absdeclor.pointers
                             :decl? new-decl?)
             table))
    :measure (absdeclor-count absdeclor))

  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  (define dimb-absdeclor-option ((absdeclor? absdeclor-optionp)
                                 (table dimb-tablep))
    :returns (mv erp (new-absdeclor? absdeclor-optionp) (new-table dimb-tablep))
    :parents (disambiguator dimb-exprs/decls/stmts)
    :short "Disambiguate an optional abstract declarator."
    (b* (((reterr) nil nil (irr-dimb-table)))
      (absdeclor-option-case
       absdeclor?
       :some (dimb-absdeclor absdeclor?.val table)
       :none (retok nil (dimb-table-fix table))))
    :measure (absdeclor-option-count absdeclor?))

  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  (define dimb-dirabsdeclor ((dirabsdeclor dirabsdeclorp) (table dimb-tablep))
    :returns (mv erp (new-dirabsdeclor dirabsdeclorp) (new-table dimb-tablep))
    :parents (disambiguator dimb-exprs/decls/stmts)
    :short "Disambiguate a direct abstract declarator."
    :long
    (xdoc::topstring
     (xdoc::p
      "The handling is similar to @(tsee dimb-dirdeclor),
       but no identifier is returned.
       There is also no handling of possible function parameter names."))
    (b* (((reterr) (irr-dirabsdeclor) (irr-dimb-table)))
      (dirabsdeclor-case
       dirabsdeclor
       :dummy-base
       (prog2$
        (raise "Internal error: dummy base case of direct abstract declarator.")
        (reterr t))
       :paren
       (b* (((erp new-absdeclor table)
             (dimb-absdeclor dirabsdeclor.unwrap table)))
         (retok (dirabsdeclor-paren new-absdeclor)
                table))
       :array
       (b* (((erp new-decl? table)
             (dimb-dirabsdeclor-option dirabsdeclor.decl? table))
            ((erp new-expr? table)
             (dimb-expr-option dirabsdeclor.expr? table)))
         (retok (make-dirabsdeclor-array :decl? new-decl?
                                         :tyquals dirabsdeclor.tyquals
                                         :expr? new-expr?)
                table))
       :array-static1
       (b* (((erp new-decl? table)
             (dimb-dirabsdeclor-option dirabsdeclor.decl? table))
            ((erp new-expr table) (dimb-expr dirabsdeclor.expr table)))
         (retok (make-dirabsdeclor-array-static1 :decl? new-decl?
                                                 :tyquals dirabsdeclor.tyquals
                                                 :expr new-expr)
                table))
       :array-static2
       (b* (((erp new-decl? table)
             (dimb-dirabsdeclor-option dirabsdeclor.decl? table))
            ((erp new-expr table) (dimb-expr dirabsdeclor.expr table)))
         (retok (make-dirabsdeclor-array-static2 :decl? new-decl?
                                                 :tyquals dirabsdeclor.tyquals
                                                 :expr new-expr)
                table))
       :array-star
       (b* (((erp new-decl? table)
             (dimb-dirabsdeclor-option dirabsdeclor.decl? table)))
         (retok (dirabsdeclor-array-star new-decl?)
                table))
       :function
       (b* (((erp new-decl? table)
             (dimb-dirabsdeclor-option dirabsdeclor.decl? table))
            (table (dimb-push-scope table))
            ((erp new-params table)
             (dimb-paramdecl-list dirabsdeclor.params table))
            (table (dimb-pop-scope table)))
         (retok (make-dirabsdeclor-function :decl? new-decl?
                                            :params new-params
                                            :ellipsis dirabsdeclor.ellipsis)
                table))))
    :measure (dirabsdeclor-count dirabsdeclor))

  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  (define dimb-dirabsdeclor-option ((dirabsdeclor? dirabsdeclor-optionp)
                                    (table dimb-tablep))
    :returns (mv erp
                 (new-dirabsdeclor? dirabsdeclor-optionp)
                 (new-table dimb-tablep))
    :parents (disambiguator dimb-exprs/decls/stmts)
    :short "Disambiguate an optional direct abstract declarator."
    (b* (((reterr) nil (irr-dimb-table)))
      (dirabsdeclor-option-case
       dirabsdeclor?
       :some (dimb-dirabsdeclor dirabsdeclor?.val table)
       :none (retok nil (dimb-table-fix table))))
    :measure (dirabsdeclor-option-count dirabsdeclor?))

  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  (define dimb-paramdecl ((paramdecl paramdeclp) (table dimb-tablep))
    :returns (mv erp (new-paramdecl paramdeclp) (new-table dimb-tablep))
    :parents (disambiguator dimb-exprs/decls/stmts)
    :short "Disambiguate a parameter declaration."
    :long
    (xdoc::topstring
     (xdoc::p
      "We start by disambiguating the declaration specifiers,
       which may result in extending the disambiguation table.
       In valid code, the @('typedef') storage class specifier
       cannot occur among the declaration specifiers of a parameter declaration,
       so we ignore the disambiguation kind returned by
       the ACL2 function that processes the declaration specifiers;
       when we call that function, we initialize the kind to @(':objfun'),
       and if the code is valid that will be also the returned kind.
       Then we call a separate function to disambiguate the parameter declarator
       (which is a notion in our abstract syntax, not in [C17]);
       see @(tsee paramdeclor))."))
    (b* (((reterr) (irr-paramdecl) (irr-dimb-table))
         ((paramdecl paramdecl) paramdecl)
         ((erp new-spec & table)
          (dimb-decl-spec-list paramdecl.spec (dimb-kind-objfun) table))
         ((erp new-decl table)
          (dimb-paramdeclor paramdecl.decl table)))
      (retok (make-paramdecl :spec new-spec :decl new-decl) table))
    :measure (paramdecl-count paramdecl))

  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  (define dimb-paramdecl-list ((paramdecls paramdecl-listp) (table dimb-tablep))
    :returns (mv erp (new-paramdecls paramdecl-listp) (new-table dimb-tablep))
    :parents (disambiguator dimb-exprs/decls/stmts)
    :short "Disambiguate a list of parameter declarations."
    :long
    (xdoc::topstring
     (xdoc::p
      "We process each one, threading through the table."))
    (b* (((reterr) nil (irr-dimb-table))
         ((when (endp paramdecls)) (retok nil (dimb-table-fix table)))
         ((erp new-paramdecl table)
          (dimb-paramdecl (car paramdecls) table))
         ((erp new-paramdecls table)
          (dimb-paramdecl-list (cdr paramdecls) table)))
      (retok (cons new-paramdecl new-paramdecls) table))
    :measure (paramdecl-list-count paramdecls))

  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  (define dimb-paramdeclor ((paramdeclor paramdeclorp) (table dimb-tablep))
    :returns (mv erp (new-paramdeclor paramdeclorp) (new-table dimb-tablep))
    :parents (disambiguator dimb-exprs/decls/stmts)
    :short "Disambiguate a parameter declarator."
    :long
    (xdoc::topstring
     (xdoc::p
      "A declarator is recursively disambiguated,
       and the identifier is added to the disambiguation table,
       as denoting an object or function.")
     (xdoc::p
      "An abstract declarator is recursively disambiguated,
       but since it does not introduce an identifier,
       the disambiguation table is left unchanged.")
     (xdoc::p
      "An absent parameter declarator is left unchanged,
       and so is the disambiguation table.")
     (xdoc::p
      "An ambiguous declarator or abstract declarator
       is disambiguated to a declarator or abstractor declarator,
       thus re-classifying the parameter declarator.
       If the disambiguation is in favor of a declarator,
       the identifier is also added to the disambiguation table.")
     (xdoc::p
      "Note that we call @(tsee dimb-declor)
       with @('nil') as the @('fundefp') flag,
       because the declarator passed to that function
       is for a parameter, not for a defined function."))
    (b* (((reterr) (irr-paramdeclor) (irr-dimb-table)))
      (paramdeclor-case
       paramdeclor
       :declor
       (b* (((erp new-declor ident table)
             (dimb-declor paramdeclor.unwrap nil table))
            (table (dimb-add-ident ident (dimb-kind-objfun) table)))
         (retok (paramdeclor-declor new-declor) table))
       :absdeclor
       (b* (((erp new-absdeclor table)
             (dimb-absdeclor paramdeclor.unwrap table)))
         (retok (paramdeclor-absdeclor new-absdeclor) (dimb-table-fix table)))
       :none
       (retok (paramdeclor-none) (dimb-table-fix table))
       :ambig
       (b* (((erp declor/absdeclor ident? table)
             (dimb-amb-declor/absdeclor paramdeclor.unwrap table)))
         (declor/absdeclor-case
          declor/absdeclor
          :declor
          (b* (((unless ident?)
                (raise "Internal error: declarator without identifier.")
                (reterr t))
               (table (dimb-add-ident ident? (dimb-kind-objfun) table)))
            (retok (paramdeclor-declor declor/absdeclor.unwrap) table))
          :absdeclor
          (retok (paramdeclor-absdeclor declor/absdeclor.unwrap)
                 (dimb-table-fix table))))))
    :measure (paramdeclor-count paramdeclor))

  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  (define dimb-tyname ((tyname tynamep) (table dimb-tablep))
    :returns (mv erp (new-tyname tynamep) (new-table dimb-tablep))
    :parents (disambiguator dimb-exprs/decls/stmts)
    :short "Disambiguate a type name."
    (b* (((reterr) (irr-tyname) (irr-dimb-table))
         ((tyname tyname) tyname)
         ((erp new-specqual table) (dimb-spec/qual-list tyname.specqual table))
         ((erp new-decl? table) (dimb-absdeclor-option tyname.decl? table)))
      (retok (make-tyname :specqual new-specqual :decl? new-decl?)
             table))
    :measure (tyname-count tyname))

  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  (define dimb-strunispec ((strunispec strunispecp) (table dimb-tablep))
    :returns (mv erp (new-strunispec strunispecp) (new-table dimb-tablep))
    :parents (disambiguator dimb-exprs/decls/stmts)
    :short "Disambiguate a structure or union specifier."
    :long
    (xdoc::topstring
     (xdoc::p
      "The disambiguation table is unaffected as we go through the members;
       the table has no information about structure and union members."))
    (b* (((reterr) (irr-strunispec) (irr-dimb-table))
         ((strunispec strunispec) strunispec)
         ((erp new-members table)
          (dimb-structdecl-list strunispec.members table)))
      (retok (make-strunispec :name strunispec.name :members new-members)
             table))
    :measure (strunispec-count strunispec))

  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  (define dimb-structdecl ((structdecl structdeclp) (table dimb-tablep))
    :returns (mv erp (new-structdecl structdeclp) (new-table dimb-tablep))
    :parents (disambiguator dimb-exprs/decls/stmts)
    :short "Disambiguate a structure declaration."
    (b* (((reterr) (irr-structdecl) (irr-dimb-table)))
      (structdecl-case
       structdecl
       :member
       (b* (((erp new-specqual table)
             (dimb-spec/qual-list structdecl.specqual table))
            ((erp new-declor table)
             (dimb-structdeclor-list structdecl.declor table)))
         (retok (make-structdecl-member :extension structdecl.extension
                                        :specqual new-specqual
                                        :declor new-declor
                                        :attrib structdecl.attrib)
                table))
       :statassert
       (b* (((erp new-statassert table)
             (dimb-statassert structdecl.unwrap table)))
         (retok (structdecl-statassert new-statassert)
                table))
       :empty (retok (structdecl-empty) (dimb-table-fix table))))
    :measure (structdecl-count structdecl))

  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  (define dimb-structdecl-list ((structdecls structdecl-listp)
                                (table dimb-tablep))
    :returns (mv erp (new-structdecls structdecl-listp) (new-table dimb-tablep))
    :parents (disambiguator dimb-exprs/decls/stmts)
    :short "Disambiguate a list of structure declarations."
    (b* (((reterr) nil (irr-dimb-table))
         ((when (endp structdecls)) (retok nil (dimb-table-fix table)))
         ((erp new-structdecl table) (dimb-structdecl (car structdecls) table))
         ((erp new-structdecls table)
          (dimb-structdecl-list (cdr structdecls) table)))
      (retok (cons new-structdecl new-structdecls) table))
    :measure (structdecl-list-count structdecls))

  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  (define dimb-structdeclor ((structdeclor structdeclorp) (table dimb-tablep))
    :returns (mv erp (new-structdeclor structdeclorp) (new-table dimb-tablep))
    :parents (disambiguator dimb-exprs/decls/stmts)
    :short "Disambiguate a structure declarator."
    :long
    (xdoc::topstring
     (xdoc::p
      "Recall that the disambiguation table
       does not keep track of structure and union members.
       This is why we ignore the identifier, if any,
       returned from disambiguating the optional declarator."))
    (b* (((reterr) (irr-structdeclor) (irr-dimb-table))
         ((structdeclor structdeclor) structdeclor)
         ((erp new-declor? & table)
          (dimb-declor-option structdeclor.declor? table))
         ((erp new-expr? table)
          (dimb-const-expr-option structdeclor.expr? table)))
      (retok (make-structdeclor :declor? new-declor? :expr? new-expr?)
             table))
    :measure (structdeclor-count structdeclor))

  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  (define dimb-structdeclor-list ((structdeclors structdeclor-listp)
                                  (table dimb-tablep))
    :returns (mv erp
                 (new-structdeclors structdeclor-listp)
                 (new-table dimb-tablep))
    :parents (disambiguator dimb-exprs/decls/stmts)
    :short "Disambiguate a list of structure declarators."
    (b* (((reterr) nil (irr-dimb-table))
         ((when (endp structdeclors)) (retok nil (dimb-table-fix table)))
         ((erp new-structdeclor table)
          (dimb-structdeclor (car structdeclors) table))
         ((erp new-structdeclors table)
          (dimb-structdeclor-list (cdr structdeclors) table)))
      (retok (cons new-structdeclor new-structdeclors) table))
    :measure (structdeclor-list-count structdeclors))

  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  (define dimb-enumspec ((enumspec enumspecp) (table dimb-tablep))
    :returns (mv erp (new-enumspec enumspecp) (new-table dimb-tablep))
    :parents (disambiguator dimb-exprs/decls/stmts)
    :short "Disambiguate an enumeration specifier."
    :long
    (xdoc::topstring
     (xdoc::p
      "This also extends the disambiguation table
       with the names of the enumerators (i.e. enumeration constants).
       The scope of an enumeration constant starts
       just after the appearance of its enumerator [C17:6.2.1/7].
       The extension of the table is actually done by
       the function that disambiguates the enumerators."))
    (b* (((reterr) (irr-enumspec) (irr-dimb-table))
         ((enumspec enumspec) enumspec)
         ((erp new-list table) (dimb-enumer-list enumspec.list table)))
      (retok (make-enumspec :name enumspec.name
                            :list new-list
                            :final-comma enumspec.final-comma)
             table))
    :measure (enumspec-count enumspec))

  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  (define dimb-enumer ((enumer enumerp) (table dimb-tablep))
    :returns (mv erp (new-enumer enumerp) (new-table dimb-tablep))
    :parents (disambiguator dimb-exprs/decls/stmts)
    :short "Disambiguate an enumerator."
    :long
    (xdoc::topstring
     (xdoc::p
      "We disambiguate the constant expression, if present.
       We also add the name to the disambiguation table."))
    (b* (((reterr) (irr-enumer) (irr-dimb-table))
         ((enumer enumer) enumer)
         ((erp new-value table) (dimb-const-expr-option enumer.value table))
         (table (dimb-add-ident enumer.name (dimb-kind-enumconst) table)))
      (retok (make-enumer :name enumer.name :value new-value) table))
    :measure (enumer-count enumer))

  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  (define dimb-enumer-list ((enumers enumer-listp) (table dimb-tablep))
    :returns (mv erp (new-enumers enumer-listp) (new-table dimb-tablep))
    :parents (disambiguator dimb-exprs/decls/stmts)
    :short "Disambiguate a list of enumerators."
    :long
    (xdoc::topstring
     (xdoc::p
      "Each enumerator updates the disambiguation table,
       which we thread through."))
    (b* (((reterr) nil (irr-dimb-table))
         ((when (endp enumers)) (retok nil (dimb-table-fix table)))
         ((erp new-enumer table) (dimb-enumer (car enumers) table))
         ((erp new-enumers table) (dimb-enumer-list (cdr enumers) table)))
      (retok (cons new-enumer new-enumers) table))
    :measure (enumer-list-count enumers))

  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  (define dimb-statassert ((statassert statassertp) (table dimb-tablep))
    :returns (mv erp (new-statassert statassertp) (new-table dimb-tablep))
    :parents (disambiguator dimb-exprs/decls/stmts)
    :short "Disambiguate a static assertion declaration."
    (b* (((reterr) (irr-statassert) (irr-dimb-table))
         ((statassert statassert) statassert)
         ((erp new-test table) (dimb-const-expr statassert.test table)))
      (retok (make-statassert :test new-test :message statassert.message)
             table))
    :measure (statassert-count statassert))

  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  (define dimb-initdeclor ((ideclor initdeclorp)
                           (kind dimb-kindp)
                           (table dimb-tablep))
    :returns (mv erp (new-ideclor initdeclorp) (new-table dimb-tablep))
    :parents (disambiguator dimb-exprs/decls/stmts)
    :short "Disambiguate an initializer declarator."
    :long
    (xdoc::topstring
     (xdoc::p
      "An initializer declarator is part of a declaration.
       At the end of the initializer declarator,
       the declared identifier is added to the disambiguation table,
       with the appropriate kind,
       which comes from the preceding declaration specifiers,
       and is passed to this function.")
     (xdoc::p
      "We pass @('nil') as the @('fundefp') flag to @(tsee dimb-declor),
       because an initializer declarator is not
       the declarator of a defined function."))
    (b* (((reterr) (irr-initdeclor) (irr-dimb-table))
         ((initdeclor ideclor) ideclor)
         ((erp new-declor ident table) (dimb-declor ideclor.declor nil table))
         ((erp new-init? table) (dimb-initer-option ideclor.init? table))
         (table (dimb-add-ident ident kind table)))
      (retok (make-initdeclor :declor new-declor
                              :asm? ideclor.asm?
                              :attribs ideclor.attribs
                              :init? new-init?)
             table))
    :measure (initdeclor-count ideclor))

  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  (define dimb-initdeclor-list ((ideclors initdeclor-listp)
                                (kind dimb-kindp)
                                (table dimb-tablep))
    :returns (mv erp (new-ideclors initdeclor-listp) (new-table dimb-tablep))
    :parents (disambiguator dimb-exprs/decls/stmts)
    :short "Disambiguate a list of initializer declarators."
    :long
    (xdoc::topstring
     (xdoc::p
      "We process each one, in order.
       The kind is the same for all of them,
       obtained from the enclosing declaration,
       and passed to this function as input."))
    (b* (((reterr) nil (irr-dimb-table))
         ((when (endp ideclors)) (retok nil (dimb-table-fix table)))
         ((erp new-ideclor table) (dimb-initdeclor (car ideclors) kind table))
         ((erp new-ideclors table)
          (dimb-initdeclor-list (cdr ideclors) kind table)))
      (retok (cons new-ideclor new-ideclors) table))
    :measure (initdeclor-list-count ideclors))

  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  (define dimb-decl ((decl declp) (table dimb-tablep))
    :returns (mv erp (new-decl declp) (new-table dimb-tablep))
    :parents (disambiguator dimb-exprs/decls/stmts)
    :short "Disambiguate a declaration."
    :long
    (xdoc::topstring
     (xdoc::p
      "First we process the declaration specifiers,
       which, as explained in @(tsee dimb-decl-spec),
       determine whether the (one or more) identifiers
       introduced by the declarators
       denote @('typedef') names or objects/functions.
       We pass the returned kind to the code that disambiguates
       the initializer declarators."))
    (b* (((reterr) (irr-decl) (irr-dimb-table)))
      (decl-case
       decl
       :decl
       (b* (((erp new-specs kind table)
             (dimb-decl-spec-list decl.specs (dimb-kind-objfun) table))
            ((erp new-init table)
             (dimb-initdeclor-list decl.init kind table)))
         (retok (make-decl-decl :extension decl.extension
                                :specs new-specs
                                :init new-init)
                table))
       :statassert
       (b* (((erp new-statassert table) (dimb-statassert decl.unwrap table)))
         (retok (decl-statassert new-statassert) table))))
    :measure (decl-count decl))

  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  (define dimb-decl-list ((decls decl-listp) (table dimb-tablep))
    :returns (mv erp (new-decls decl-listp) (new-table dimb-tablep))
    :parents (disambiguator dimb-exprs/decls/stmts)
    :short "Disambiguate a list of declarations."
    :long
    (xdoc::topstring
     (xdoc::p
      "The disambiguation table is threaded through."))
    (b* (((reterr) nil (dimb-table-fix table))
         ((when (endp decls)) (retok nil (dimb-table-fix table)))
         ((erp new-decl table) (dimb-decl (car decls) table))
         ((erp new-decls table) (dimb-decl-list (cdr decls) table)))
      (retok (cons new-decl new-decls) table))
    :measure (decl-list-count decls))

  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  (define dimb-label ((label labelp) (table dimb-tablep))
    :returns (mv erp (new-label labelp) (new-table dimb-tablep))
    :parents (disambiguator dimb-exprs/decls/stmts)
    :short "Disambiguate a label."
    :long
    (xdoc::topstring
     (xdoc::p
      "This may affect the disambiguation table,
       for a label that is a constant expression."))
    (b* (((reterr) (irr-label) (irr-dimb-table)))
      (label-case
       label
       :name (retok (label-fix label) (dimb-table-fix table))
       :casexpr (b* (((erp new-expr table)
                      (dimb-const-expr label.expr table))
                     ((erp new-range? table)
                      (dimb-const-expr-option label.range? table)))
                  (retok (make-label-casexpr :expr new-expr
                                             :range? new-range?)
                         table))
       :default (retok (label-fix label) (dimb-table-fix table))))
    :measure (label-count label))

  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  (define dimb-stmt ((stmt stmtp) (table dimb-tablep))
    :returns (mv erp (new-stmt stmtp) (new-table dimb-tablep))
    :parents (disambiguator dimb-exprs/decls/stmts)
    :short "Disambiguate a statement."
    :long
    (xdoc::topstring
     (xdoc::p
      "A compound statement form a new scope.
       Thus we push a new scope before the block,
       which we pop after the block.")
     (xdoc::p
      "A selection statement forms a new scope, as do its sub-statements
       [C17:6.8.4/3].")
     (xdoc::p
      "An iteration statement forms a new scope, as do its sub-statements
       [C17:6.8.5/5]."))
    (b* (((reterr) (irr-stmt) (irr-dimb-table)))
      (stmt-case
       stmt
       :labeled
       (b* (((erp new-label table) (dimb-label stmt.label table))
            ((erp new-stmt table) (dimb-stmt stmt.stmt table)))
         (retok (make-stmt-labeled :label new-label :stmt new-stmt)
                table))
       :compound
       (b* ((table (dimb-push-scope table))
            ((erp new-items table) (dimb-block-item-list stmt.items table))
            (table (dimb-pop-scope table)))
         (retok (stmt-compound new-items) table))
       :expr
       (b* (((erp new-expr? table) (dimb-expr-option stmt.expr? table)))
         (retok (stmt-expr new-expr?) table))
       :if
       (b* ((table (dimb-push-scope table))
            ((erp new-test table) (dimb-expr stmt.test table))
            (table (dimb-push-scope table))
            ((erp new-then table) (dimb-stmt stmt.then table))
            (table (dimb-pop-scope table))
            (table (dimb-pop-scope table)))
         (retok (make-stmt-if :test new-test :then new-then) table))
       :ifelse
       (b* ((table (dimb-push-scope table))
            ((erp new-test table) (dimb-expr stmt.test table))
            (table (dimb-push-scope table))
            ((erp new-then table) (dimb-stmt stmt.then table))
            (table (dimb-pop-scope table))
            (table (dimb-push-scope table))
            ((erp new-else table) (dimb-stmt stmt.else table))
            (table (dimb-pop-scope table))
            (table (dimb-pop-scope table)))
         (retok (make-stmt-ifelse :test new-test :then new-then :else new-else)
                table))
       :switch
       (b* ((table (dimb-push-scope table))
            ((erp new-target table) (dimb-expr stmt.target table))
            (table (dimb-push-scope table))
            ((erp new-body table) (dimb-stmt stmt.body table))
            (table (dimb-pop-scope table))
            (table (dimb-pop-scope table)))
         (retok (make-stmt-switch :target new-target :body new-body) table))
       :while
       (b* ((table (dimb-push-scope table))
            ((erp new-test table) (dimb-expr stmt.test table))
            (table (dimb-push-scope table))
            ((erp new-body table) (dimb-stmt stmt.body table))
            (table (dimb-pop-scope table))
            (table (dimb-pop-scope table)))
         (retok (make-stmt-while :test new-test :body new-body) table))
       :dowhile
       (b* ((table (dimb-push-scope table))
            (table (dimb-push-scope table))
            ((erp new-body table) (dimb-stmt stmt.body table))
            (table (dimb-pop-scope table))
            ((erp new-test table) (dimb-expr stmt.test table))
            (table (dimb-pop-scope table)))
         (retok (make-stmt-dowhile :body new-body :test new-test) table))
       :for-expr
       (b* ((table (dimb-push-scope table))
            ((erp new-init table) (dimb-expr-option stmt.init table))
            ((erp new-test table) (dimb-expr-option stmt.test table))
            ((erp new-next table) (dimb-expr-option stmt.next table))
            (table (dimb-push-scope table))
            ((erp new-body table) (dimb-stmt stmt.body table))
            (table (dimb-pop-scope table)))
         (retok (make-stmt-for-expr :init new-init
                                    :test new-test
                                    :next new-next
                                    :body new-body)
                table))
       :for-decl
       (b* ((table (dimb-push-scope table))
            ((erp new-init table) (dimb-decl stmt.init table))
            ((erp new-test table) (dimb-expr-option stmt.test table))
            ((erp new-next table) (dimb-expr-option stmt.next table))
            (table (dimb-push-scope table))
            ((erp new-body table) (dimb-stmt stmt.body table))
            (table (dimb-pop-scope table)))
         (retok (make-stmt-for-decl :init new-init
                                    :test new-test
                                    :next new-next
                                    :body new-body)
                table))
       :for-ambig
       (b* ((table (dimb-push-scope table))
            ((erp decl/expr table) (dimb-amb-decl/stmt stmt.init table))
            ((erp new-test table) (dimb-expr-option stmt.test table))
            ((erp new-next table) (dimb-expr-option stmt.next table))
            (table (dimb-push-scope table))
            ((erp new-body table) (dimb-stmt stmt.body table))
            (table (dimb-pop-scope table)))
         (decl/stmt-case
          decl/expr
          :decl (retok (make-stmt-for-decl :init decl/expr.unwrap
                                           :test new-test
                                           :next new-next
                                           :body new-body)
                       table)
          :stmt (retok (make-stmt-for-expr :init decl/expr.unwrap
                                           :test new-test
                                           :next new-next
                                           :body new-body)
                       table)))
       :goto
       (retok (stmt-fix stmt) (dimb-table-fix table))
       :continue
       (retok (stmt-fix stmt) (dimb-table-fix table))
       :break
       (retok (stmt-fix stmt) (dimb-table-fix table))
       :return
       (b* (((erp new-expr? table) (dimb-expr-option stmt.expr? table)))
         (retok (stmt-return new-expr?) table))
       :asm
       (retok (stmt-fix stmt) (dimb-table-fix table))))
    :measure (stmt-count stmt))

  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  (define dimb-block-item ((item block-itemp) (table dimb-tablep))
    :returns (mv erp (new-item block-itemp) (new-table dimb-tablep))
    :parents (disambiguator dimb-exprs/decls/stmts)
    :short "Disambiguate a block item."
    :long
    (xdoc::topstring
     (xdoc::p
      "An ambiguous declaration or (expression) statement
       is disambiguated and re-classified."))
    (b* (((reterr) (irr-block-item) (irr-dimb-table)))
      (block-item-case
       item
       :decl
       (b* (((erp new-decl table) (dimb-decl item.unwrap table)))
         (retok (block-item-decl new-decl) table))
       :stmt
       (b* (((erp new-stmt table) (dimb-stmt item.unwrap table)))
         (retok (block-item-stmt new-stmt) table))
       :ambig
       (b* (((erp decl/stmt table) (dimb-amb-decl/stmt item.unwrap table)))
         (decl/stmt-case
          decl/stmt
          :decl (retok (block-item-decl decl/stmt.unwrap) table)
          :stmt (retok (block-item-stmt (stmt-expr decl/stmt.unwrap)) table)))))
    :measure (block-item-count item))

  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  (define dimb-block-item-list ((items block-item-listp) (table dimb-tablep))
    :returns (mv erp (new-items block-item-listp) (new-table dimb-tablep))
    :parents (disambiguator dimb-exprs/decls/stmts)
    :short "Disambiguate a list of block items."
    (b* (((reterr) nil (irr-dimb-table))
         ((when (endp items)) (retok nil (dimb-table-fix table)))
         ((erp new-item table) (dimb-block-item (car items) table))
         ((erp new-items table) (dimb-block-item-list (cdr items) table)))
      (retok (cons new-item new-items) table))
    :measure (block-item-list-count items))

  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  (define dimb-amb-expr/tyname ((expr/tyname amb-expr/tyname-p)
                                (add-parens-p booleanp)
                                (table dimb-tablep))
    :returns (mv erp (expr-or-tyname expr/tyname-p) (new-table dimb-tablep))
    :parents (disambiguator dimb-exprs/decls/stmts)
    :short "Disambiguate an ambiguous expression or type name."
    :long
    (xdoc::topstring
     (xdoc::p
      "An ambiguous expression or type name is represented as
       a pair of an expression and a type name
       (with the same concrete syntax appearance).
       We attempt to disambiguate both the expression and the type name,
       independently from each other.
       In valid code, one of them must succeed and the other one must fail:
       then we disambiguate in favor of the one that succeeded.
       If none or both succeed, the code must be invalid.")
     (xdoc::p
      "If the ambiguous expression or type name
       is disambiguated to an expression,
       if the @('add-parens-p') flag is @('t')
       we parenthesize the expression.
       This is needed because, for instance,
       in a @('sizeof(A)') expression where A is
       a possibly ambiguous expression or type name,
       the actual expression would be @('(A)'), not just @('A'),
       because @('sizeof') can be applied to
       an unparenthesized unary expression (e.g. @('sizeof x')).
       In this case, the @('add-parens-p') is set to @('t')
       by the caller of this disambiguation function.
       On the other hand, in a construct like @('_Alignas(A)'),
       where @('A') is a possibly ambiguous expression or type name,
       the expression is just @('A'),
       because the parentheses are always required:
       they are part of the syntax of @('_Alignas'),
       not part of the expression as in the case of
       @('sizeof') applied to an expression.
       In this case, the @('add-parens-p') flag is set to @('nil')
       by the caller of this disambiguation function."))
    (b* (((reterr) (irr-expr/tyname) (irr-dimb-table))
         ((amb-expr/tyname expr/tyname) expr/tyname)
         ((mv erp-expr new-expr table-expr)
          (dimb-expr expr/tyname.expr table))
         ((mv erp-tyname new-tyname table-tyname)
          (dimb-tyname expr/tyname.tyname table)))
      (if erp-expr
          ;; expr fails:
          (if erp-tyname
              ;; tyname fails:
              (reterr (msg "In the ambiguous expression or type name ~x0, ~
                            neither the expression nor the type name ~
                            can be successfully disambiguated. ~
                            The code must be invalid, ~
                            because at least one must succeed.~%~%~
                            These are the failures for each:~%~%~
                            ~@1~%~%~@2"
                           (amb-expr/tyname-fix expr/tyname)
                           erp-expr
                           erp-tyname))
            ;; tyname succeeds:
            (retok (expr/tyname-tyname new-tyname) table-tyname))
        ;; expr succeeds:
        (if erp-tyname
            ;; tyname fails:
            (b* ((new-expr (if add-parens-p
                               (expr-paren new-expr)
                             new-expr)))
              (retok (expr/tyname-expr new-expr) table-expr))
          ;; tyname succeeds:
          (reterr (msg "In the ambiguous expression or type name ~x0, ~
                        both the expression and the type name ~
                        are successfully disambiguated. ~
                        The code must be invalid, ~
                        because at most one must succeed."
                       (amb-expr/tyname-fix expr/tyname))))))
    :measure (amb-expr/tyname-count expr/tyname))

  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  (define dimb-amb-declor/absdeclor ((declor/absdeclor amb-declor/absdeclor-p)
                                     (table dimb-tablep))
    :returns (mv erp
                 (declor-or-absdeclor declor/absdeclor-p)
                 (ident? ident-optionp)
                 (new-table dimb-tablep))
    :parents (disambiguator dimb-exprs/decls/stmts)
    :short "Disambiguate an ambiguous declarator or abstract declarator."
    :long
    (xdoc::topstring
     (xdoc::p
      "An ambiguous declarator or abstract declarator is represented as
       a pair of a declarator and an abstract declarator
       (with the same concrete syntax appearance).
       We attempt to disambiguate
       both the declarator and the abstract declarator,
       independently from each other.
       If both fail, the code is invalid.
       If one of them succeeds while the other fails,
       we have a disambiguation.
       There are cases in which both can succeed,
       for instance in @('void f(int(x));')
       the @('(x)') could be a parenthesized declarator for identifier @('x'),
       but if a @('typedef') for @('x') is in scope,
       it could be also an abstract function declarator,
       where @('x') is the type of the (inner) parameter.
       [C17:6.7.6.3/11] says that in this case
       the @('typedef') interpretation takes priority.
       Thus, in general,
       if both attempted disambiguations
       (as declarator and as abstract declarator)
       succeed, we check if
       the identifier returned by the successful disambiguation as declarator
       denotes a @('typedef') name:
       if the check succeeds,
       we keep the disambiguation as abstract declarator
       and discard the one as declarator.
       If instead the check fails, we return an error:
       we conjecture that this should only happen if the code is invalid,
       but this needs further investigation.")
     (xdoc::p
      "If the ambiguous declarator or abstract declarator
       turns out to be a declarator,
       we also return the identifier it declares.
       If it turns out to be an abstract declarator instead,
       we return @('nil').
       So, in general, this function returns an optional identifier,
       besides the disambiguated declarator or abstract declarator.")
     (xdoc::p
      "In the call of @(tsee dimb-declor)
       we pass @('nil') as the @('fundefp') flag,
       because if we are disambiguating a declarator or abstract declarator,
       it means that we are disambiguating a parameter declarator,
       and not the declarator of a defined function."))
    (b* (((reterr) (irr-declor/absdeclor) nil (irr-dimb-table))
         ((amb-declor/absdeclor declor/absdeclor) declor/absdeclor)
         ((mv erp-declor new-declor ident table-declor)
          (dimb-declor declor/absdeclor.declor nil table))
         ((mv erp-absdeclor new-absdeclor table-absdeclor)
          (dimb-absdeclor declor/absdeclor.absdeclor table)))
      (if erp-declor
          ;; declor fails:
          (if erp-absdeclor
              ;; absdeclor fails:
              (reterr (msg "In the ambiguous ~
                            declarator or abstract declarator ~x0, ~
                            neither the declarator nor the abstract declarator ~
                            can be successfully disambiguated. ~
                            The code must be invalid, ~
                            because at least one must succeed.~%~%~
                            These are the failures for each:~%~%~
                            ~@1~%~%~@2"
                           (amb-declor/absdeclor-fix declor/absdeclor)
                           erp-declor
                           erp-absdeclor))
            ;; absdeclor succeeds:
            (retok (declor/absdeclor-absdeclor new-absdeclor)
                   nil
                   table-absdeclor))
        ;; declor succeeds:
        (if erp-absdeclor
            ;; absdeclor fails:
            (retok (declor/absdeclor-declor new-declor)
                   ident
                   table-declor)
          ;; absdeclor succeeds:
          (b* ((kind (dimb-lookup-ident ident table)))
            (if (equal kind (dimb-kind-typedef))
                (retok (declor/absdeclor-absdeclor new-absdeclor)
                       nil
                       table-absdeclor)
              (reterr (msg "In the ambiguous ~
                            declarator or abstract declarator ~x0, ~
                            both the declarator and the abstract declarator ~
                            are successfully disambiguated, ~
                            and the identifier ~x1 in the declarator ~
                            is not a typedef name. ~
                            The code must be invalid, ~
                            because at most one must succeed."
                           (amb-declor/absdeclor-fix declor/absdeclor)
                           ident)))))))
    :measure (amb-declor/absdeclor-count declor/absdeclor))

  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  (define dimb-amb-decl/stmt ((decl/stmt amb-decl/stmt-p) (table dimb-tablep))
    :returns (mv erp (decl-or-stmt decl/stmt-p) (new-table dimb-tablep))
    :parents (disambiguator dimb-exprs/decls/stmts)
    :short "Disambiguate an ambiguous declaration or statement."
    :long
    (xdoc::topstring
     (xdoc::p
      "An ambiguous declaration or statement is represented as
       a pair of a declaration and an expression
       (with the same concrete syntax appearance);
       the latter is an expression and not a statement because
       the statement in question is always an expression statement,
       and so it suffices to represent the expression.
       We attempt to disambiguate both the declaration and the expression,
       independently from each other.
       In valid code, one of them must succeed and the other one must fail:
       then we disambiguate in favor of the one that succeeded.
       If none or both succeed, the code must be invalid."))
    (b* (((reterr) (irr-decl/stmt) (irr-dimb-table))
         ((amb-decl/stmt decl/stmt) decl/stmt)
         ((mv erp-decl new-decl table-decl) (dimb-decl decl/stmt.decl table))
         ((mv erp-expr new-expr table-expr) (dimb-expr decl/stmt.stmt table)))
      (if erp-decl
          ;; decl fails:
          (if erp-expr
              ;; stmt fails:
              (reterr (msg "In the ambiguous declaration or statement ~x0, ~
                            neither the declaration nor the expression ~
                            can be successfully disambiguated. ~
                            The code must be invalid, ~
                            because at least one must succeed.~%~%~
                            These are the failures for each:~%~%~
                            ~@1~%~%~@2"
                           (amb-decl/stmt-fix decl/stmt)
                           erp-decl
                           erp-expr))
            ;; stmt succeeds:
            (retok (decl/stmt-stmt new-expr) table-expr))
        ;; decl succeeds:
        (if erp-expr
            ;; stmt fails:
            (retok (decl/stmt-decl new-decl) table-decl)
          ;; stmt succeeds:
          (reterr (msg "In the ambiguous declaration or statement ~x0, ~
                        both the declaration and the statement ~
                        are successfully disambiguated. ~
                        The code must be invalid, ~
                        because at most one must succeed."
                       (amb-decl/stmt-fix decl/stmt))))))
    :measure (amb-decl/stmt-count decl/stmt))

  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  :hints (("Goal" :in-theory (enable o< o-finp)))

  :verify-guards nil ; done below

  ///

  (fty::deffixequiv-mutual dimb-exprs/decls/stmts)

  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  (defret-mutual unambp-of-dimb-exprs/decls/stmts
    (defret expr-unambp-of-dimb-expr
      (implies (not erp)
               (expr-unambp new-expr))
      :fn dimb-expr)
    (defret expr-list-unambp-of-dimb-expr-list
      (implies (not erp)
               (expr-list-unambp new-exprs))
      :fn dimb-expr-list)
    (defret expr-option-unambp-of-dimb-expr-option
      (implies (not erp)
               (expr-option-unambp new-expr?))
      :fn dimb-expr-option)
    (defret const-expr-unambp-of-dimb-const-expr
      (implies (not erp)
               (const-expr-unambp new-cexpr))
      :fn dimb-const-expr)
    (defret const-expr-option-unambp-of-dimb-const-expr-option
      (implies (not erp)
               (const-expr-option-unambp new-cexpr?))
      :fn dimb-const-expr-option)
    (defret genassoc-unambp-of-dimb-genassoc
      (implies (not erp)
               (genassoc-unambp new-assoc))
      :fn dimb-genassoc)
    (defret genassoc-list-unambp-of-dimb-genassoc-list
      (implies (not erp)
               (genassoc-list-unambp new-assocs))
      :fn dimb-genassoc-list)
    (defret member-designor-unambp-of-dimb-member-designor
      (implies (not erp)
               (member-designor-unambp new-memdes))
      :fn dimb-member-designor)
    (defret type-spec-unambp-of-dimb-type-spec
      (implies (not erp)
               (type-spec-unambp new-tyspec))
      :fn dimb-type-spec)
    (defret spec/qual-unambp-of-dimb-spec/qual
      (implies (not erp)
               (spec/qual-unambp new-specqual))
      :fn dimb-spec/qual)
    (defret spec/qual-list-unambp-of-dimb-spec/qual-list
      (implies (not erp)
               (spec/qual-list-unambp new-specquals))
      :fn dimb-spec/qual-list)
    (defret align-spec-unambp-of-dimb-align-spec
      (implies (not erp)
               (align-spec-unambp new-alignspec))
      :fn dimb-align-spec)
    (defret decl-spec-unambp-of-dimb-decl-spec
      (implies (not erp)
               (decl-spec-unambp new-declspec))
      :fn dimb-decl-spec)
    (defret decl-spec-list-unambp-of-dimb-decl-spec-list
      (implies (not erp)
               (decl-spec-list-unambp new-declspecs))
      :fn dimb-decl-spec-list)
    (defret initer-unambp-of-dimb-initer
      (implies (not erp)
               (initer-unambp new-initer))
      :fn dimb-initer)
    (defret initer-option-unambp-of-dimb-initer-option
      (implies (not erp)
               (initer-option-unambp new-initer?))
      :fn dimb-initer-option)
    (defret desiniter-unambp-of-dimb-desiniter
      (implies (not erp)
               (desiniter-unambp new-desiniter))
      :fn dimb-desiniter)
    (defret desiniter-list-unambp-of-dimb-desiniter-list
      (implies (not erp)
               (desiniter-list-unambp new-desiniters))
      :fn dimb-desiniter-list)
    (defret designor-unambp-of-dimb-designor
      (implies (not erp)
               (designor-unambp new-design))
      :fn dimb-designor)
    (defret designor-list-unambp-of-dimb-designor-list
      (implies (not erp)
               (designor-list-unambp new-designs))
      :fn dimb-designor-list)
    (defret declor-unambp-of-dimb-declor
      (implies (not erp)
               (declor-unambp new-declor))
      :fn dimb-declor)
    (defret declor-option-unambp-of-dimb-declor-option
      (implies (not erp)
               (declor-option-unambp new-declor?))
      :fn dimb-declor-option)
    (defret dirdeclor-unambp-of-dimb-dirdeclor
      (implies (not erp)
               (dirdeclor-unambp new-dirdeclor))
      :fn dimb-dirdeclor)
    (defret absdeclor-unambp-of-dimb-absdeclor
      (implies (not erp)
               (absdeclor-unambp new-absdeclor))
      :fn dimb-absdeclor)
    (defret absdeclor-option-unambp-of-dimb-absdeclor-option
      (implies (not erp)
               (absdeclor-option-unambp new-absdeclor?))
      :fn dimb-absdeclor-option)
    (defret dirabsdeclor-unambp-of-dimb-dirabsdeclor
      (implies (not erp)
               (dirabsdeclor-unambp new-dirabsdeclor))
      :fn dimb-dirabsdeclor)
    (defret dirabsdeclor-option-unambp-of-dimb-dirabsdeclor-option
      (implies (not erp)
               (dirabsdeclor-option-unambp new-dirabsdeclor?))
      :fn dimb-dirabsdeclor-option)
    (defret paramdecl-unambp-of-dimb-paramdecl
      (implies (not erp)
               (paramdecl-unambp new-paramdecl))
      :fn dimb-paramdecl)
    (defret paramdecl-list-unambp-of-dimb-paramdecl-list
      (implies (not erp)
               (paramdecl-list-unambp new-paramdecls))
      :fn dimb-paramdecl-list)
    (defret paramdeclor-unambp-of-dimb-paramdeclor
      (implies (not erp)
               (paramdeclor-unambp new-paramdeclor))
      :fn dimb-paramdeclor)
    (defret tyname-unambp-of-dimb-tyname
      (implies (not erp)
               (tyname-unambp new-tyname))
      :fn dimb-tyname)
    (defret strunispec-unambp-of-dimb-strunispec
      (implies (not erp)
               (strunispec-unambp new-strunispec))
      :fn dimb-strunispec)
    (defret structdecl-unambp-of-dimb-structdecl
      (implies (not erp)
               (structdecl-unambp new-structdecl))
      :fn dimb-structdecl)
    (defret structdecl-list-unambp-of-dimb-structdecl-list
      (implies (not erp)
               (structdecl-list-unambp new-structdecls))
      :fn dimb-structdecl-list)
    (defret structdeclor-unambp-of-dimb-structdeclor
      (implies (not erp)
               (structdeclor-unambp new-structdeclor))
      :fn dimb-structdeclor)
    (defret structdeclor-list-unambp-of-dimb-structdeclor-list
      (implies (not erp)
               (structdeclor-list-unambp new-structdeclors))
      :fn dimb-structdeclor-list)
    (defret enumspec-unambp-of-dimb-enumspec
      (implies (not erp)
               (enumspec-unambp new-enumspec))
      :fn dimb-enumspec)
    (defret enumer-unambp-of-dimb-enumer
      (implies (not erp)
               (enumer-unambp new-enumer))
      :fn dimb-enumer)
    (defret enumer-list-unambp-of-dimb-enumer-list
      (implies (not erp)
               (enumer-list-unambp new-enumers))
      :fn dimb-enumer-list)
    (defret statassert-unambp-of-dimb-statassert
      (implies (not erp)
               (statassert-unambp new-statassert))
      :fn dimb-statassert)
    (defret initdeclor-unambp-of-dimb-initdeclor
      (implies (not erp)
               (initdeclor-unambp new-ideclor))
      :fn dimb-initdeclor)
    (defret initdeclor-list-unambp-of-dimb-initdeclor-list
      (implies (not erp)
               (initdeclor-list-unambp new-ideclors))
      :fn dimb-initdeclor-list)
    (defret decl-unambp-of-dimb-decl
      (implies (not erp)
               (decl-unambp new-decl))
      :fn dimb-decl)
    (defret decl-list-unambp-of-dimb-decl-list
      (implies (not erp)
               (decl-list-unambp new-decls))
      :fn dimb-decl-list)
    (defret label-unambp-of-dimb-label
      (implies (not erp)
               (label-unambp new-label))
      :fn dimb-label)
    (defret stmt-unambp-of-dimb-stmt
      (implies (not erp)
               (stmt-unambp new-stmt))
      :fn dimb-stmt)
    (defret block-item-unambp-of-dimb-block-item
      (implies (not erp)
               (block-item-unambp new-item))
      :fn dimb-block-item)
    (defret block-item-list-unambp-of-dimb-block-item-list
      (implies (not erp)
               (block-item-list-unambp new-items))
      :fn dimb-block-item-list)
    (defret expr/tyname-unambp-of-dimb-amb-expr/tyname
      (implies (not erp)
               (expr/tyname-unambp expr-or-tyname))
      :fn dimb-amb-expr/tyname)
    (defret declor/absdeclor-unambp-of-dimb-amb-declor/absdeclor
      (implies (not erp)
               (declor/absdeclor-unambp declor-or-absdeclor))
      :fn dimb-amb-declor/absdeclor)
    (defret decl/stmt-unambp-of-dimb-amb-decl/stmt
      (implies (not erp)
               (decl/stmt-unambp decl-or-stmt))
      :fn dimb-amb-decl/stmt))

  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  (verify-guards dimb-expr))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define dimb-fundef ((fundef fundefp) (table dimb-tablep))
  :returns (mv erp (new-fundef fundefp) (new-table dimb-tablep))
  :short "Disambiguate a function definition."
  :long
  (xdoc::topstring
   (xdoc::p
    "We process the declaration specifiers,
     obtaining the kind of the identifier declared by the declarator,
     which in valid code must be @(':objfun'),
     but we do not check this explicitly.")
   (xdoc::p
    "Then we process the declarator, passing @('t') as the @('fundefp') flag,
     because we are processing the declarator of a defined function.
     In valid code, this declarator will include a function declarator
     with either parameter declarations or identifiers,
     after it has been processed.
     Because of the @('fundefp') flag set to @('t'),
     the disambiguation table returned from @(tsee dimb-declor)
     will contain a newly pushed scope for the function definition.
     If the (disambiguated) declarator has parameter declarations,
     those will have added the formal parameters of the function to that scope.
     If instead the (disambiguated) declarator has just identifiers,
     the new scope will be empty,
     but the declarator will be followed, in the function definition,
     by declarations for the identifiers (again, assuming the code is valid).")
   (xdoc::p
    "We process those declarations, which will add the function parameters
     to the scope that was added when processing the declarator.")
   (xdoc::p
    "Then we add the declared function to the disambiguation table,
     so that it can be referenced from the body, in a recursive call.")
   (xdoc::p
    "We extend the disambiguation table with the identifier @('__func__')
     [C17:6.4.2.2].")
   (xdoc::p
    "After all of that, we disambiguate the body of the function definition,
     which is a block (i.e. compound statement) in valid code.
     But we do not push a new scope for the block,
     because the scope pushed by @(tsee dimb-declor)
     is already the one for the function body.")
   (xdoc::p
    "At the end, we pop the scope for the function definition,
     and we add the function to the table,
     so that it is available in the rest of the translation unit."))
  (b* (((reterr) (irr-fundef) (irr-dimb-table))
       ((fundef fundef) fundef)
       ((erp new-spec & table)
        (dimb-decl-spec-list fundef.spec (dimb-kind-objfun) table))
       ((erp new-declor ident table) (dimb-declor fundef.declor t table))
       ((erp new-decls table) (dimb-decl-list fundef.decls table))
       (table (dimb-add-ident-objfun ident table))
       (table (dimb-add-ident-objfun (ident "__func__") table))
       ((unless (stmt-case fundef.body :compound))
        (reterr (msg "The body of the function definition ~x0 ~
                      is not a compound statement; ~
                      the code is invalid."
                     (fundef-fix fundef))))
       ((erp new-items table)
        (dimb-block-item-list (stmt-compound->items fundef.body) table))
       (table (dimb-pop-scope table))
       (table (dimb-add-ident ident (dimb-kind-objfun) table)))
    (retok (make-fundef :extension fundef.extension
                        :spec new-spec
                        :declor new-declor
                        :decls new-decls
                        :body (stmt-compound new-items))
           table))
  :hooks (:fix)

  ///

  (defret fundef-unambp-of-dimb-fundef
    (implies (not erp)
             (fundef-unambp new-fundef))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define dimb-extdecl ((extdecl extdeclp) (table dimb-tablep))
  :returns (mv erp (new-extdecl extdeclp) (new-table dimb-tablep))
  :short "Disambiguate an external declaration."
  (b* (((reterr) (irr-extdecl) (irr-dimb-table)))
    (extdecl-case
     extdecl
     :fundef
     (b* (((erp new-fundef table) (dimb-fundef extdecl.unwrap table)))
       (retok (extdecl-fundef new-fundef) table))
     :decl
     (b* (((erp new-decl table) (dimb-decl extdecl.unwrap table)))
       (retok (extdecl-decl new-decl) table))
     :empty
     (retok (extdecl-fix extdecl) (dimb-table-fix table))
     :asm
     (retok (extdecl-fix extdecl) (dimb-table-fix table))))
  :hooks (:fix)

  ///

  (defret extdecl-unambp-of-dimb-extdecl
    (implies (not erp)
             (extdecl-unambp new-extdecl))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define dimb-extdecl-list ((edecls extdecl-listp) (table dimb-tablep))
  :returns (mv erp (new-edecls extdecl-listp) (new-table dimb-tablep))
  :short "Disambiguate a list of external declarations."
  (b* (((reterr) nil (irr-dimb-table))
       ((when (endp edecls)) (retok nil (dimb-table-fix table)))
       ((erp new-edecl table) (dimb-extdecl (car edecls) table))
       ((erp new-edecls table) (dimb-extdecl-list (cdr edecls) table)))
    (retok (cons new-edecl new-edecls) table))
  :hooks (:fix)

  ///

  (defret extdecl-list-unambp-of-dimb-extdecl-list
    (implies (not erp)
             (extdecl-list-unambp new-edecls))
    :hints (("Goal" :induct t))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define dimb-transunit ((tunit transunitp) (gcc booleanp))
  :returns (mv erp (new-tunit transunitp))
  :short "Disambiguate a translation unit."
  :long
  (xdoc::topstring
   (xdoc::p
    "We initialize the disambiguation table,
     we disambiguate all the external declarations in order,
     and we discard the final disambiguation table.")
   (xdoc::p
    "If the GCC flag is @('nil') (i.e. no GCC extensions),
     the initial disambiguation table is empty.
     If the flag is @('t'), for now the only difference is that
     we initialize the disambiguation table with some GCC built-ins.
     For now we only add some built-ins
     that we have observed in some preprocessed files.
     We should revisit this, adding all the GCC built-ins,
     with clear and accurate references.")
   (xdoc::p
    "We also add entries for certain built-in variables
     corresponding to the x86 registers, i.e. @('__eax') etc.
     We could not find those documented in the GCC manual,
     but we found them in practical code.
     Experiments suggest that these variables are somewhat restricted in usage.
     The normal pattern seems to be something like")
   (xdoc::codeblock
    "unsigned long __eax = __eax;")
   (xdoc::p
    "after which one can use @('__eax') as a regular variable.
     However, without the declaration above,
     @('__eax') cannot be used as a regular variable.
     This is odd, because the validity of the declaration above
     presupposes that @('__eax') is already in scope.
     It is not clear why such a declaration is needed in the first place.
     To add to the strangeness,
     one can change the above initializer to @('__eax + 1')
     (and presumably other similar expressions)
     and the compiler acceptes it.")
   (xdoc::p
    "However, none of this matters for the disambiguator,
     which does not need to validate the code,
     and is only required to return correct results
     only if the code is indeed valid
     (even though validity is checked after disambiguation).
     We add these special variables to the initial disambiguation table,
     so that declarations such as the one above
     do not cause an error during disambiguation.
     The declaration itself is handled by the disambiguator
     by overriding any preceding entry with the same name
     (see @(tsee dimb-add-ident)),
     so after a declaration like the one above
     @('__eax') is still in the table, with the right kind,
     and can be used as an expression in scope.
     However, note that these variables only make sense on an x86 platform:
     we should refine our GCC flag with
     a richer description of the C implementation."))
  (b* (((reterr) (irr-transunit))
       (edecls (transunit->decls tunit))
       (table (dimb-init-table))
       (table
         (if gcc
             (dimb-add-idents-objfun
              (list (ident "__atomic_signal_fence")
                    (ident "__builtin_add_overflow")
                    (ident "__builtin_bswap16")
                    (ident "__builtin_bswap32")
                    (ident "__builtin_bswap64")
                    (ident "__builtin_choose_expr")
                    (ident "__builtin_clz")
                    (ident "__builtin_clzl")
                    (ident "__builtin_clzll")
                    (ident "__builtin_constant_p")
                    (ident "__builtin_ctzl")
                    (ident "__builtin_dynamic_object_size")
                    (ident "__builtin_expect")
                    (ident "__builtin_memchr")
                    (ident "__builtin_memcmp")
                    (ident "__builtin_memcpy")
                    (ident "__builtin_memset")
                    (ident "__builtin_mul_overflow")
                    (ident "__builtin_object_size")
                    (ident "__builtin_return_address")
                    (ident "__builtin_strcpy")
                    (ident "__builtin_strlen")
                    (ident "__builtin_strncat")
                    (ident "__builtin_strncpy")
                    (ident "__builtin_sub_overflow")
                    (ident "__builtin_unreachable")
                    (ident "__builtin_va_end")
                    (ident "__builtin_va_start")
                    (ident "__eax")
                    (ident "__ebx")
                    (ident "__ecx")
                    (ident "__edx")
                    (ident "__esi")
                    (ident "__edi")
                    (ident "__ebp")
                    (ident "__esp")
                    (ident "__sync_add_and_fetch")
                    (ident "__sync_and_and_fetch")
                    (ident "__sync_bool_compare_and_swap")
                    (ident "__sync_fetch_and_add")
                    (ident "__sync_fetch_and_and")
                    (ident "__sync_fetch_and_nand")
                    (ident "__sync_fetch_and_or")
                    (ident "__sync_fetch_and_sub")
                    (ident "__sync_fetch_and_xor")
                    (ident "__sync_lock_release")
                    (ident "__sync_lock_test_and_set")
                    (ident "__sync_nand_and_fetch")
                    (ident "__sync_or_and_fetch")
                    (ident "__sync_sub_and_fetch")
                    (ident "__sync_synchronize")
                    (ident "__sync_val_compare_and_swap")
                    (ident "__sync_xor_and_fetch"))
              table)
           table))
       ((erp new-edecls &) (dimb-extdecl-list edecls table)))
    (retok (make-transunit :decls new-edecls :info nil)))
  :hooks (:fix)

  ///

  (defret transunit-unambp-of-dimb-transunit
    (implies (not erp)
             (transunit-unambp new-tunit))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define dimb-transunit-ensemble ((tuens transunit-ensemblep) (gcc booleanp))
  :returns (mv erp (new-tuens transunit-ensemblep))
  :short "Disambiguate a translation unit ensembles."
  :long
  (xdoc::topstring
   (xdoc::p
    "We also pass a flag saying whether GCC extensions should be accepted.")
   (xdoc::p
    "We disambiguate all the translation units, independently.
     We leave the file path mapping unchanged."))
  (b* (((reterr) (irr-transunit-ensemble))
       (tumap (transunit-ensemble->unwrap tuens))
       ((erp new-tumap) (dimb-transunit-ensemble-loop tumap gcc)))
    (retok (transunit-ensemble new-tumap)))
  :hooks (:fix)

  :prepwork

  ((define dimb-transunit-ensemble-loop ((tumap filepath-transunit-mapp)
                                         (gcc booleanp))
     :returns (mv erp (new-tumap filepath-transunit-mapp
                                 :hyp (filepath-transunit-mapp tumap)))
     :parents nil
     (b* (((reterr) nil)
          ((when (omap::emptyp tumap)) (retok nil))
          ((mv path tunit) (omap::head tumap))
          ((erp new-tunit) (dimb-transunit tunit gcc))
          ((erp new-tumap)
           (dimb-transunit-ensemble-loop (omap::tail tumap) gcc)))
       (retok (omap::update path new-tunit new-tumap)))
     :verify-guards :after-returns

     ///

     (fty::deffixequiv dimb-transunit-ensemble-loop
       :args ((gcc booleanp)))

     (defret filepath-transunit-map-unambp-of-dimb-transunit-ensemble-loop
       (implies (not erp)
                (filepath-transunit-map-unambp new-tumap))
       :hyp (filepath-transunit-mapp tumap)
       :hints (("Goal" :induct t)))))

  ///

  (defret transunit-ensemble-unambp-of-dimb-transunit-ensemble
    (implies (not erp)
             (transunit-ensemble-unambp new-tuens))))
