; FTY Library
;
; Copyright (C) 2020 Kestrel Institute (http://www.kestrel.edu)
;
; License: A 3-clause BSD license. See the LICENSE file distributed with ACL2.
;
; Author: Alessandro Coglio (coglio@kestrel.edu)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package "FTY")

(include-book "defbyte")

(include-book "ihs/basic-definitions" :dir :system)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defxdoc defbyte-ihs-theorems

  :parents (defbyte)

  :short (xdoc::topstring-p
          "Introduce some theorems about a @(tsee defbyte) instance
           and functions in the "
          (xdoc::seetopic "acl2::ihs" "IHS") " library.")

  :long

  (xdoc::topstring

   (xdoc::h3 "Introduction")

   (xdoc::p
    "Since @(tsee unsigned-byte-p) and @(tsee signed-byte-p)
     have certain formal relations with functions in IHS,
     similar relations can be proved for instances of @(tsee defbyte).")

   (xdoc::p
    "This macro automates this process,
     by including the IHS basic definitions
     and by generating theorems.
     For now this macro generates just one theorem;
     more will be added in the future as needed.")

   (xdoc::p
    "The reason for not having @(tsee defbyte) itself generate these theorems
     is to avoid always including IHS when using @(tsee defbyte).
     Having a separate macro leads to greater modularity.")

   (xdoc::h3 "General Form")

   (xdoc::codeblock
    "(defbyte-ihs-theorems type)")

   (xdoc::h3 "Inputs")

   (xdoc::desc
    "@('type')"
    (xdoc::p
     "A symbol that names a fixtype previously introduced via @(tsee defbyte).
      This specifies the @(tsee defbyte) instance
      for which the theorems are generated."))

   (xdoc::h3 "Generated Events")

   (xdoc::desc
    "@('bytep-of-loghead-of-size')"
    (xdoc::p
     "One of the following")
    (xdoc::ul
     (xdoc::li
      "If the @(tsee defbyte) instance is unsigned,
       a rewrite rule saying that
       @('(loghead size ...)'), where @('size') is the byte size,
       satisfies the recognizer of the bytes.")
     (xdoc::li
      "If the @(tsee defbyte) instance is signed,
       a rewrite rule saying that
       @('(logext size ...)'), where @('size') is the byte size,
       satisfies the recognizer of the bytes.")))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(acl2::defxdoc+ defbyte-ihs-theorems-implementation
  :parents (defbyte-ihs-theorems)
  :short "Implementation of @(tsee defbyte-ihs-theorems)."
  :order-subtopics t
  :default-parent t)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define defbyte-ihs-theorems-fn (type (wrld plist-worldp))
  :returns (event "A @(tsee acl2::pseudo-event-formp).")
  :mode :program
  :short "Event generated by @(tsee defbyte-ihs-theorems)."
  (b* (;; validate the TYPE input:
       ((unless (symbolp type))
        (raise "The TYPE input must be a symbol, ~
                but it is ~x0 instead." type))
       (defbyte-table (table-alist *defbyte-table-name* wrld))
       (defbyte-pair (assoc-eq type defbyte-table))
       ((unless defbyte-pair)
        (raise "The TYPE input ~x0 must name a type ~
                previously introduced via DEFBYTE, ~
                but this is not the case." type))
       ;; retrieve size and signedness from the DEFBYTE table:
       (defbyte-info (cdr defbyte-pair))
       (size (defbyte-info->size defbyte-info))
       (signed (defbyte-info->signed defbyte-info))
       ;; retrieve type recognizer from the fixtype table:
       (fty-table (get-fixtypes-alist wrld))
       (fty-info (find-fixtype type fty-table))
       (bytep (fixtype->pred fty-info))
       ;; choice between LOGHEAD and LOGEXT:
       (loghead/logext (if signed 'acl2::logext 'acl2::loghead))
       ;; name of the generated theorem:
       (bytep-of-loghead/logext-of-size
        (acl2::packn-pos (list bytep '-of- loghead/logext '-of- size) bytep))
       ;; variable in the generated theorem:
       (x (intern-in-package-of-symbol "X" bytep))
       ;; generated theorem:
       (event
        `(defrule ,bytep-of-loghead/logext-of-size
           (,bytep (,loghead/logext ,size ,x))
           :enable ,bytep
           :prep-books ((include-book "arithmetic-5/top" :dir :system)))))
    event))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defsection defbyte-ihs-theorems-macro-definition
  :short "Definition of the @(tsee defbyte-ihs-theorems) macro."
  :long "@(def defbyte-ihs-theorems)"
  (defmacro defbyte-ihs-theorems (type)
    `(make-event (defbyte-ihs-theorems-fn ',type (w state)))))
