/*
 * Decompiled with CFR 0.152.
 */
package htsjdk.samtools;

import htsjdk.samtools.AbstractBAMFileIndex;
import htsjdk.samtools.BAMIndexContent;
import htsjdk.samtools.BAMIndexMetaData;
import htsjdk.samtools.BAMIndexWriter;
import htsjdk.samtools.Bin;
import htsjdk.samtools.BinaryBAMIndexWriter;
import htsjdk.samtools.Chunk;
import htsjdk.samtools.GenomicIndexUtil;
import htsjdk.samtools.LinearIndex;
import htsjdk.samtools.SAMException;
import htsjdk.samtools.SAMFileHeader;
import htsjdk.samtools.SAMSequenceRecord;
import htsjdk.samtools.ValidationStringency;
import htsjdk.samtools.cram.build.ContainerParser;
import htsjdk.samtools.cram.build.CramIO;
import htsjdk.samtools.cram.structure.AlignmentSpan;
import htsjdk.samtools.cram.structure.Container;
import htsjdk.samtools.cram.structure.ContainerIO;
import htsjdk.samtools.cram.structure.CramHeader;
import htsjdk.samtools.cram.structure.Slice;
import htsjdk.samtools.seekablestream.SeekableStream;
import htsjdk.samtools.util.BlockCompressedFilePointerUtil;
import htsjdk.samtools.util.Log;
import htsjdk.samtools.util.ProgressLogger;
import htsjdk.samtools.util.RuntimeIOException;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.TreeSet;

public class CRAMBAIIndexer {
    private final int numReferences;
    private final BAMIndexWriter outputWriter;
    private int currentReference = 0;
    private final BAMIndexBuilder indexBuilder;

    public CRAMBAIIndexer(File output, SAMFileHeader fileHeader) {
        this.numReferences = fileHeader.getSequenceDictionary().size();
        this.indexBuilder = new BAMIndexBuilder(fileHeader);
        this.outputWriter = new BinaryBAMIndexWriter(this.numReferences, output);
    }

    public CRAMBAIIndexer(OutputStream output, SAMFileHeader fileHeader) {
        this.numReferences = fileHeader.getSequenceDictionary().size();
        this.indexBuilder = new BAMIndexBuilder(fileHeader);
        this.outputWriter = new BinaryBAMIndexWriter(this.numReferences, output);
    }

    public void processContainer(Container container, ValidationStringency validationStringency) {
        try {
            if (container == null || container.isEOF()) {
                return;
            }
            int sliceIndex = 0;
            for (Slice slice : container.slices) {
                slice.containerOffset = container.offset;
                slice.index = sliceIndex++;
                if (slice.isMultiref()) {
                    ContainerParser parser = new ContainerParser(this.indexBuilder.bamHeader);
                    Map<Integer, AlignmentSpan> refSet = parser.getReferences(container, validationStringency);
                    Slice fakeSlice = new Slice();
                    slice.containerOffset = container.offset;
                    slice.index = sliceIndex++;
                    AlignmentSpan unmappedSpan = refSet.remove(-1);
                    for (int refId : new TreeSet<Integer>(refSet.keySet())) {
                        AlignmentSpan span = refSet.get(refId);
                        fakeSlice.sequenceId = refId;
                        fakeSlice.containerOffset = slice.containerOffset;
                        fakeSlice.offset = slice.offset;
                        fakeSlice.index = slice.index;
                        fakeSlice.alignmentStart = span.getStart();
                        fakeSlice.alignmentSpan = span.getSpan();
                        fakeSlice.nofRecords = span.getCount();
                        this.processSingleReferenceSlice(fakeSlice);
                    }
                    if (unmappedSpan == null) continue;
                    AlignmentSpan span = unmappedSpan;
                    fakeSlice.sequenceId = -1;
                    fakeSlice.containerOffset = slice.containerOffset;
                    fakeSlice.offset = slice.offset;
                    fakeSlice.index = slice.index;
                    fakeSlice.alignmentStart = 0;
                    fakeSlice.alignmentSpan = 0;
                    fakeSlice.nofRecords = span.getCount();
                    this.processSingleReferenceSlice(fakeSlice);
                    continue;
                }
                this.processSingleReferenceSlice(slice);
            }
        }
        catch (IOException e) {
            throw new RuntimeIOException("Failed to read cram container", e);
        }
    }

    public void processSingleReferenceSlice(Slice slice) {
        try {
            int reference = slice.sequenceId;
            if (reference == -1) {
                return;
            }
            if (slice.sequenceId == -2) {
                throw new SAMException("Expecting a single reference slice.");
            }
            if (reference != this.currentReference) {
                this.advanceToReference(reference);
            }
            this.indexBuilder.processSingleReferenceSlice(slice);
        }
        catch (Exception e) {
            throw new SAMException("Exception creating BAM index for slice " + slice, e);
        }
    }

    public void finish() {
        this.advanceToReference(this.numReferences);
        this.outputWriter.writeNoCoordinateRecordCount(this.indexBuilder.getNoCoordinateRecordCount());
        this.outputWriter.close();
    }

    private void advanceToReference(int nextReference) {
        while (this.currentReference < nextReference) {
            BAMIndexContent content = this.indexBuilder.processReference(this.currentReference);
            this.outputWriter.writeReference(content);
            ++this.currentReference;
            this.indexBuilder.startNewReference();
        }
    }

    public static void createIndex(SeekableStream stream, File output, Log log, ValidationStringency validationStringency) throws IOException {
        CramHeader cramHeader = CramIO.readCramHeader(stream);
        if (cramHeader.getSamFileHeader().getSortOrder() != SAMFileHeader.SortOrder.coordinate) {
            throw new SAMException("Expecting a coordinate sorted file.");
        }
        CRAMBAIIndexer indexer = new CRAMBAIIndexer(output, cramHeader.getSamFileHeader());
        boolean totalRecords = false;
        Container container = null;
        ProgressLogger progressLogger = new ProgressLogger(log, 1, "indexed", "slices");
        do {
            try {
                String sequenceName;
                long offset = stream.position();
                container = ContainerIO.readContainer(cramHeader.getVersion(), (InputStream)stream);
                if (container == null || container.isEOF()) break;
                container.offset = offset;
                indexer.processContainer(container, validationStringency);
                if (null == log) continue;
                switch (container.sequenceId) {
                    case -1: {
                        sequenceName = "?";
                        break;
                    }
                    case -2: {
                        sequenceName = "???";
                        break;
                    }
                    default: {
                        sequenceName = cramHeader.getSamFileHeader().getSequence(container.sequenceId).getSequenceName();
                    }
                }
                progressLogger.record(sequenceName, container.alignmentStart);
            }
            catch (IOException e) {
                throw new RuntimeException("Failed to read cram container", e);
            }
        } while (!container.isEOF());
        indexer.finish();
    }

    private class BAMIndexBuilder {
        private final SAMFileHeader bamHeader;
        private Bin[] bins;
        private int binsSeen = 0;
        private final long[] index = new long[LinearIndex.MAX_LINEAR_INDEX_SIZE];
        private int largestIndexSeen = -1;
        private final BAMIndexMetaData indexStats = new BAMIndexMetaData();

        BAMIndexBuilder(SAMFileHeader header) {
            this.bamHeader = header;
        }

        private int computeIndexingBin(Slice slice) {
            int alignmentEnd = slice.alignmentStart + slice.alignmentSpan - 1;
            int alignmentStart = slice.alignmentStart - 1;
            if (alignmentEnd <= alignmentStart) {
                alignmentEnd = alignmentStart + 1;
            }
            return GenomicIndexUtil.regionToBin(alignmentStart, alignmentEnd);
        }

        private void processSingleReferenceSlice(Slice slice) {
            Bin bin;
            this.indexStats.recordMetaData(slice);
            int alignmentStart = slice.alignmentStart;
            if (alignmentStart == 0) {
                return;
            }
            int reference = slice.sequenceId;
            if (reference != CRAMBAIIndexer.this.currentReference) {
                throw new SAMException("Unexpected reference " + reference + " when constructing index for " + CRAMBAIIndexer.this.currentReference + " for record " + slice);
            }
            int binNum = this.computeIndexingBin(slice);
            if (this.bins == null) {
                SAMSequenceRecord seq = this.bamHeader.getSequence(reference);
                this.bins = seq == null ? new Bin[37451] : new Bin[AbstractBAMFileIndex.getMaxBinNumberForSequenceLength(seq.getSequenceLength()) + 1];
            }
            if (this.bins[binNum] != null) {
                bin = this.bins[binNum];
            } else {
                this.bins[binNum] = bin = new Bin(reference, binNum);
                ++this.binsSeen;
            }
            long chunkStart = slice.containerOffset << 16 | (long)slice.index;
            long chunkEnd = (slice.containerOffset << 16 | (long)slice.index) + 1L;
            Chunk newChunk = new Chunk(chunkStart, chunkEnd);
            List<Chunk> oldChunks = bin.getChunkList();
            if (!bin.containsChunks()) {
                bin.addInitialChunk(newChunk);
            } else {
                Chunk lastChunk = bin.getLastChunk();
                if (BlockCompressedFilePointerUtil.areInSameOrAdjacentBlocks(lastChunk.getChunkEnd(), chunkStart)) {
                    lastChunk.setChunkEnd(chunkEnd);
                } else {
                    oldChunks.add(newChunk);
                    bin.setLastChunk(newChunk);
                }
            }
            int alignmentEnd = slice.alignmentStart + slice.alignmentSpan;
            int startWindow = LinearIndex.convertToLinearIndexOffset(alignmentStart);
            int endWindow = alignmentEnd == 0 ? (startWindow = LinearIndex.convertToLinearIndexOffset(alignmentStart - 1)) : LinearIndex.convertToLinearIndexOffset(alignmentEnd);
            if (endWindow > this.largestIndexSeen) {
                this.largestIndexSeen = endWindow;
            }
            for (int win = startWindow; win <= endWindow; ++win) {
                if (this.index[win] != 0L && chunkStart >= this.index[win]) continue;
                this.index[win] = chunkStart;
            }
        }

        public BAMIndexContent processReference(int reference) {
            if (reference != CRAMBAIIndexer.this.currentReference) {
                throw new SAMException("Unexpected reference " + reference + " when constructing index for " + CRAMBAIIndexer.this.currentReference);
            }
            if (this.binsSeen == 0) {
                return null;
            }
            long[] newIndex = new long[this.largestIndexSeen + 1];
            long lastNonZeroOffset = 0L;
            for (int i = 0; i <= this.largestIndexSeen; ++i) {
                if (this.index[i] == 0L) {
                    this.index[i] = lastNonZeroOffset;
                } else {
                    lastNonZeroOffset = this.index[i];
                }
                newIndex[i] = this.index[i];
            }
            LinearIndex linearIndex = new LinearIndex(reference, 0, newIndex);
            return new BAMIndexContent(reference, this.bins, this.binsSeen, this.indexStats, linearIndex);
        }

        public long getNoCoordinateRecordCount() {
            return this.indexStats.getNoCoordinateRecordCount();
        }

        void startNewReference() {
            this.bins = null;
            if (this.binsSeen > 0) {
                Arrays.fill(this.index, 0L);
            }
            this.binsSeen = 0;
            this.largestIndexSeen = -1;
            this.indexStats.newReference();
        }
    }
}

