/*************************************************************************
 *
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * Copyright 2009 by Sun Microsystems, Inc.
 *
 * OpenOffice.org - a multi-platform office productivity suite
 *
 * $RCSfile: GADEPSOSolverImpl.java,v $
 * $Revision: 1.1 $
 *
 * This file is part of OpenOffice.org.
 *
 * OpenOffice.org is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * OpenOffice.org is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with OpenOffice.org.  If not, see
 * <http://www.openoffice.org/license.html>
 * for a copy of the LGPLv3 License.
 *
 ************************************************************************/

package com.sun.star.NLPSolver;

import com.sun.star.NLPSolver.dialogs.IEvolutionarySolverStatusDialog;
import com.sun.star.uno.XComponentContext;
import com.sun.star.lib.uno.helper.Factory;
import com.sun.star.lang.IllegalArgumentException;
import com.sun.star.lang.XSingleComponentFactory;
import com.sun.star.registry.XRegistryKey;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Random;
import java.util.logging.Level;
import java.util.logging.Logger;
import net.adaptivebox.deps.DEPSAgent;
import net.adaptivebox.deps.behavior.DEGTBehavior;
import net.adaptivebox.deps.behavior.PSGTBehavior;
import net.adaptivebox.global.IUpdateCycleEngine;
import net.adaptivebox.knowledge.ILibEngine;
import net.adaptivebox.knowledge.Library;
import net.adaptivebox.knowledge.SearchPoint;


public final class GADEPSOSolverImpl extends BaseEvolutionarySolver
   implements com.sun.star.lang.XServiceInfo
{
    private static final String m_implementationName = GADEPSOSolverImpl.class.getName();
    private static final String[] m_serviceNames = {
        "com.sun.star.sheet.Solver",
        "com.sun.star.beans.PropertySet"
    };

    public GADEPSOSolverImpl( XComponentContext context )
    {
        super(context, "GADEPSO Nonlinear Problem Solver");

        registerProperty(m_agentSwitchRate);
        registerProperty(m_factor);
        registerProperty(m_CR);
        registerProperty(m_c1);
        registerProperty(m_c2);
        registerProperty(m_weight);
        registerProperty(m_CL);
    }

    public static XSingleComponentFactory __getComponentFactory( String sImplementationName ) {
        XSingleComponentFactory xFactory = null;

        if ( sImplementationName.equals( m_implementationName ) )
            xFactory = Factory.createComponentFactory(GADEPSOSolverImpl.class, m_serviceNames);
        return xFactory;
    }

    public static boolean __writeRegistryServiceInfo( XRegistryKey xRegistryKey ) {
        return Factory.writeRegistryServiceInfo(m_implementationName,
                                                m_serviceNames,
                                                xRegistryKey);
    }

    // com.sun.star.lang.XServiceInfo:
    public String getImplementationName() {
         return m_implementationName;
    }

    public boolean supportsService( String sService ) {
        int len = m_serviceNames.length;

        for( int i=0; i < len; i++) {
            if (sService.equals(m_serviceNames[i]))
                return true;
        }
        return false;
    }

    public String[] getSupportedServiceNames() {
        return m_serviceNames;
    }

    protected Random m_random = new Random();

    protected class DecisionVariable {
        protected CellMap CellMap;
        protected int OriginalVariable;
        protected int MinValue;
        protected int MaxValue;

        protected int getRandomValue() {
            return m_random.nextInt(MaxValue - MinValue) + MinValue;
        }
    }

    protected class DecisionMaker implements Comparable<DecisionMaker> {
        private ArrayList<DecisionVariable> m_decisionVariables;
        protected double[] Decision;
        protected SearchPoint BestPoint;

        protected DecisionMaker(ArrayList<DecisionVariable> decisionVariables) {
            m_decisionVariables = decisionVariables;
            Decision = new double[decisionVariables.size()];
            BestPoint = m_problemEncoder.getFreshSearchPoint();
        }

        protected void applyDecision() {
            for (int i = 0; i < Decision.length; i++) {
                CellMap cellMap = m_decisionVariables.get(i).CellMap;
                m_variableData[cellMap.Range][cellMap.Row][cellMap.Col] = Decision[i];
            }
        }

        protected void mutate() {
            for (int i = 0; i < Decision.length; i++)
                if (Math.random() < 0.1) //TODO : make property
                    Decision[i] = m_decisionVariables.get(i).getRandomValue();
        }

        protected DecisionMaker crossOver(DecisionMaker partner) {
            DecisionMaker result = new DecisionMaker(m_decisionVariables);
            for (int i = 0; i < Decision.length; i++)
                result.Decision[i] = (Math.random() < 0.5) ? Decision[i] : partner.Decision[i];
            return result;
        }

        public int compareTo(DecisionMaker o) {
            return m_specCompareEngine.compare(this.BestPoint.getEncodeInfo(), o.BestPoint.getEncodeInfo());
        }
    }

    // com.sun.star.sheet.XSolver:
    private DEPSAgent[] m_agents;
    private DecisionMaker[] m_decisionMakers;

    protected ArrayList<DecisionVariable> m_decisionVariables = new ArrayList<DecisionVariable>();

    private PropertyInfo<Double> m_agentSwitchRate = new PropertyInfo<Double>("AgentSwitchRate", 0.5, "Agent Switch Rate (DE Probability)");
    // --DE
    private PropertyInfo<Double> m_factor = new PropertyInfo<Double>("DE.Factor", 0.5, "DE: Scaling Factor (0-1.2)");
    private PropertyInfo<Double> m_CR = new PropertyInfo<Double>("DE.CR", 0.9, "DE: Crossover Probability (0-1)");
    // --PS
    private PropertyInfo<Double> m_c1 = new PropertyInfo<Double>("PS.C1", 1.494, "PS: Cognitive Constant");
    private PropertyInfo<Double> m_c2 = new PropertyInfo<Double>("PS.C2", 1.494, "PS: Social Constant");
    private PropertyInfo<Double> m_weight = new PropertyInfo<Double>("PS.Weight", 0.729, "PS: Constriction Coefficient");
    private PropertyInfo<Double> m_CL = new PropertyInfo<Double>("PS.CL", 0.0, "PS: CL (0-0.005)");


    @Override
    protected void prepareVariables(double[][] variableBounds) {
        m_variables.clear();
        m_decisionVariables.clear();
        for (int i = 0; i < m_variableCount; i++) {
            if (variableBounds[i][0] == 0.0 &&
                    variableBounds[i][1] == 1.0 &&
                    variableBounds[i][2] == 1.0) {
                DecisionVariable var = new DecisionVariable();
                var.CellMap = m_variableMap[i];
                var.MinValue = 0;
                var.MaxValue = 1;
                m_decisionVariables.add(var);
            } else {
                Variable var = new Variable(m_variableMap[i], i);
                var.MinValue = variableBounds[i][0];
                var.MaxValue = variableBounds[i][1];
                var.Granularity = variableBounds[i][2];
                m_variables.add(var);
            }
        }
    }
    
    public void solve() {
        try {
            m_librarySize.setValue(m_swarmSize.getValue()); //DEPS' library is as large as the swarm
        } catch (IllegalArgumentException ex) {
            Logger.getLogger(DEPSSolverImpl.class.getName()).log(Level.SEVERE, null, ex);
        }

        try {
        initializeSolve();

        //Init:
        m_agents = new DEPSAgent[m_swarmSize.getValue()];
        for (int i = 0; i < m_swarmSize.getValue(); i++) {
            m_agents[i] = new DEPSAgent();
            m_agents[i].setProblemEncoder(m_problemEncoder);
            m_agents[i].setPbest(m_library.getSelectedPoint(i));

            DEGTBehavior deGTBehavior = new DEGTBehavior();
            deGTBehavior.FACTOR = m_factor.getValue();
            deGTBehavior.CR = m_CR.getValue();

            PSGTBehavior psGTBehavior = new PSGTBehavior();
            psGTBehavior.c1 = m_c1.getValue();
            psGTBehavior.c2 = m_c2.getValue();
            psGTBehavior.CL = m_CL.getValue();
            psGTBehavior.weight = m_weight.getValue();

            m_agents[i].switchP = m_agentSwitchRate.getValue();
            m_agents[i].setGTBehavior(deGTBehavior);
            m_agents[i].setGTBehavior(psGTBehavior);

            m_agents[i].setSpecComparator(m_specCompareEngine);
            if (m_agents[i] instanceof ILibEngine)
                ((ILibEngine)m_agents[i]).setLibrary(m_library);
        }

        int decisionMakerCount = 10; //TODO : make property
        m_decisionMakers = new DecisionMaker[decisionMakerCount];
        m_decisionMakers[0] = new DecisionMaker(m_decisionVariables);
        if (m_useRandomStartingPoint.getValue()) {
            for (int i = 0; i < m_decisionVariables.size(); i++)
                m_decisionMakers[0].Decision[i] = m_decisionVariables.get(i).getRandomValue();
        } else {
            for (int i = 0; i < m_decisionVariables.size(); i++) {
                CellMap cellMap = m_decisionVariables.get(i).CellMap;
                m_decisionMakers[0].Decision[i] = m_variableData[cellMap.Range][cellMap.Row][cellMap.Col];
            }
        }

        for (int i = 1; i < decisionMakerCount; i++) {
            m_decisionMakers[i] = new DecisionMaker(m_decisionVariables);
            for (int j = 0; j < m_decisionVariables.size(); j++)
                m_decisionMakers[i].Decision[j] = m_decisionVariables.get(j).getRandomValue();
        }

        //Learn
        m_solverStatusDialog.setVisible(true);
        m_solverStatusDialog.setMaxIterations(m_learningCycles.getValue());
        m_solverStatusDialog.setMaxStagnation(m_required.getValue());
        do {
            for (int gaCycle = 1; gaCycle <= m_learningCycles.getValue() &&
                    m_solverStatusDialog.getUserState() != IEvolutionarySolverStatusDialog.CANCEL; gaCycle++) {
                for (int di = 0; di < decisionMakerCount; di++) {
                    m_decisionMakers[di].applyDecision();

                    m_toleratedCount = 0;
                    m_toleratedMin = -1.0 * m_tolerance.getValue();
                    m_toleratedMax = m_tolerance.getValue();
                    for (int learningCycle = 1; learningCycle <= m_learningCycles.getValue() &&
                            m_toleratedCount < m_required.getValue() &&
                            m_solverStatusDialog.getUserState() != IEvolutionarySolverStatusDialog.CANCEL; learningCycle++) {
                        m_library.refreshGbest(m_specCompareEngine);

                        for (int i = 0; i < m_swarmSize.getValue(); i++)
                            m_agents[i].generatePoint();

                        for (int i = 0; i < m_swarmSize.getValue(); i++)
                            m_agents[i].learn();

                        for (int i = 0; i < m_swarmSize.getValue(); i++) {
                            SearchPoint agentPoint = m_agents[i].getMGState();
                            boolean inRange = (agentPoint.getObjectiveValue() >= m_toleratedMin && agentPoint.getObjectiveValue() <= m_toleratedMax);
                            if (Library.replace(m_envCompareEngine, agentPoint, m_decisionMakers[di].BestPoint) && !inRange) {
                                m_toleratedMin = agentPoint.getObjectiveValue() - m_tolerance.getValue();
                                m_toleratedMax = agentPoint.getObjectiveValue() + m_tolerance.getValue();
                                m_toleratedCount = 0;
                            }
                            if (Library.replace(m_envCompareEngine, agentPoint, m_totalBestPoint))
                                m_solverStatusDialog.setBestSolution(m_totalBestPoint.getObjectiveValue(), m_totalBestPoint.isFeasible());
                        }

                        if (m_specCompareEngine instanceof IUpdateCycleEngine)
                            ((IUpdateCycleEngine)m_specCompareEngine).updateCycle(learningCycle);

                        m_solverStatusDialog.setIteration(learningCycle);
                        m_solverStatusDialog.setStagnation(m_toleratedCount);
                    }
                }

                Arrays.sort(m_decisionMakers);
            }
        } while (m_solverStatusDialog.waitForUser() == IEvolutionarySolverStatusDialog.CONTINUE);

        finalizeSolve();
        } catch (Exception ex) {
            Logger.getLogger(DEPSSolverImpl.class.getName()).log(Level.SEVERE, null, ex);
        }
    }

    @Override
    protected void applySolution() {
        DecisionMaker decisionMaker = m_decisionMakers[0];
        for (int i = 0; i < m_decisionVariables.size(); i++) {
            int var = m_decisionVariables.get(i).OriginalVariable;
            m_variableCells[var].setValue(decisionMaker.Decision[i]);
            m_currentParameters[var] = decisionMaker.Decision[i];
        }

        double[] location = decisionMaker.BestPoint.getLocation();

        //make sure, the "Integer" variable type is met
        m_problemEncoder.getDesignSpace().getMappingPoint(location);

        //get the function value for our optimal point
        for (int i = 0; i < m_variables.size(); i++) {
            int var = m_variables.get(i).OriginalVariable;
            m_variableCells[var].setValue(location[i]);
            m_currentParameters[var] = location[i];
        }
        m_functionValue = m_objectiveCell.getValue();
    }

}
