/* Copyright (C) 2001, 2007 United States Government as represented by
   the Administrator of the National Aeronautics and Space Administration.
   All Rights Reserved.
 */
package gov.nasa.worldwind.servers.wms.generators;

import gov.nasa.worldwind.geom.Angle;
import gov.nasa.worldwind.servers.wms.*;
import gov.nasa.worldwind.servers.wms.xml.*;
import gov.nasa.worldwind.servers.wms.formats.ImageFormatter;
import gov.nasa.worldwind.servers.wms.formats.BufferedImageFormatter;
import gov.nasa.worldwind.servers.wms.utilities.WaveletCodec;
import gov.nasa.worldwind.geom.Sector;

import javax.imageio.ImageIO;
import javax.xml.bind.JAXBContext;
import javax.xml.bind.Marshaller;
import java.io.IOException;
import java.io.File;
import java.io.FilenameFilter;
import java.io.ByteArrayOutputStream;
import java.util.Properties;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.awt.image.BufferedImage;
import java.awt.image.WritableRaster;
import java.awt.image.DataBuffer;
import java.awt.*;
import java.lang.Exception;

/**
 * A MapGenerator implementation that serves a collection of Landsat imagery. The imagery is
 * presumed to be broken up into 2x2 degree tiles, with a naming scheme of dd{ns}ddd{ew}.tif".
 * <p>
 * The GDAL utility <code>gdalwarp</code> is used to extract subregions from the tiles.
 * </p>
 * <p>The implementation also attempts to use wavelet encodings of these files to reconstruct
 * small-scale representations of the files. These encodings reside in files named after the
 * individual tiles with a ".wvt" suffix appended. The encodings are presumed to be co-located with 
 * the tiles, unless otherwise specified with an optional configuration property (see below).</p>
 *
 * <p>Several optional properties may be included in the XML configuration of the corresponding
 * {@link gov.nasa.worldwind.servers.wms.MapSource} element:
 *
 * <pre>
 *   &lt;!-- if a tile's footprint in a map request is below this size (in pixels),
 *        the image is reconstructed from a wavelet encoding --&gt;
 *   &lt;property name="wavelet_image_threshold" value="..." /&gt;
 *
 *   &lt;!-- amount of wavelet encodings to preload ( size in pixels, sq.), --&gt;
 *   &lt;property name="wavelet_preload_size" value="..." /&gt;
 *
 *   &lt;!-- root directory where the wavelet encodings reside; the encodings are
 *        otherwise presumed to be co-located with the image tiles. --&gt;
 *   &lt;property name="wavelet_encoding_root_dir" value="..." /&gt;
 * 
 *   &lt;!-- A color value (integer in range 0 -- 255), whereby a pixel is considered
 *        to be transparent of all three of its color components are below this value --&gt;
 *   &lt;property name="background_color_threshold" value="..." /&gt;
 * </pre>
 * 
 * @author brownrigg
 * @version $Id$
 */

public class EsatGenerator implements MapGenerator {


    public ServiceInstance getServiceInstance() { return new EsatServiceInstance(); }

    public boolean initialize(MapSource mapSource) throws IOException, WMSServiceException {
        boolean success = true;  // Assume the best...

        try {
            this.mapSource = mapSource;
            String rootDir = mapSource.getRootDir();

            this.gdalPath = WMSServer.getConfiguration().getGDALPath();
            if (this.gdalPath == null)
                this.gdalPath = "";
            else if (!this.gdalPath.endsWith(File.separator))
                this.gdalPath += File.separator;   // make this well-formed for convenience later...

            // Get these optional properties...
            Properties props = mapSource.getProperties();
            String srcName = mapSource.getName();
            this.smallImageSize = parseIntProperty(props, WAVELET_IMAGE_THRESHOLD, smallImageSize, srcName);
            int tmp = parseIntProperty(props, WAVELET_PRELOAD_SIZE, preloadRes, srcName);
            if (!WaveletCodec.isPowerOfTwo(tmp)) {
                SysLog.inst().warning(srcName + ": value given for \"" + WAVELET_PRELOAD_SIZE + "\" must be power of two");
                SysLog.inst().warning("  given: " + tmp + ",  overriding with default of: " + this.preloadRes);
            }
            else
                this.preloadRes = tmp;

            this.encodingRootDir = props.getProperty(WAVELET_ROOT_DIR);
            if (this.encodingRootDir == null)
                this.encodingRootDir = rootDir;

            this.blackThreshold = parseIntProperty(props, BACKGROUND_COLOR_THRESHOLD, blackThreshold, srcName);

            // Get all the files from rootDir that appear to be esat tiles...
            Pattern regex = Pattern.compile("(\\d\\d)([ns])(\\d\\d\\d)([ew])\\x2etif",
                Pattern.CASE_INSENSITIVE);
            File rootDirFile = new File(rootDir);
            File[] tiles = rootDirFile.listFiles(new EsatTilenameFilter(regex));

            tileIndex = new EsatTile[180][90];

            for (File tile : tiles) {
                try {
                    String filename = tile.getName();
                    Matcher parser = regex.matcher(filename);
                    if (!parser.matches())
                        throw new IllegalArgumentException("ESAT-tilename not according to expectations: " + filename);
                    double lat = Double.parseDouble(parser.group(1));
                    if (parser.group(2).equalsIgnoreCase("s")) lat = -lat;
                    double lon = Double.parseDouble(parser.group(3));
                    if (parser.group(4).equalsIgnoreCase("w")) lon = -lon;
                    // tiles are named after their *eastern* most edge, but we want indexing based on
                    // a simple right-handed coordinate system, with indices increase to the east&north.
                    // Hence we translate the incoming longitude by the tile-size.
                    int ix = lonToIndex(lon-TILE_SIZE);
                    int iy = latToIndex(lat);
                    tileIndex[ix][iy] = new EsatTile();
                    tileIndex[ix][iy].file = tile;
                    File codec = new File(this.encodingRootDir + File.separator + tile.getName() + WaveletCodec.WVT_EXT);
                    tileIndex[ix][iy].codec =  WaveletCodec.loadPartially(codec, this.preloadRes);
                } catch (Exception ex) {
                    SysLog.inst().warning("Error preloading Esat Wavelet: " + ex.toString());
                }
            }
        }

        catch (Exception ex) {
            success = false;
            SysLog.inst().stackTrace(ex);
        }

        return success;
    }
 
    public Sector getBBox() {
        return new Sector(
            Angle.fromDegreesLatitude(MIN_LAT), 
            Angle.fromDegreesLatitude(MAX_LAT),
            Angle.fromDegreesLongitude(MIN_LON),
            Angle.fromDegreesLongitude(MAX_LON));

    }    
    
    public String[] getCRS() {
        return new String[] {crsStr};
    }
    
    //
    // Convert lat-lon to indices into tileIndex array. Our convention here is
    // that -180,-90 is [0][0], with indices increasing to the east and north.
    //
    private static int lonToIndex(double lon) {
        return (int)((lon-MIN_LON) / TILE_SIZE);
    }

    private static int latToIndex(double lat) {
        return (int)((lat-MIN_LAT) / TILE_SIZE);
    }

    //
    // Convenience method for parsing/validating optional configuration parameters.
    //
    private int parseIntProperty(Properties props, String key, int defaultVal, String name) {
        int retval = defaultVal;
        String tmp = props.getProperty(key);
        if (tmp != null) {
            try {
               retval = Integer.parseInt(tmp);
            }  catch (NumberFormatException ex) {
                SysLog.inst().warning("Could not decode '" + key +
                    "' property in config. for " + name);
            }
        }

        return retval;
    }

    // --------------------------------------------
    // class EsatServiceInstance
    //
    // Used to manage per-request state.
    //
    public class EsatServiceInstance implements ServiceInstance {

        public ImageFormatter serviceRequest(WMSGetMapRequest req) throws IOException, WMSServiceException {
            this.threadId = Thread.currentThread().getId();

            try {
                // determine the tiles overlapped by the request...
                int iMinX = Math.max(0, Math.min(lonToIndex(req.getBBoxXMin()), tileIndex.length-1));
                int iMaxX = Math.max(0, Math.min(lonToIndex(req.getBBoxXMax()), tileIndex.length-1));
                int iMinY = Math.max(0, Math.min(latToIndex(req.getBBoxYMin()), tileIndex[0].length-1));
                int iMaxY = Math.max(0, Math.min(latToIndex(req.getBBoxYMax()), tileIndex[0].length-1));
                Sector reqSector = Sector.fromDegrees(req.getBBoxYMin(), req.getBBoxYMax(), req.getBBoxXMin(),
                        req.getBBoxXMax());

                // the image to be created...
                BufferedImage reqImage = new BufferedImage(req.getWidth(), req.getHeight(), BufferedImage.TYPE_4BYTE_ABGR);
                Graphics2D g2d = (Graphics2D) reqImage.getGraphics();

                int debugMinFrameRes = Integer.MAX_VALUE;
                int debugMaxFrameRes = -Integer.MAX_VALUE;

                for (int ix = iMinX; ix <= iMaxX; ix++) {
                    for (int iy = iMinY; iy <= iMaxY; iy++) {
                        // tiles over the ocean don't exist...
                        if (tileIndex[ix][iy] == null) continue;

                        Sector tileSector = Sector.fromDegrees(MIN_LAT + iy*TILE_SIZE, MIN_LAT + (iy+1)*TILE_SIZE,
                                MIN_LON + ix*TILE_SIZE, MIN_LON + (ix+1)*TILE_SIZE);
                        Sector overlap = reqSector.intersection(tileSector);
                        if (overlap == null) continue;

                        // find size of the tile's footprint at the requested image resolution...
                        int footprintX = (int) (TILE_SIZE * reqImage.getWidth() / reqSector.getDeltaLonDegrees());
                        int footprintY = (int) (TILE_SIZE * reqImage.getHeight() / reqSector.getDeltaLatDegrees());

                        // Destination subimage...
                        int dx1 = (int) ((overlap.getMinLongitude().degrees - reqSector.getMinLongitude().degrees)
                                * reqImage.getWidth() / reqSector.getDeltaLonDegrees());
                        int dx2 = (int) ((overlap.getMaxLongitude().degrees - reqSector.getMinLongitude().degrees)
                                * reqImage.getWidth() / reqSector.getDeltaLonDegrees());
                        int dy1 = (int) ((reqSector.getMaxLatitude().degrees - overlap.getMaxLatitude().degrees)
                                * reqImage.getHeight() / reqSector.getDeltaLatDegrees());
                        int dy2 = (int) ((reqSector.getMaxLatitude().degrees - overlap.getMinLatitude().degrees)
                                * reqImage.getHeight() / reqSector.getDeltaLatDegrees());


                        // Depending upon footprint, either get image from the tile, or reconstruct
                        // it from a wavelet encoding.
                        BufferedImage sourceImage;
                        int sx1, sx2, sy1, sy2;
                        if (footprintX > smallImageSize || footprintY > smallImageSize) {
                            sourceImage = getImageFromSource(tileIndex[ix][iy], overlap, (dx2-dx1), (dy2-dy1));
                            if (sourceImage == null) continue;
                            sx1 = sy1 = 0;
                            sx2 = sourceImage.getWidth();
                            sy2 = sourceImage.getHeight();
                        } else {
                            int maxRes = footprintX;
                            maxRes = (footprintY > maxRes) ? footprintY : maxRes;
                            int power = (int) Math.ceil(Math.log(maxRes) / Math.log(2.));
                            int res = (int) Math.pow(2., power);
                            sourceImage = getImageFromWaveletEncoding(tileIndex[ix][iy], res);
                            if (sourceImage == null) continue;

                            // find overlap coordinates in source image...
                            sx1 = (int) ((overlap.getMinLongitude().degrees - tileSector.getMinLongitude().degrees)
                                    * sourceImage.getWidth() / tileSector.getDeltaLonDegrees());
                            sx2 = (int) ((overlap.getMaxLongitude().degrees - tileSector.getMinLongitude().degrees)
                                    * sourceImage.getWidth() / tileSector.getDeltaLonDegrees());
                            sx1 = Math.max(0, sx1);
                            sx2 = Math.min(sourceImage.getWidth() - 1, sx2);

                            sy1 = (int) ((tileSector.getMaxLatitude().degrees - overlap.getMaxLatitude().degrees)
                                    * sourceImage.getHeight() / tileSector.getDeltaLatDegrees());
                            sy2 = (int) ((tileSector.getMaxLatitude().degrees - overlap.getMinLatitude().degrees)
                                    * sourceImage.getHeight() / tileSector.getDeltaLatDegrees());
                            sy1 = Math.max(0, sy1);
                            sy2 = Math.min(sourceImage.getHeight() - 1, sy2);

                            // debugging and performance analysis info...
                            if (res < debugMinFrameRes) debugMinFrameRes = res;
                            if (res > debugMaxFrameRes) debugMaxFrameRes = res;

                        }

                        g2d.drawImage(sourceImage, dx1, dy1, dx2, dy2, sx1, sy1, sx2, sy2, null);
                        /***
                        g2d.drawImage(sourceImage, dx1, dy1, dx2, dy2, 0, 0, sourceImage.getWidth(),
                                sourceImage.getHeight(), null);
                        ***/
                    }
                }
                SysLog.inst().info("   " + (iMaxX-iMinX+1)*(iMaxY-iMinY+1) + " tiles in request" +
                        "min/max recon. sizes: " + debugMinFrameRes + ", " + debugMaxFrameRes);

                setBlackToTransparent(reqImage);
                return new BufferedImageFormatter(reqImage);

            } catch (Exception ex) {
                SysLog.inst().stackTrace(ex);
                throw new WMSServiceException("ESat request failed: " + ex.getMessage());
            }
        }

        // Not implemented at present.
        public java.util.List<File> serviceRequest(WMSGetImageryListRequest req) throws IOException, WMSServiceException {
            return null;
        }

        //
        // Attempts to return the specified image as a BufferedImage. Returns null on failure.
        //
        private BufferedImage getImageFromSource(EsatTile tile, Sector extent, int xres, int yres) {
            BufferedImage sourceImage = null;
            File tmpFile = new File(getTempFilename());
            try {
                StringBuilder cmd = new StringBuilder();
                cmd.append(gdalPath);
                
              cmd.append("gdalwarp").
                        append(" -of Gtiff").append(" -te ").
                        append(extent.getMinLongitude()).append(" ").
                        append(extent.getMinLatitude()).append(" ").
                        append(extent.getMaxLongitude()).append(" ").
                        append(extent.getMaxLatitude()).append(" ");
                cmd.append(" -ts ").
                        append(xres).append(" ").
                        append(yres).append(" ");                
                cmd.append(" -rc ");  // cubic filtering
                cmd.append(tile.file.getAbsolutePath()).append(" ").
                        append(tmpFile.getAbsolutePath());
                
                /*****

                cmd.append("gdal_translate -not_strict").
                        append(" -of png").append(" -projwin ").
                        append(extent.getMinLongitude()).append(" ").
                        append(extent.getMaxLatitude()).append(" ").
                        append(extent.getMaxLongitude()).append(" ").
                        append(extent.getMinLatitude()).append(" ");
                cmd.append(" -outsize ").
                        append(xres).append(" ").
                        append(yres).append(" ");
                cmd.append(tile.file.getAbsolutePath()).append(" ").
                        append(tmpFile.getAbsolutePath());
                ******/
                
                SysLog.inst().info("Thread " + threadId + ", exec'ing: " + cmd.toString());

                // Call upon a GDAL utility to make our map...
                Process proc = null;
                try {
                    proc = Runtime.getRuntime().exec(cmd.toString());
                    proc.waitFor();
                    if (proc.exitValue() != 0) {
                        // fetch enough of the stdout/stderr to give a hint...
                        byte[] buff = new byte[1024];
                        int len = proc.getInputStream().read(buff);
                        if (len <= 0)   // try stderr?
                            len = proc.getErrorStream().read(buff);
                        throw new WMSServiceException("Thread " + threadId + ", failed to execute GDAL: " +
                                ((len > 0) ? new String(buff, 0, len) : "**unknown**"));
                    }
                    sourceImage = ImageIO.read(tmpFile);
                } catch (InterruptedException ex) {
                    throw new WMSServiceException(ex.toString());
                } finally {
                    try {
                        proc.getInputStream().close();
                        proc.getErrorStream().close();
                        proc.getOutputStream().close();
                        tmpFile.delete();
                    } catch (Exception ex) {/* best effort */}
                }

            } catch (Exception ex) {/* best effort */}

            return sourceImage;
        }


        //
        // Attempts to reconstruct the given FrameFile as a BufferedImage from a WaveletEncoding.
        // Returns null if encoding does not exist or on any other failure.
        //
        private BufferedImage getImageFromWaveletEncoding(EsatTile tile, int resolution) {
            BufferedImage sourceImage = null;
            try {

                if (resolution <= 0 || tile.codec == null)
                    return sourceImage;

                WaveletCodec codec = null;
                if (resolution <= EsatGenerator.this.preloadRes)
                    codec = tile.codec;
                else
                    // read wavelet file...
                    codec = WaveletCodec.loadPartially(new File(EsatGenerator.this.encodingRootDir +
                            File.separator + tile.file.getName() + WaveletCodec.WVT_EXT),
                            resolution);


                if (codec != null)
                    sourceImage = codec.reconstruct(resolution);

            } catch (Exception ex) {
                SysLog.inst().stackTrace(ex);
                SysLog.inst().warning("Failed to reconstruct wavelet from " + tile.file.getName() + ": " + ex.toString());
            }

            return sourceImage;
        }

        //
        // Sets "black" pixels in the image to be transparent. This is done to for edge-blending with
        // underlying imagery.  As the wavelet-reconstructions are not exact, the color black is
        // defined here by a threshold value.
        //
        private void setBlackToTransparent(BufferedImage image) {
            WritableRaster raster = image.getRaster();
            int[] pixel = new int[4];
            int width = image.getWidth();
            int height = image.getHeight();
            for (int j = 0; j < height; j++) {
                for (int i = 0; i < width; i++) {
                    // We know, by the nature of this source, that we are dealing with RGBA rasters...
                    raster.getPixel(i, j, pixel);
                    if (pixel[0] <= EsatGenerator.this.blackThreshold &&
                            pixel[1] <= EsatGenerator.this.blackThreshold &&
                            pixel[2] <= EsatGenerator.this.blackThreshold)
                    {
                        pixel[3] = 0;
                        raster.setPixel(i, j, pixel);
                    }
                }
            }
        }

        private String getTempFilename() {
            StringBuilder tmp = new StringBuilder();
            tmp.append(WMSServer.getConfiguration().getWorkDirectory()).
                    append(File.separator).append("WMStmp").
                    append(Integer.toString((int) (Math.random() * 100000.)));
            return tmp.toString();
        }


        public void freeResources() { /* No-op */ }

        private long threadId;
    }

    // ----------------------------------------------------
    // class EsatTile
    //
    // A bundle of info we keep track of for each tile.
    //
    private static class EsatTile {
        File         file;
        WaveletCodec codec;
    }

    // class EsatTilenameFilter
    //
    // A class used to filter a list of filenames, favoring those that conform to the
    // Esat-tile filenaming convention.
    //
    private class EsatTilenameFilter implements FilenameFilter {

        public EsatTilenameFilter(Pattern regex) {
            super();
            this.regex = regex;
        }

        public boolean accept(File dir, String name) {
            Matcher matcher = regex.matcher(name);
            return matcher.matches();
        }

        private Pattern regex;
    }

    private MapSource mapSource = null;
    private String encodingRootDir = null;
    private EsatTile[][] tileIndex;
    private String gdalPath = null;

    private static final double MIN_LON = -180.;
    private static final double MAX_LON = 180.;
    private static final double MIN_LAT = -90.;
    private static final double MAX_LAT = 90.;
    private static final int TILE_SIZE = 2;  // 2 degree square tiles.
    private static final String crsStr="EPSG:4326";

    // performance tuning parameters...
    private int smallImageSize = 512;
    private int preloadRes = 32;
    private int blackThreshold = 3;

    // Configuration property keys...
    private static final String WAVELET_IMAGE_THRESHOLD = "wavelet_image_threshold";
    private static final String WAVELET_PRELOAD_SIZE = "wavelet_preload_size";
    private static final String WAVELET_ROOT_DIR = "wavelet_encoding_root_dir";
    private static final String GDAL_PATH = "gdalPath";
    private static final String BACKGROUND_COLOR_THRESHOLD = "background_color_threshold";
}
