/* XRACER (C) 1999-2000 Richard W.M. Jones <rich@annexia.org> and other AUTHORS
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * $Id: lcd.c,v 1.3 2000/03/19 23:48:47 rich Exp $
 */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>

#include <GL/glu.h>

#include "xracer.h"
#include "xracer-lcd.h"
#include "xracer-log.h"

/* Assume an LCD shaped as below. The bits, numbered
 * right to left from zero, are marked.
 *         7
 *      ------
 *   6 |      | 5
 *     |   4  |
 *      ------
 *   3 |      | 2
 *     |   1  |
 *      ------   o 0
 */
static int digit[] = {
  0xEE, /* 0 */
  0x24, /* 1 */
  0xBA, /* 2 */
  0xB6, /* 3 */
  0x74, /* 4 */
  0xD6, /* 5 */
  0xDE, /* 6 */
  0xA4, /* 7 */
  0xFE, /* 8 */
  0xF6, /* 9 */
  0x00  /* space */
};

/* Each digit is stored on a display list. This array is initialized
 * in xrLCDInit ().
 */
static int digit_dlist[22];

/* Some configurable parameters. */
#define DIGIT_HEIGHT  1.0
#define DIGIT_WIDTH   0.5
#define DIGIT_SLANT   0.16667
#define DIGIT_ADVANCE 0.22917   /* Controls distance between digits. */
#define DIGIT_A       0.04167   /* Controls distance between bars. */
#define DIGIT_B       0.0625    /* Controls thickness of bars. */

#if 0
#define DIGIT_HEIGHT  48
#define DIGIT_WIDTH   24
#define DIGIT_SLANT   8
#define DIGIT_ADVANCE 11        /* Controls distance between digits. */
#define DIGIT_A       2         /* Controls distance between bars. */
#define DIGIT_B       3         /* Controls thickness of bars. */
#endif

#define USE_DIGIT_ALPHA   1     /* If set, blank bars have a slight alpha. */
#define DIGIT_ALPHA   0.1       /* Amount of alpha on transparent bars. */

#define USE_SLANT     1		/* If you want the digits to slant right. */

static void
draw_hbar (GLfloat x, GLfloat y)
{
  glBegin (GL_POLYGON);
  glVertex2f (x + DIGIT_A, y);
  glVertex2f (x + 2 * DIGIT_A, y + DIGIT_B);
  glVertex2f (x + DIGIT_WIDTH - 2 * DIGIT_A, y + DIGIT_B);
  glVertex2f (x + DIGIT_WIDTH - DIGIT_A, y);
  glVertex2f (x + DIGIT_WIDTH - 2 * DIGIT_A, y - DIGIT_B);
  glVertex2f (x + 2 * DIGIT_A, y - DIGIT_B);
  glEnd ();
}

static void
draw_vbar (GLfloat x, GLfloat y)
{
  glBegin (GL_POLYGON);
#if USE_SLANT
#define SLANT(xx, yy) (xx) + DIGIT_SLANT/2 * ((yy)-y) / (DIGIT_HEIGHT/2), (yy)
#else
#define SLANT(xx,yy) (xx),(yy)
#endif
  glVertex2f (SLANT (x, y + DIGIT_A));
  glVertex2f (SLANT (x + DIGIT_B, y + 2 * DIGIT_A));
  glVertex2f (SLANT (x + DIGIT_B, y + DIGIT_HEIGHT/2 - 2 * DIGIT_A));
  glVertex2f (SLANT (x, y + DIGIT_HEIGHT/2 - DIGIT_A));
  glVertex2f (SLANT (x - DIGIT_B, y + DIGIT_HEIGHT/2 - 2 * DIGIT_A));
  glVertex2f (SLANT (x - DIGIT_B, y + 2 * DIGIT_A));
  glEnd ();
}

static void
draw_point (GLfloat x, GLfloat y)
{
  glBegin (GL_POLYGON);
  glVertex2f (x + DIGIT_B, y);
  glVertex2f (x + DIGIT_B/2, y + DIGIT_B/2);
  glVertex2f (x, y + DIGIT_B);
  glVertex2f (x - DIGIT_B/2, y + DIGIT_B/2);
  glVertex2f (x - DIGIT_B, y);
  glVertex2f (x - DIGIT_B/2, y - DIGIT_B/2);
  glVertex2f (x, y - DIGIT_B);
  glVertex2f (x + DIGIT_B/2, y - DIGIT_B/2);
  glEnd ();
}

static int
make_lcd_digit (int d, int point,
                const GLfloat *colour)
{
  int b = digit[d];
  int dl;

  dl = glGenLists (1);
  if (dl == 0)
    xrLogFatal ("glGenLists: cannot allocate display list: %s",
		gluErrorString (glGetError ()));

  glNewList (dl, GL_COMPILE);

#if !USE_DIGIT_ALPHA
  glColor4fv (colour);
#endif

#if !USE_DIGIT_ALPHA
#define DO(b, s) if (b) s
#else
#define DO(b, s) glColor4f (colour[0], colour[1], colour[2], b ? 1.0 : 0.1); s
#endif
  DO (b & 128, draw_hbar (DIGIT_SLANT, DIGIT_HEIGHT));
  DO (b & 64,  draw_vbar (DIGIT_SLANT/2, DIGIT_HEIGHT/2));
  DO (b & 32,  draw_vbar (DIGIT_SLANT/2 + DIGIT_WIDTH, DIGIT_HEIGHT/2));
  DO (b & 16,  draw_hbar (DIGIT_SLANT/2, DIGIT_HEIGHT/2));
  DO (b & 8,   draw_vbar (0, 0));
  DO (b & 4,   draw_vbar (DIGIT_WIDTH, 0));
  DO (b & 2,   draw_hbar (0, 0));
#undef DO
  if (point)
    {
      glColor4fv (colour);
      draw_point (DIGIT_WIDTH + DIGIT_ADVANCE/2, 0);
    }

  glEndList ();

  return dl;
}

/* Program-level initializations. */
void
xrLCDInit ()
{
  /* Generate the display lists. */
  int i, j, c = 0;

  /* Generate the display lists. */
  for (i = 0; i < 11; ++i)
    for (j = 0; j < 2; ++j)
      digit_dlist[c++] = make_lcd_digit (i, j, xrColourGreen);
}

/* Draw a single LCD digit. */
void
xrLCDDrawDigit (int c, int point,
                GLfloat x, GLfloat y,
                GLfloat scale)
{
  glPushMatrix ();
  glTranslatef (x, y, 0);
  glScalef (scale, scale, 0);
  glCallList (digit_dlist[c*2+point]);
  glPopMatrix ();
}

/* Draw a string of LCD digits. */
void
xrLCDDrawString (const char *digits,
                 GLfloat x, GLfloat y,
                 GLfloat scale)
{
  int c, point;

  while ((c = *digits++))
    {
      if (c >= '0' && c <= '9')
        {
          if (digits[0] == '.')
            {
              point = 1;
              digits++;
            }
          else
            point = 0;
          xrLCDDrawDigit (c - '0', point, x, y, scale);
          x += scale/1.37;
        }
      else if (c == ' ')
        {
          xrLCDDrawDigit (10, 0, x, y, scale);
          x += scale/1.37;
        }
    }
}

/* As above, but use familiar printf style. */
void
xrLCDPrintf (GLfloat x, GLfloat y, GLfloat scale, const char *fs, ...)
{
  va_list args;
  char message[1024];

  va_start (args, fs);
  vsnprintf (message, sizeof message, fs, args);
  va_end (args);

  xrLCDDrawString (message, x, y, scale);
}
