# -*- coding: utf-8 -*-
#
# Copyright (C) 2007-2020 Edgewall Software
# Copyright (C) 2007 Alec Thomas <alec@swapoff.org>
# All rights reserved.
#
# This software is licensed as described in the file COPYING, which
# you should have received as part of this distribution. The terms
# are also available at https://trac.edgewall.org/wiki/TracLicense.
#
# This software consists of voluntary contributions made by many
# individuals. For the exact contribution history, see the revision
# history and logs, available at https://trac.edgewall.org/.

from trac.core import *
from trac.perm import IPermissionPolicy, IPermissionRequestor

revision = "$Rev$"
url = "$URL$"

class SecurityTicketsPolicy(Component):
    """Prevent public access to security sensitive tickets.

    Add the VULNERABILITY_VIEW permission as a pre-requisite for any
    other permission check done on tickets that have the words
    "security" or "vulnerability" in the summary or keywords fields.

    Once this plugin is enabled, you'll have to insert it at the appropriate
    place in your list of permission policies, e.g.
    {{{
    [trac]
    permission_policies = SecurityTicketsPolicy, AuthzPolicy,
                          DefaultPermissionPolicy, LegacyAttachmentPolicy
    }}}
    """

    implements(IPermissionPolicy, IPermissionRequestor)

    # IPermissionPolicy methods

    def check_permission(self, action, username, resource, perm):
        # We add the 'VULNERABILITY_VIEW' pre-requisite for any action
        # other than 'VULNERABILITY_VIEW' itself, as this would lead
        # to recursion.
        if action == 'VULNERABILITY_VIEW':
            return

        # Check whether we're dealing with a ticket resource
        while resource:
            if resource.realm == 'ticket':
                break
            resource = resource.parent

        if resource and resource.realm == 'ticket' and resource.id is not None:
            for keywords, summary in self.env.db_query(
                    "SELECT keywords, summary FROM ticket WHERE id=%s",
                    (resource.id,)):
                fields = ''.join(f for f in (keywords, summary) if f).lower()
                if 'security' in fields or 'vulnerability' in fields:
                    if 'VULNERABILITY_VIEW' not in perm:
                        return False

    # IPermissionRequestor methods

    def get_permission_actions(self):
        yield 'VULNERABILITY_VIEW'
