# Copyright 2014, Tresys Technology, LLC
#
# SPDX-License-Identifier: GPL-2.0-only
#
import pytest
import setools


@pytest.mark.obj_args("tests/library/initsidquery.conf")
class TestInitialSIDQuery:

    def test_unset(self, compiled_policy: setools.SELinuxPolicy) -> None:
        """Initial SID query with no criteria"""
        # query with no parameters gets all SIDs.
        sids = sorted(compiled_policy.initialsids())

        q = setools.InitialSIDQuery(compiled_policy)
        q_sids = sorted(q.results())

        assert sids == q_sids

    def test_name_exact(self, compiled_policy: setools.SELinuxPolicy) -> None:
        """Initial SID query with exact match"""
        q = setools.InitialSIDQuery(compiled_policy, name="kernel", name_regex=False)

        sids = sorted(str(s) for s in q.results())
        assert ["kernel"] == sids

    def test_name_regex(self, compiled_policy: setools.SELinuxPolicy) -> None:
        """Initial SID query with regex match"""
        q = setools.InitialSIDQuery(compiled_policy, name="(security|unlabeled)", name_regex=True)

        sids = sorted(str(s) for s in q.results())
        assert ["security", "unlabeled"] == sids

    def test_user_exact(self, compiled_policy: setools.SELinuxPolicy) -> None:
        """Initial SID query with context user exact match"""
        q = setools.InitialSIDQuery(compiled_policy, user="user10", user_regex=False)

        sids = sorted(str(s) for s in q.results())
        assert ["fs"] == sids

    def test_user_regex(self, compiled_policy: setools.SELinuxPolicy) -> None:
        """Initial SID query with context user regex match"""
        q = setools.InitialSIDQuery(compiled_policy, user="user11(a|b)", user_regex=True)

        sids = sorted(str(s) for s in q.results())
        assert ["file", "file_labels"] == sids

    def test_role_exact(self, compiled_policy: setools.SELinuxPolicy) -> None:
        """Initial SID query with context role exact match"""
        q = setools.InitialSIDQuery(compiled_policy, role="role20_r", role_regex=False)

        sids = sorted(str(s) for s in q.results())
        assert ["any_socket"] == sids

    def test_role_regex(self, compiled_policy: setools.SELinuxPolicy) -> None:
        """Initial SID query with context role regex match"""
        q = setools.InitialSIDQuery(compiled_policy, role="role21(a|c)_r", role_regex=True)

        sids = sorted(str(s) for s in q.results())
        assert ["netmsg", "port"] == sids

    def test_type_exact(self, compiled_policy: setools.SELinuxPolicy) -> None:
        """Initial SID query with context type exact match"""
        q = setools.InitialSIDQuery(compiled_policy, type_="type30", type_regex=False)

        sids = sorted(str(s) for s in q.results())
        assert ["node"] == sids

    def test_type_regex(self, compiled_policy: setools.SELinuxPolicy) -> None:
        """Initial SID query with context type regex match"""
        q = setools.InitialSIDQuery(compiled_policy, type_="type31(b|c)", type_regex=True)

        sids = sorted(str(s) for s in q.results())
        assert ["icmp_socket", "tcp_socket"] == sids

    def test_range_exact(self, compiled_policy: setools.SELinuxPolicy) -> None:
        """Initial SID query with context range exact match"""
        q = setools.InitialSIDQuery(compiled_policy, range_="s0:c1 - s0:c0.c4")

        sids = sorted(str(s) for s in q.results())
        assert ["sysctl_modprobe"] == sids

    def test_range_overlap1(self, compiled_policy: setools.SELinuxPolicy) -> None:
        """Initial SID query with context range overlap match (equal)"""
        q = setools.InitialSIDQuery(compiled_policy, range_="s1:c1 - s1:c0.c4", range_overlap=True)

        sids = sorted(str(s) for s in q.results())
        assert ["sysctl"] == sids

    def test_range_overlap2(self, compiled_policy: setools.SELinuxPolicy) -> None:
        """Initial SID query with context range overlap match (subset)"""
        q = setools.InitialSIDQuery(compiled_policy, range_="s1:c1,c2 - s1:c0.c3",
                                    range_overlap=True)

        sids = sorted(str(s) for s in q.results())
        assert ["sysctl"] == sids

    def test_range_overlap3(self, compiled_policy: setools.SELinuxPolicy) -> None:
        """Initial SID query with context range overlap match (superset)"""
        q = setools.InitialSIDQuery(compiled_policy, range_="s1 - s1:c0.c4", range_overlap=True)

        sids = sorted(str(s) for s in q.results())
        assert ["sysctl"] == sids

    def test_range_overlap4(self, compiled_policy: setools.SELinuxPolicy) -> None:
        """Initial SID query with context range overlap match (overlap low level)"""
        q = setools.InitialSIDQuery(compiled_policy, range_="s1 - s1:c1,c2", range_overlap=True)

        sids = sorted(str(s) for s in q.results())
        assert ["sysctl"] == sids

    def test_range_overlap5(self, compiled_policy: setools.SELinuxPolicy) -> None:
        """Initial SID query with context range overlap match (overlap high level)"""
        q = setools.InitialSIDQuery(compiled_policy, range_="s1:c1,c2 - s1:c0.c4",
                                    range_overlap=True)

        sids = sorted(str(s) for s in q.results())
        assert ["sysctl"] == sids

    def test_range_subset1(self, compiled_policy: setools.SELinuxPolicy) -> None:
        """Initial SID query with context range subset match"""
        q = setools.InitialSIDQuery(compiled_policy, range_="s2:c1,c2 - s2:c0.c3",
                                    range_overlap=True)

        sids = sorted(str(s) for s in q.results())
        assert ["sysctl_fs"] == sids

    def test_range_subset2(self, compiled_policy: setools.SELinuxPolicy) -> None:
        """Initial SID query with context range subset match (equal)"""
        q = setools.InitialSIDQuery(compiled_policy, range_="s2:c1 - s2:c1.c3", range_overlap=True)

        sids = sorted(str(s) for s in q.results())
        assert ["sysctl_fs"] == sids

    def test_range_superset1(self, compiled_policy: setools.SELinuxPolicy) -> None:
        """Initial SID query with context range superset match"""
        q = setools.InitialSIDQuery(compiled_policy, range_="s3 - s3:c0.c4", range_superset=True)

        sids = sorted(str(s) for s in q.results())
        assert ["sysctl_kernel"] == sids

    def test_range_superset2(self, compiled_policy: setools.SELinuxPolicy) -> None:
        """Initial SID query with context range superset match (equal)"""
        q = setools.InitialSIDQuery(compiled_policy, range_="s3:c1 - s3:c1.c3",
                                    range_superset=True)

        sids = sorted(str(s) for s in q.results())
        assert ["sysctl_kernel"] == sids

    def test_range_proper_subset1(self, compiled_policy: setools.SELinuxPolicy) -> None:
        """Initial SID query with context range proper subset match"""
        q = setools.InitialSIDQuery(compiled_policy, range_="s4:c1,c2", range_subset=True,
                                    range_proper=True)

        sids = sorted(str(s) for s in q.results())
        assert ["sysctl_net"] == sids

    def test_range_proper_subset2(self, compiled_policy: setools.SELinuxPolicy) -> None:
        """Initial SID query with context range proper subset match (equal)"""
        q = setools.InitialSIDQuery(compiled_policy, range_="s4:c1 - s4:c1.c3", range_subset=True,
                                    range_proper=True)

        sids = sorted(str(s) for s in q.results())
        assert [] == sids

    def test_range_proper_subset3(self, compiled_policy: setools.SELinuxPolicy) -> None:
        """Initial SID query with context range proper subset match (equal low only)"""
        q = setools.InitialSIDQuery(compiled_policy, range_="s4:c1 - s4:c1.c2", range_subset=True,
                                    range_proper=True)

        sids = sorted(str(s) for s in q.results())
        assert ["sysctl_net"] == sids

    def test_range_proper_subset4(self, compiled_policy: setools.SELinuxPolicy) -> None:
        """Initial SID query with context range proper subset match (equal high only)"""
        q = setools.InitialSIDQuery(compiled_policy, range_="s4:c1,c2 - s4:c1.c3",
                                    range_subset=True, range_proper=True)

        sids = sorted(str(s) for s in q.results())
        assert ["sysctl_net"] == sids

    def test_range_proper_superset1(self, compiled_policy: setools.SELinuxPolicy) -> None:
        """Initial SID query with context range proper superset match"""
        q = setools.InitialSIDQuery(compiled_policy, range_="s5 - s5:c0.c4", range_superset=True,
                                    range_proper=True)

        sids = sorted(str(s) for s in q.results())
        assert ["sysctl_net_unix"] == sids

    def test_range_proper_superset2(self, compiled_policy: setools.SELinuxPolicy) -> None:
        """Initial SID query with context range proper superset match (equal)"""
        q = setools.InitialSIDQuery(compiled_policy, range_="s5:c1 - s5:c1.c3",
                                    range_superset=True, range_proper=True)

        sids = sorted(str(s) for s in q.results())
        assert [] == sids

    def test_range_proper_superset3(self, compiled_policy: setools.SELinuxPolicy) -> None:
        """Initial SID query with context range proper superset match (equal low)"""
        q = setools.InitialSIDQuery(compiled_policy, range_="s5:c1 - s5:c1.c4",
                                    range_superset=True, range_proper=True)

        sids = sorted(str(s) for s in q.results())
        assert ["sysctl_net_unix"] == sids

    def test_range_proper_superset4(self, compiled_policy: setools.SELinuxPolicy) -> None:
        """Initial SID query with context range proper superset match (equal high)"""
        q = setools.InitialSIDQuery(compiled_policy, range_="s5 - s5:c1.c3", range_superset=True,
                                    range_proper=True)

        sids = sorted(str(s) for s in q.results())
        assert ["sysctl_net_unix"] == sids
