/*	Manager

PIRL CVS ID: Manager.java,v 1.55 2012/04/16 06:04:10 castalia Exp

Copyright (C) 2008-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/
package	PIRL.Conductor;

import	PIRL.Conductor.Maestro.Remote_Management_Exception;
import	PIRL.Conductor.Maestro.Error_Report;

//	PIRL packages
import	PIRL.Configuration.Configuration;
import	PIRL.Configuration.Configuration_Exception;
import	PIRL.PVL.Parameter;
import	PIRL.Viewers.Splash_Screen;
import	PIRL.Viewers.Stream_Monitor;
import	PIRL.Viewers.Dialog_Box;
import	PIRL.Viewers.Icons;
import	PIRL.Viewers.Parameter_View;
import	PIRL.Viewers.View_Locator;
import	PIRL.Viewers.Blinker;
import	PIRL.Viewers.Percent_Bar;

import	javax.swing.JRootPane;
import	javax.swing.JFrame;
import	javax.swing.JMenuBar;
import	javax.swing.JMenu;
import	javax.swing.JMenuItem;
import	javax.swing.JCheckBoxMenuItem;
import	javax.swing.JPanel;
import	javax.swing.JSplitPane;
import	javax.swing.Box;
import	javax.swing.BorderFactory;
import	javax.swing.JScrollPane;
import	javax.swing.JOptionPane;
import	javax.swing.JButton;
import	javax.swing.JLabel;
import	javax.swing.ImageIcon;
import	javax.swing.KeyStroke;
import	javax.swing.ToolTipManager;
import	javax.swing.UIManager;
import	javax.swing.SwingUtilities;
import	javax.swing.text.SimpleAttributeSet;
import	javax.swing.text.StyleConstants;
import	javax.swing.border.Border;
import	java.awt.Toolkit;
import	java.awt.Color;
import	java.awt.Dimension;
import	java.awt.GridBagLayout;
import	java.awt.GridBagConstraints;
import	java.awt.Insets;
import	java.awt.event.WindowAdapter;
import	java.awt.event.WindowEvent;
import	java.awt.event.ActionEvent;
import	java.awt.event.ActionListener;
import	java.awt.event.KeyEvent;
import	java.util.Vector;
import	java.util.Iterator;


/**	A <I>Manager</I> is a GUI for managing a Conductor.
<p>
<h5>
	Configuration parameters
</h5><p>
	The following configuration parameters may be used by a Conductor
	Manager:
<dl>
<dt>Splash_Screen
<dd>While the Manager is initializing the GUI it will, by default,
	display a spash screen. To disable this feature set the Splash_Screen
	value to "disabled", "false", "no" or 0. The value may also be set to
	the minimum number of seconds that the splash screen should remain
	visible; not less than three seconds (unless 0) is allowed.

<dt>Manager_Width
<dd>The width of the Manager window (pixels). Default: 750.

<dt>Manager_Height
<dd>The height of the Manager window (pixels). Default: 463.

<dt>Manager_Location_X
<dd>The initial horizontal screen position of the Manager window.
	Default: 300.

<dt>Manager_Location_Y
<dd>The initial vertical screen position of the Manager window. Default:
	100.

<dt>Monitor_Width
<dd>The width of the log stream monitor pane (pixels). Default: 700.

<dt>Monitor_Height
<dd>The height of the loc stream monitor pane (pixels). Default: 400.

<dt>Tooltips
<dd>If set to "enabled", "true", "yes" or 1 (case insensitive) GUI
	component tooltips will be enabled on startup. Default: enabled.
</dl>
<p>
	@author		Bradford Castalia - UA/PIRL
	@version	1.55
*/
public class Manager
	extends JFrame
	implements Processing_Listener
{
/**	Class identification name with source code version and date.
*/
public static final String
	ID = "PIRL.Conductor.Manager (1.55 2012/04/16 06:04:10)";


//	Configuration:

/**	Configuration parameters.
*/
public static final String

	//	Manager:
	SPLASH_SCREEN_PARAMETER_NAME			= "Splash_Screen",
	MANAGER_WIDTH_PARAMETER_NAME			= "Manager_Width",
	MANAGER_HEIGHT_PARAMETER_NAME			= "Manager_Height",
	MANAGER_LOCATION_X_PARAMETER_NAME		= "Manager_Location_X",
	MANAGER_LOCATION_Y_PARAMETER_NAME		= "Manager_Location_Y",
	TOOLTIPS_PARAMETER_NAME					= "Tooltips",

	//	Log Monitor:
	MONITOR_WIDTH_PARAMETER_NAME			= "Monitor_Width",
	MONITOR_HEIGHT_PARAMETER_NAME			= "Monitor_Height";

/**	Class-relative name of the directory containing icon/image files.
*/
public static final String
	ICONS_DIRECTORY					= "Icons";

/**	Minimum splash screen display time (seconds) if enabled.
*/
public static final int
	SPLASH_SCREEN_MINIMUM_TIME		= 3;
private int
	Splash_Screen_Time				= SPLASH_SCREEN_MINIMUM_TIME;
private static final String
	SPLASH_SCREEN_IMAGE_FILENAME	= "Conductor_Splash.png";

private int
	Manager_Width					= 750,
	Manager_Height					= 463,
	Manager_Location_X				= 300,
	Manager_Location_Y				= 100;


//	Conductor:

private String
	Conductor_Identification;

//	The Conductor being managed, or its remote proxy.
private Management
	The_Management					= null;

//	The Conductor configuration.
private Configuration
	Conductor_Configuration			= null;
private Parameter_View
	Configuration_View				= null;

//	Sources:

private Sources_Table
	Sources_Table;
private Sources_Table_Model
	Sources							= new Sources_Table_Model
										(Conductor.SOURCES_FIELD_NAMES);
public static final int
	DEFAULT_MAX_SOURCES_ROWS		= 20;
private static int
	Default_Max_Sources_Rows		= DEFAULT_MAX_SOURCES_ROWS;
private int
	Max_Sources_Rows				= Default_Max_Sources_Rows;

//	Procedures:

private Procedures_Table
	Procedures_Table;
private Procedures_Table_Model
	Procedures						= new Procedures_Table_Model ();

//	Running, success and failure.

private volatile int
	Processing_State				= Conductor.WAITING;
public static final int
	WARNING							= -8,
	EXITING							= -9;
private volatile boolean
	Quit_Pending					= false,
	Closing							= false;
private JButton
	Processing_Button;
private static final String
	START_LABEL						= "Start",
	WAIT_LABEL						= "Wait",
	STOP_LABEL						= "Stop",
	DONE_LABEL						= "Done";

private volatile int
	Sequential_Failures				= 0,
	Stop_On_Failure					= 0;
private JLabel
	Success_Count_Label,
	Failure_Count_Label,
	Sequential_Failure_Count_Label,
	Stop_On_Failure_Label;
public static final String
	SUCCESS_ICON_NAME				= "success_medium.gif",
	FAILURE_ICON_NAME				= "error_medium.gif",
	WARNING_ICON_NAME				= "warning_medium.gif";
private static ImageIcon
	Success_Icon					= null,
	Failure_Icon					= null,
	Warning_Icon					= null;
private Percent_Bar
	Success_Count_Bar,
	Failure_Count_Bar;
private Blinker
	Status_Blinker;
private static final int
	POLLING_BLINKER_RATE			= 1000,
	RUNNING_BLINKER_RATE			= 750,
	WARNING_BLINKER_RATE			= 500,
	FAILURE_BLINKER_RATE			= 250;

public static final boolean
	DEFAULT_STATUS_ANNUNCIATOR		= true;
private JCheckBoxMenuItem
	Annunciator_Checkbox;


//	Log Monitor:

private JFrame
	Log_Monitor_Window				= null;

private Stream_Monitor
	Log_Monitor						= new Stream_Monitor ();

public static final boolean
	DEFAULT_LOG_WHILE_CLOSED		= false;
private JCheckBoxMenuItem
	Log_While_Closed_Checkbox;

private int
	Log_Monitor_Width				= 700,
	Log_Monitor_Height				= 400;

private static final SimpleAttributeSet
	stdout_STYLE					= new SimpleAttributeSet (),
	stderr_STYLE					= new SimpleAttributeSet ();
static
	{
	StyleConstants.setFontFamily	(stdout_STYLE, "Monospaced");
	StyleConstants.setBold			(stdout_STYLE, true);

	StyleConstants.setFontFamily	(stderr_STYLE, "Monospaced");
	StyleConstants.setBold			(stderr_STYLE, true);
	StyleConstants.setBackground	(stderr_STYLE, Color.YELLOW);
	}


//	Miscellaneous.

private JCheckBoxMenuItem
	Tooltips_Checkbox;
private boolean
	Tooltips_Enabled			= true;

private JMenu
	Options_Menu;
private JMenuItem
	Quit_Menu_Item;

private static final Toolkit
	TOOLKIT						= Toolkit.getDefaultToolkit ();
private static final int
	MENU_SHORTCUT_KEY_MASK		= TOOLKIT.getMenuShortcutKeyMask ();

private static final String
	NL							= System.getProperty ("line.separator");


// Debug control.
private static final int
	DEBUG_OFF					= 0,
	DEBUG_CONSTRUCTOR			= 1 << 1,
	DEBUG_CONFIG				= 1 << 2,
	DEBUG_PROCESSING_LISTENER	= 1 << 3,
	DEBUG_PROCESSING_ACTIONS	= 1 << 4,
	DEBUG_PROCESSING_STATE		= 1 << 14,
	DEBUG_GUI_ACTIONS			= 1 << 5,
	DEBUG_SOURCES				= 1 << 6,
	DEBUG_SOURCES_RENDERER		= 1 << 7,
	DEBUG_SOURCES_MODEL			= 1 << 8,
	DEBUG_PROCEDURES			= 1 << 9,
	DEBUG_PROCEDURES_RENDERER	= 1 << 10,
	DEBUG_PROCEDURES_MODEL		= 1 << 11,
	DEBUG_LOG_MONITOR			= 1 << 12,
	DEBUG_ICONS					= 1 << 13,
	DEBUG_ALL					= -1,

	DEBUG						= DEBUG_OFF;

/*==============================================================================
	Constructors
*/
/**	Constructs a Conductor Manager.
<p>
	The Configuration object is expected to contain all the necessary
	information Conductor needs to connect to the database as well as
	any other Conductor parameters it might use.
<p>
	@param	management	A Condcutor_Mangement object that is a Conductor
		or a proxy - such as a Stage_Manager system - for managing a
		Conductor.
	@throws	IllegalArgumentException	If the management is null.
	@throws	Remote_Management_Exception	If a Conductor Management protocol
		error occurs. This can only happen if the management is not
		{@link #Local_Management(Management) local} to a Conductor.
*/
public Manager
	(
	Management		management,
	Configuration	configuration
	)
	throws Remote_Management_Exception
{
super ("Conductor Manager 1.55");
if ((DEBUG & DEBUG_CONSTRUCTOR) != 0)
	System.out.println
		(">>> Manager");

if ((The_Management = management) == null)
	{
	if (Local_Management (The_Management))
		Dialog_Box.Error ("No Conductor provided to manage!");
	throw new IllegalArgumentException (ID
		+ "No Conductor provided to manage!");
	}

if ((DEBUG & DEBUG_CONSTRUCTOR) != 0)
	System.out.println
		("    Configure the Manager ...");
//	Must be done before the Conductor_Configuration is obtained.
Configure (configuration);

//	Use the cross-platform LAF.
try {UIManager.setLookAndFeel
	(UIManager.getCrossPlatformLookAndFeelClassName ());}
catch (Exception exception) {/* Just leave the existing LAF. */}

Splash_Screen
	splash = null;
if (Local_Management (The_Management) &&
	Splash_Screen_Time > 0)
	{
	if ((DEBUG & DEBUG_CONSTRUCTOR) != 0)
		System.out.println
			("    Splash_Screen ...");
	//	Set the location of Icons to be relative to this class.
	Class
		class_relative = Icons.Class_Relative ();
	String
		icons_directory = Icons.Directory (ICONS_DIRECTORY, this.getClass ());

	//	Present a splash screen.
	splash = new Splash_Screen (SPLASH_SCREEN_IMAGE_FILENAME).Start ();

	//	Restore the previous Icons directory.
	Icons.Directory (icons_directory, class_relative);
	}
else
	Splash_Screen_Time = 0;

if ((DEBUG & DEBUG_CONSTRUCTOR) != 0)
	System.out.println
		("    Conductor_Configuration  ...");
//	Must be done after Configure but before GUI setup.
if ((Conductor_Configuration = The_Management.Configuration ()) == null)
	{
	//	Provide an empty configuration.
	try {Conductor_Configuration = new Configuration ((Parameter)null);}
	catch (Configuration_Exception exception) {/* Shouldn't happen */}
	}

if ((DEBUG & DEBUG_CONSTRUCTOR) != 0)
	System.out.println
		("    Icons ...");
Load_Icons ();

if ((DEBUG & DEBUG_CONSTRUCTOR) != 0)
	System.out.println
		("    Menus ...");
setJMenuBar (Menus ());
if ((DEBUG & DEBUG_CONSTRUCTOR) != 0)
	System.out.println
		("    Panels ...");
getContentPane ().add (Panels ());
setPreferredSize (new Dimension (Manager_Width, Manager_Height));
pack ();
setLocation (Manager_Location_X, Manager_Location_Y);

setDefaultCloseOperation (JFrame.DO_NOTHING_ON_CLOSE);
addWindowListener (new WindowAdapter ()
	{
	public void windowClosing (WindowEvent event)
		{Close ();}
	public void windowClosed (WindowEvent event)
		{Close ();}
	});

//	Initialize the log monitor window.
if ((DEBUG & DEBUG_CONSTRUCTOR) != 0)
	System.out.println
		("    Log_Window ...");
Log_Window ();

//	Initialize the Conductor state variables and their GUI displays.


if ((DEBUG & DEBUG_CONSTRUCTOR) != 0)
	System.out.println
		("    Report_Processing_State  ...");
Report_Processing_State (The_Management.Processing_State ());

if ((DEBUG & DEBUG_CONSTRUCTOR) != 0)
	System.out.println
		("    Refresh_Procedure_Records ...");
Refresh_Procedure_Records ();

if ((DEBUG & DEBUG_CONSTRUCTOR) != 0)
	System.out.println
		("    Report_Success_Count ...");
Report_Success_Count ();

if ((DEBUG & DEBUG_CONSTRUCTOR) != 0)
	System.out.println
		("    Set_Stop_on_Failure ...");
Set_Stop_on_Failure
	(Config_Value (Conductor.STOP_ON_FAILURE_PARAMETER, Stop_On_Failure));

if ((DEBUG & DEBUG_CONSTRUCTOR) != 0)
	System.out.println
		("    Report_Sequential_Failures ...");
Report_Sequential_Failures (The_Management.Sequential_Failures ());

if (Splash_Screen_Time > 0)
	{
	//	Have the splash screen stop and display the GUI.
	splash
		.Parent (this)
		.Duration (Splash_Screen_Time);
	splash = null;	//	Dispose of the splash screen.
	}
else
	setVisible (true);

if ((DEBUG & DEBUG_CONSTRUCTOR) != 0)
	System.out.println
		("    Add_Processing_Listener ...");
The_Management.Add_Processing_Listener (this);

if ((DEBUG & DEBUG_CONSTRUCTOR) != 0)
	System.out.println
		("<<< Manager");
}


public Manager
	(
	Management		management
	)
	throws Remote_Management_Exception
{this (management, null);}


private Manager () {}

/*==============================================================================
	Accessors
*/
public Management Management ()
{return The_Management;}


public int Default_Max_Sources_Rows ()
{return Default_Max_Sources_Rows;}

public Manager Default_Max_Sources_Rows
	(
	int		rows
	)
{
if (rows < DEFAULT_MAX_SOURCES_ROWS)
	rows = DEFAULT_MAX_SOURCES_ROWS;
Default_Max_Sources_Rows = rows;
return this;
}

public int Max_Sources_Rows ()
{return Max_Sources_Rows;}

public Manager Max_Sources_Rows
	(
	int		rows
	)
{
if (rows < DEFAULT_MAX_SOURCES_ROWS)
	rows = DEFAULT_MAX_SOURCES_ROWS;
Max_Sources_Rows = rows;
return this;
}

/*==============================================================================
	Configuration
*/
private void Configure
	(
	Configuration	configuration
	)
{
/*	Temporary use of Conductor_Configuration.

	The Config_Value methods presume the use of the Conductor_Configuration.
	Obviously it is critical that this method be called before the
	Conductor_Configuration is obtained from The_Management.
*/
if ((Conductor_Configuration = configuration) == null)
	//	Nothing to configure.
	return;
if ((DEBUG & DEBUG_CONFIG) != 0)
	System.out.println
		(">>> Manager.Configure -" + NL
		+ Conductor_Configuration.Description ());

int
	value;
Manager_Location_X = Config_Value
	(MANAGER_LOCATION_X_PARAMETER_NAME, Manager_Location_X);
Manager_Location_Y = Config_Value
	(MANAGER_LOCATION_Y_PARAMETER_NAME, Manager_Location_Y);

value = Config_Value
	(MANAGER_WIDTH_PARAMETER_NAME, Manager_Width);
if (value < 10)
	value = 10;
Manager_Width = value;

value = Config_Value
	(MANAGER_HEIGHT_PARAMETER_NAME, Manager_Height);
if (value < 10)
	value = 10;
Manager_Height = value;

value = Config_Value
	(MONITOR_WIDTH_PARAMETER_NAME, Log_Monitor_Width);
if (value < 10)
	value = 10;
Log_Monitor_Width = value;

value = Config_Value
	(MONITOR_HEIGHT_PARAMETER_NAME, Log_Monitor_Height);
if (value < 10)
	value = 10;
Log_Monitor_Height = value;

//	Splash screen display.
Splash_Screen_Time = Config_Value (SPLASH_SCREEN_PARAMETER_NAME, -1);
if (Splash_Screen_Time < 0)
	{
	//	Try for a boolean value.
	if (Config_Flag (SPLASH_SCREEN_PARAMETER_NAME, false))
		Splash_Screen_Time = SPLASH_SCREEN_MINIMUM_TIME;
	else
		Splash_Screen_Time = 0;
	}
else
if (Splash_Screen_Time > 0 &&
	Splash_Screen_Time < SPLASH_SCREEN_MINIMUM_TIME)
	Splash_Screen_Time = SPLASH_SCREEN_MINIMUM_TIME;

Tooltips_Enabled = Conductor_Configuration.Enabled
	(TOOLTIPS_PARAMETER_NAME, Tooltips_Enabled);
if ((DEBUG & DEBUG_CONFIG) != 0)
	System.out.println
		("<<< Manager.Configure");
}

/**	Get the configuration pathname for a parameter name.
<p>
	If the name is not an {@link Configuration#Is_Absolute_Pathname(String)
	absolute pathname} the {@link Conductor#CONDUCTOR_GROUP} name is
	prepended to the name to form an absolute pathname.
<p>
	@param	name	A parameter name String.
	@return	The possibily modified abosulte pathname String.
*/
public String Config_Pathname
	(
	String	name
	)
{
if (name != null &&
	Conductor_Configuration != null &&
	! Configuration.Is_Absolute_Pathname (name))
	return
		Conductor_Configuration.Path_Delimiter () + Conductor.CONDUCTOR_GROUP +
		Conductor_Configuration.Path_Delimiter () + name;
return name;
}

/**	Get the String value of a configuration parameter.
<p>
	The first value of a Parameter with an Array Value will be returned.
<p>
	@param	name	The name of the parameter from which to obtain
		the value. If the name may be {@link #Config_Pathname(String)
		qualified} before use.
	@return	A String value. This will be null if the parameter could
		not be found in the Conductor configuration.
*/
public String Config_Value
	(
	String	name
	)
{
if (Conductor_Configuration == null)
	return null;
if ((DEBUG & DEBUG_CONFIG) != 0)
	System.out.println
		(">-< Config_Value: " + name + " = "
			+ Conductor_Configuration.Get_One (Config_Pathname (name)));
return Conductor_Configuration.Get_One (Config_Pathname (name));
}


private int Config_Value
	(
	String	name,
	int		default_value
	)
{
name = Config_Value (name);
if (name != null)
	{
	try {return Integer.parseInt (name);}
	catch (NumberFormatException exception) {}
	}
return default_value;
}


private boolean Config_Flag
	(
	String	name,
	boolean	default_value
	)
{
if ((DEBUG & DEBUG_CONFIG) != 0)
	System.out.println
		(">-< Config_Flag: " + name + " = "
			+ ((Conductor_Configuration == null) ? default_value :
				Conductor_Configuration.Enabled
					(Config_Pathname (name), default_value)));
if (Conductor_Configuration == null)
	return default_value;
return Conductor_Configuration.Enabled (Config_Pathname (name), default_value);
}


private void Report_Configuration_Change
	(
	Configuration	configuration
	)
{
Conductor_Configuration = configuration;
Set_Stop_on_Failure
	(Config_Value (Conductor.STOP_ON_FAILURE_PARAMETER, Stop_On_Failure));

if (Configuration_View != null &&
	Configuration_View.isVisible ())
	Configuration_View.Parameter_Pane ().Parameter (Conductor_Configuration);
}

/*==============================================================================
	GUI
*/
private JMenuBar Menus ()
{
JMenuBar
	menu_bar = new JMenuBar ();
JMenu
	menu;
JMenuItem
	menu_item;

//	File.
menu = new JMenu ("File");

Quit_Menu_Item = new JMenuItem ("Quit");
Quit_Menu_Item.setMnemonic (KeyEvent.VK_Q);
Quit_Menu_Item.setAccelerator (KeyStroke.getKeyStroke
	(KeyEvent.VK_Q, MENU_SHORTCUT_KEY_MASK));
Quit_Menu_Item.addActionListener (new ActionListener()
	{public void actionPerformed (ActionEvent event)
		{Quit_Conductor ();}});

if (! Local_Management (The_Management))
	{
	Quit_Menu_Item.setAccelerator (KeyStroke.getKeyStroke
	(KeyEvent.VK_Q, ActionEvent.ALT_MASK));

	menu_item = new JMenuItem ("Close");
	menu_item.setMnemonic (KeyEvent.VK_C);
	menu_item.setAccelerator (KeyStroke.getKeyStroke
		(KeyEvent.VK_W, MENU_SHORTCUT_KEY_MASK));
	menu_item.addActionListener (new ActionListener()
		{public void actionPerformed (ActionEvent event)
			{Close ();}});
	menu.add (menu_item);
	}

menu.add (Quit_Menu_Item);

menu_bar.add (menu);

//	Options.
Options_Menu = new JMenu ("Options");

menu_item = new JMenuItem ("Polling Interval ...");
menu_item.setMnemonic (KeyEvent.VK_P);
menu_item.setAccelerator (KeyStroke.getKeyStroke
	(KeyEvent.VK_P, MENU_SHORTCUT_KEY_MASK));
menu_item.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
		{Set_Poll_Interval ();}});
Options_Menu.add (menu_item);

menu_item = new JMenuItem ("Default Value ...");
menu_item.setMnemonic (KeyEvent.VK_V);
menu_item.setAccelerator (KeyStroke.getKeyStroke
	(KeyEvent.VK_V, MENU_SHORTCUT_KEY_MASK));
menu_item.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
		{Set_Default_Value ();}});
Options_Menu.add (menu_item);

menu_item = new JMenuItem ("Stop On Failure ...");
menu_item.setMnemonic (KeyEvent.VK_F);
menu_item.setAccelerator (KeyStroke.getKeyStroke
	(KeyEvent.VK_F, MENU_SHORTCUT_KEY_MASK));
menu_item.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
		{Set_Stop_on_Failure ();}});
Options_Menu.add (menu_item);

menu_item = new JMenuItem ("Reset Sequential Failures");
menu_item.setMnemonic (KeyEvent.VK_R);
menu_item.setAccelerator (KeyStroke.getKeyStroke
	(KeyEvent.VK_R, MENU_SHORTCUT_KEY_MASK));
menu_item.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
		{Reset_Sequential_Failures ();}});
Options_Menu.add (menu_item);

menu_bar.add (Options_Menu);

//	View:
menu = new JMenu ("View");

menu_item = new JMenuItem ("Configuration");
menu_item.setMnemonic (KeyEvent.VK_C);
menu_item.setAccelerator (KeyStroke.getKeyStroke
	(KeyEvent.VK_C, ActionEvent.ALT_MASK));
menu_item.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
		{View_Configuration ();}});
menu.add (menu_item);

menu_item = new JMenuItem ("Log Monitor");
menu_item.setMnemonic (KeyEvent.VK_L);
menu_item.setAccelerator (KeyStroke.getKeyStroke
	(KeyEvent.VK_L, ActionEvent.ALT_MASK));
menu_item.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
	{Show_Log_Monitor ();}});
menu.add (menu_item);

Tooltips_Checkbox
	= new JCheckBoxMenuItem ("Tooltips", Tooltips_Enabled);
ToolTipManager.sharedInstance ().setEnabled (Tooltips_Enabled);
Tooltips_Checkbox.setMnemonic (KeyEvent.VK_T);
Tooltips_Checkbox.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
	{ToolTipManager.sharedInstance ().setEnabled
		(Tooltips_Enabled = ! Tooltips_Enabled);}});
menu.add (Tooltips_Checkbox);

Annunciator_Checkbox
	= new JCheckBoxMenuItem ("Annunciator", DEFAULT_STATUS_ANNUNCIATOR);
Annunciator_Checkbox.setMnemonic (KeyEvent.VK_A);
Annunciator_Checkbox.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
	{Status_Blinking ();}});
menu.add (Annunciator_Checkbox);

menu.addSeparator ();

menu_item = new JMenuItem ("Max Sources ...");
menu_item.setMnemonic (KeyEvent.VK_S);
menu_item.setAccelerator (KeyStroke.getKeyStroke
	(KeyEvent.VK_S, ActionEvent.ALT_MASK));
menu_item.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
		{Set_Max_Sources ();}});
menu.add (menu_item);

menu_bar.add (menu);

return menu_bar;
}


private JPanel Panels ()
{
JPanel
	panel = new JPanel (new GridBagLayout ()),
	sources_panel,
	procedures_panel;
GridBagConstraints
	location = new GridBagConstraints ();

//	Pipeline identification:
location.anchor		= GridBagConstraints.WEST;
location.gridwidth	= GridBagConstraints.REMAINDER;
location.insets		= new Insets (5, 5, 5, 5);
panel.add (Identification_Panel (), location);

//	Sources and Procedures tables.
JSplitPane
	split_pane = new JSplitPane (JSplitPane.VERTICAL_SPLIT, 
 		Sources_Panel (), Procedures_Panel ());
split_pane.setOneTouchExpandable (true);
location.anchor		= GridBagConstraints.WEST;
location.fill		= GridBagConstraints.BOTH;
location.weightx	= 1.0;
location.weighty	= 1.0;
location.insets		= new Insets (0, 5, 5, 5);
panel.add (split_pane, location);

//	Counts and Control.
location.anchor		= GridBagConstraints.WEST;
location.fill		= GridBagConstraints.HORIZONTAL;
location.weighty	= 0.0;
panel.add (Counts_and_Control_Panel (), location);

return panel;
}


private JPanel Identification_Panel ()
{
JPanel
	panel = new JPanel ();
Conductor_Identification
	= Config_Value (Conductor.CATALOG_PARAMETER)
	+ '.'
	+ Config_Value (Conductor.PIPELINE_PARAMETER)
	+ " on "
	+ Config_Value (Conductor.CONDUCTOR_ID_PARAMETER);
panel.add (new JLabel ("Pipeline: " + Conductor_Identification));

setTitle ("Conductor Manager - " + Conductor_Identification);
return panel;
}

/*------------------------------------------------------------------------------
	Status Counts and Control
*/
private JPanel Counts_and_Control_Panel ()
{
JPanel
	panel = new JPanel (new GridBagLayout ());
GridBagConstraints
	location = new GridBagConstraints ();

//	Counts.
location.anchor		= GridBagConstraints.WEST;
location.fill		= GridBagConstraints.BOTH;
location.weightx	= 1.0;
location.weighty	= 1.0;
location.insets		= new Insets (0, 5, 5, 5);
panel.add (Status_Counts_Panel (), location);

//	Processing control.
Processing_Button = new JButton (START_LABEL);
Processing_Button.setOpaque (true);
Processing_Button.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent e)
	{Processing_Action ();}});
Status_Blinker = new Blinker (Processing_Button,
	Colors.RUNNING_BLINKER, Blinker.INDEFINITELY, 0, RUNNING_BLINKER_RATE);
getRootPane ().setDefaultButton (Processing_Button);
location.weightx	= 0.0;
location.weighty	= 0.0;
location.gridwidth	= GridBagConstraints.REMAINDER;
panel.add (Processing_Button, location);

return panel;
}


private JPanel Status_Counts_Panel ()
{
JPanel
	panel = new JPanel (new GridBagLayout ());
GridBagConstraints
	location = new GridBagConstraints ();
JLabel
	label;
int
	width,
	height;

//	Success.
if (Success_Icon != null)
	{
	label = new JLabel (Success_Icon);
	width = Success_Icon.getIconWidth ();
	width += width / 2;
	height = Success_Icon.getIconHeight ();
	location.insets	= new Insets (0, 5, 5, 5);
	}
else
	{
	label = new JLabel ("Success:");
	label.setOpaque (true);
	label.setBackground (Colors.SUCCESS);
	width = label.getWidth ();
	height = label.getHeight ();
	location.insets		= new Insets (0, 5, 5, 0);
	}
label.setToolTipText
	("Source records successfully processed");
location.anchor		= GridBagConstraints.EAST;
location.fill		= GridBagConstraints.BOTH;
location.weightx	= 0.0;
location.weighty	= 1.0;
panel.add (label, location);

Success_Count_Label	= new JLabel ("0", JLabel.RIGHT);
Success_Count_Label.setMinimumSize (new Dimension (width, height));
Success_Count_Label.setOpaque (true);
Success_Count_Label.setBackground (Colors.SUCCESS);
Success_Count_Label.setToolTipText
	("Source records successfully processed: Total count");
location.insets		= new Insets (0, 0, 5, 0);
panel.add (Success_Count_Label, location);

Success_Count_Bar	= new Percent_Bar ();
Success_Count_Bar.setForeground (Colors.RUNNING_BLINKER);
Success_Count_Bar.setOpaque (false);
Success_Count_Bar.setPreferredSize (new Dimension (50, height));
Success_Count_Bar.setToolTipText
	("Source records successfully processed: Percent of all processed");
location.anchor		= GridBagConstraints.WEST;
location.weightx	= 1.0;
location.gridwidth	= GridBagConstraints.REMAINDER;
panel.add (Success_Count_Bar, location);

//	Failure.
if (Failure_Icon != null)
	{
	label = new JLabel (Failure_Icon);
	location.insets	= new Insets (0, 5, 5, 5);
	}
else
	{
	label = new JLabel ("Failure:");
	label.setOpaque (true);
	label.setBackground (Colors.FAILURE);
	location.insets	= new Insets (0, 5, 5, 0);
	}
label.setToolTipText
	("Source records failed processing");
location.anchor		= GridBagConstraints.EAST;
location.weightx	= 0.0;
location.gridwidth	= 1;
panel.add (label, location);

Failure_Count_Label	= new JLabel ("0", JLabel.RIGHT);
Failure_Count_Label.setOpaque (true);
Failure_Count_Label.setBackground (Colors.FAILURE);
Failure_Count_Label.setMinimumSize (new Dimension (width, height));
Failure_Count_Label.setToolTipText
	("Source records failed processing: Total count");
location.insets		= new Insets (0, 0, 5, 0);
panel.add (Failure_Count_Label, location);

Failure_Count_Bar	= new Percent_Bar ();
Failure_Count_Bar.setForeground (Colors.FAILURE_BLINKER);
Failure_Count_Bar.setOpaque (false);
Failure_Count_Bar.setPreferredSize (new Dimension (50, height));
Failure_Count_Bar.setToolTipText
	("Source records failed processing: Percent of all processed");
location.anchor		= GridBagConstraints.WEST;
location.weightx	= 1.0;
location.gridwidth	= GridBagConstraints.REMAINDER;
panel.add (Failure_Count_Bar, location);

//	Sequential.
location.anchor		= GridBagConstraints.EAST;
location.weightx	= 0.0;
location.gridwidth	= 1;
if (Warning_Icon != null)
	{
	location.insets	= new Insets (0, 5, 5, 5);
	label = new JLabel (Warning_Icon);
	label.setToolTipText
		("Source records sequentially failed processing");
	panel.add (label, location);
	}
else
	panel.add (Box.createHorizontalGlue (), location);

Sequential_Failure_Count_Label =
	new JLabel (String.valueOf (Sequential_Failures), JLabel.RIGHT);
Sequential_Failure_Count_Label.setOpaque (true);
Sequential_Failure_Count_Label.setBackground (Colors.WARNING);
Sequential_Failure_Count_Label.setMinimumSize (new Dimension (width, height));
Sequential_Failure_Count_Label.setToolTipText
	("Source records sequentially failed processing: Count");
location.insets		= new Insets (0, 0, 5, 0);
panel.add (Sequential_Failure_Count_Label, location);

Stop_On_Failure_Label = new JLabel ("/" + Stop_On_Failure + " Sequential");
Stop_On_Failure_Label.setOpaque (true);
Stop_On_Failure_Label.setBackground (Colors.WARNING);
Stop_On_Failure_Label.setToolTipText
	("Source records sequentially failed processing: Stop at amount");
location.anchor		= GridBagConstraints.WEST;
location.fill		= GridBagConstraints.VERTICAL;
location.gridwidth	= GridBagConstraints.REMAINDER;
panel.add (Stop_On_Failure_Label, location);

return panel;
}

/*------------------------------------------------------------------------------
	Sources
*/
private JPanel Sources_Panel ()
{
JPanel
	panel = new JPanel (new GridBagLayout ());
GridBagConstraints
	location = new GridBagConstraints ();

panel.setBorder (BorderFactory.createTitledBorder ("Sources"));

Sources_Table = new Sources_Table (Sources);
JScrollPane
	scroll_pane		= new JScrollPane (Sources_Table);
scroll_pane.setPreferredSize (new Dimension (100, 100));
scroll_pane.setVerticalScrollBarPolicy
	(JScrollPane.VERTICAL_SCROLLBAR_ALWAYS);
scroll_pane.setHorizontalScrollBarPolicy
	(JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED);
location.anchor		= GridBagConstraints.WEST;
location.fill		= GridBagConstraints.BOTH;
location.weightx	= 1.0;
location.weighty	= 1.0;
location.gridwidth	= GridBagConstraints.REMAINDER;
panel.add (scroll_pane, location);

return panel;
}

//..............................................................................

private void Sources_Changed
	(
	int		row
	)
{
if (row >= Sources.getRowCount ())
	return;
if (row < 0)
	Sources.fireTableRowsUpdated (0, Sources.getRowCount () - 1);
else
	Sources.fireTableRowsUpdated (row, row);
}


private void Report_Source_Record
	(
	Vector	record
	)
{
if ((DEBUG & DEBUG_SOURCES) != 0)
	System.out.println
		(">>> Manager.Report_Source_Record: " + record);
if (record == null)
	{
	if ((DEBUG & DEBUG_SOURCES) != 0)
		System.out.println
			("<<< Manager.Report_Source_Record");
	return;
	}
try
	{
	if ((DEBUG & DEBUG_SOURCES) != 0)
		System.out.println
			("    Getting record field " + Conductor.SOURCE_NUMBER_FIELD);
	String
		source_number = (String)record.get (Conductor.SOURCE_NUMBER_FIELD);
	if ((DEBUG & DEBUG_SOURCES) != 0)
		System.out.println
			("    Record source number: " + source_number + NL
			+"    Sources ("
					+ Sources.Current_Row + ',' + Conductor.SOURCE_NUMBER_FIELD
					+ ") field: " 
					+ (String)Sources.getValueAt
						(Sources.Current_Row, Conductor.SOURCE_NUMBER_FIELD));
	if (Sources.Current_Row >= 0 &&
		source_number
			.equals ((String)Sources.getValueAt
				(Sources.Current_Row, Conductor.SOURCE_NUMBER_FIELD)))
		{
		//	Update the source record.
		if ((DEBUG & DEBUG_SOURCES) != 0)
			System.out.println
				("    Update Sources record " + Sources.Current_Row);
		Sources.getDataVector ().set (Sources.Current_Row, record);
		Sources_Changed (Sources.Current_Row);
		}
	else
		{
		//	New source record.
		Sources.Current_Row++;
		if ((DEBUG & DEBUG_SOURCES) != 0)
			System.out.println
				("    Source_Row: " + Sources.Current_Row);
		if (Sources.Current_Row == Max_Sources_Rows)
			{
			if ((DEBUG & DEBUG_SOURCES) != 0)
				System.out.println
					("    Removing Sources record 0");
			Sources.removeRow (0);
			Sources.Current_Row--;
			Sources_Changed (Sources.Current_Row - 1);
			}

		if ((DEBUG & DEBUG_SOURCES) != 0)
			System.out.println
				("    Adding Sources record " + Sources.Current_Row);
		Sources.addRow (record);

		//	Auto-scroll the table if the previous last row was visible.
		int
			row = Sources.getRowCount () - 2;
		if (row < 0 ||
			Sources_Table.Table_Row_Is_Visible (Sources_Table, row))
			Sources_Table.Show_Table_Row (Sources_Table, ++row);
		}
	}
catch (ArrayIndexOutOfBoundsException exception)
	{
	//	This shouldn't happen if the record that was sent is valid.
	if ((DEBUG & DEBUG_SOURCES) != 0)
		System.out.println
			("!!! " + exception);
	}
if ((DEBUG & DEBUG_SOURCES) != 0)
	System.out.println
		("<<< Manager.Report_Source_Record");
}

/**	Update the success/failure status reporting.
<p>
	Every time a source record completes processing the sequential
	failures count is reported. If the source record completed processing
	successfully the sequential failures count will be set to zero. The
	count will also be set to zero when Conductor acknowledges a {@link
	#Reset_Sequential_Failures()) request.
*/
private void Report_Sequential_Failures
	(
	int		sequential_failures
	)
{
if ((DEBUG & DEBUG_PROCESSING_ACTIONS) != 0)
	System.out.println
		(">>> Manager.Report_Sequential_Failures: " + sequential_failures);
if ((Sequential_Failures = sequential_failures) == 0)
	{
	Sequential_Failure_Count_Label.setText ("0");
	Stop_On_Failure_Label.setText
		("/" + Stop_On_Failure + " Sequential");
	if (Processing_State > 0)
		//	While running.
		Report_Success_Count ();
	else if (Status_Blinker.isRunning ())
		{
		//	While stopped.
		Status_Blinker.stop ();
		Reset_Procedures_Table ();
		}
	}
else
	{
	Report_Success_Count ();
	Sequential_Failure_Count_Label.setText
		(String.valueOf (Sequential_Failures));
	Stop_On_Failure_Label.setText
		("/" + Stop_On_Failure + " Sequential");
	}

Status_Blinker_State ();
if ((DEBUG & DEBUG_PROCESSING_ACTIONS) != 0)
	System.out.println
		("<<< Manager.Report_Sequential_Failures");
}


private void Report_Success_Count ()
{
if ((DEBUG & DEBUG_PROCESSING_ACTIONS) != 0)
	System.out.println
		(">>> Manager.Report_Success_Count");
String
	success_string = Config_Value (Conductor.SOURCE_SUCCESS_COUNT);
if (success_string == null)
	return;
String
	failure_string = Config_Value (Conductor.TOTAL_FAILURE_COUNT);
if (failure_string == null)
	return;
int
	success_count,
	failure_count;
try
	{
	success_count = Integer.parseInt (success_string);
	failure_count = Integer.parseInt (failure_string);
	}
catch (NumberFormatException exception)
	{return;}

Success_Count_Label.setText (success_string);
Failure_Count_Label.setText (failure_string);
Success_Count_Bar.Percent
	((100.0 * success_count) / (success_count + failure_count));
Failure_Count_Bar.Percent
	(100.0 - Success_Count_Bar.Percent ());
if ((DEBUG & DEBUG_PROCESSING_ACTIONS) != 0)
	System.out.println
		("    Success_Count = " + success_count + NL
		+"    Failure_Count = " + failure_count + NL
		+"        Success % = " + Success_Count_Bar.Percent () + NL
		+"        Failure % = " + Failure_Count_Bar.Percent () + NL
		+"<<< Manager.Report_Success_Count");
}


private void Report_Error_Condition
	(
	String	report
	)
{
if (report == null)
	return;

TOOLKIT.beep ();
new Error_Report (Conductor_Identification,
	"The Conductor reported an error condition.",
	report, this);
}

/*------------------------------------------------------------------------------
	Procedures
*/
private JPanel Procedures_Panel ()
{
JPanel
	panel = new JPanel (new GridBagLayout ());
GridBagConstraints
	location = new GridBagConstraints ();

panel.setBorder (BorderFactory.createTitledBorder ("Procedures"));

Procedures_Table = new Procedures_Table (Procedures);
JScrollPane
	scroll_pane		= new JScrollPane (Procedures_Table);
scroll_pane.setPreferredSize (new Dimension (100, 50));
scroll_pane.setVerticalScrollBarPolicy
	(JScrollPane.VERTICAL_SCROLLBAR_ALWAYS);
scroll_pane.setHorizontalScrollBarPolicy
	(JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED);
location.anchor		= GridBagConstraints.WEST;
location.fill		= GridBagConstraints.BOTH;
location.weightx	= 1.0;
location.weighty	= 1.0;
location.gridwidth	= GridBagConstraints.REMAINDER;
panel.add (scroll_pane, location);

return panel;
}

//..............................................................................

/**	Refresh the table of procedures.
<p>
	The current table of procedures is obtained from the Management.
	This table is expected to have its records sorted and its fields
	ordered.
*/
private void Refresh_Procedure_Records ()
{
if ((DEBUG & DEBUG_PROCEDURES) != 0)
	System.out.println
		(">>> Manager.Refresh_Procedure_Records");
//	Get the procedures table.
Vector
	table;
try {table = The_Management.Procedures ();}
catch (Remote_Management_Exception exception)
	{
	new Error_Report ("Management Protocol Failure",
		"Failed to refresh the procedure records.",
		exception, this);
	return;
	}
if ((DEBUG & DEBUG_PROCEDURES) != 0)
	{
	System.out.println
		("    " + (table.size () - 1)
			+ " procedure records -");
	Iterator
		records = table.iterator ();
	while (records.hasNext ())
		System.out.println ((Vector)records.next ());
	}
if (table.size () == 0)
	//	No procedures.
	{
	new Error_Report ("No Procedures",
		"An empty procedures table was obtained.",
		this);
	return;
	}

Vector
	field_names = (Vector)table.remove (0);
synchronized (Procedures)
	{
	//	Automatically generates a table snapshot.
	Procedures.setDataVector (table, field_names);
	Procedures.Current_Row = -1;
	}
if ((DEBUG & DEBUG_PROCEDURES) != 0)
	System.out.println
		("    Procedures.Current_Row = -1" + NL
		+"<<< Manager.Refresh_Procedure_Records");
}


private void Report_Procedure_Record
	(
	Vector	record
	)
{
if ((DEBUG & DEBUG_PROCEDURES) != 0)
	System.out.println
		(">>> Manager.Report_Procedure_Record: " + record);
if (record == null)
	{
	Reset_Procedures_Table ();
	if ((DEBUG & DEBUG_PROCEDURES) != 0)
		System.out.println
			("<<< Manager.Report_Procedure_Record");
	return;
	}
if (Config_Value (Conductor.TOTAL_PROCEDURE_RECORDS_PARAMETER,
		Procedures.getRowCount ())
	> Procedures.getRowCount ())
	{
	if ((DEBUG & DEBUG_PROCEDURES) != 0)
		System.out.println
			("    " + Conductor.TOTAL_PROCEDURE_RECORDS_PARAMETER
				+ " = "
				+ Config_Value (Conductor.TOTAL_PROCEDURE_RECORDS_PARAMETER) + NL
			+"    Procedures rows = " + Procedures.getRowCount ());
	Refresh_Procedure_Records ();
	if ((DEBUG & DEBUG_PROCEDURES) != 0)
		System.out.println
			("<<< Manager.Report_Procedure_Record");
	return;
	}

int
	row =
		Config_Value (Conductor.PROCEDURE_COUNT_PARAMETER, -1) - 1;
if ((DEBUG & DEBUG_PROCEDURES) != 0)
	System.out.println
		("    row = " + row);
if (row < Procedures.Current_Row)
	//	Moved back to a previous procedure.
	Reset_Procedures_Table ();
Procedures.Current_Row = row;
if (Procedures.Current_Row >= 0)
	{
	//	Replace the current record in the table.
	if ((DEBUG & DEBUG_PROCEDURES) != 0)
		System.out.println
			("    Replacing record " + Procedures.Current_Row);
	try
		{
		synchronized (Procedures)
			{
			Procedures.Record (Procedures.Current_Row, record);

			//	Auto-scroll the table.
			Sources_Table.Show_Table_Row
				(Procedures_Table, Procedures.Current_Row);
			}
		}
	catch (ArrayIndexOutOfBoundsException exception)
		{
		/*	This shouldn't happen if the record that was sent is valid.

			It is possible, though very unlikely, that the procedures
			table changed while the Conductor was stopped and then when
			it was started again a procedure record report was sent from
			the Conductor before the new procedures table could be
			fetched and updated resulting in a procedure record from the
			new table that is not in the range of the old table. In this
			case the procedure record report will have no effect.
			However, after a change in the procedures table the procedure
			record reports always start from row zero, so the chance of
			procedure record reports out running the table size before
			the procedures changed notice can be handled by fetching and
			updating the table is quite slim.
		*/
		if ((DEBUG & DEBUG_PROCEDURES) != 0)
			System.out.println
				("!!! " + exception);
		}
	}
if ((DEBUG & DEBUG_PROCEDURES) != 0)
	System.out.println
		("<<< Manager.Report_Procedure_Record");
}


private void Reset_Procedures_Table ()
{
if ((DEBUG & DEBUG_PROCEDURES) != 0)
	System.out.println
		(">>> Manager.Reset_Procedures_Table");
if (Procedures.Current_Row >= 0)
	{
	//	Restore the table snapshot.
	if ((DEBUG & DEBUG_PROCEDURES) != 0)
		System.out.println
			("    Procedures.Current_Row reset from "
				+ Procedures.Current_Row + " to -1" + NL
			+"    Restoring Procedures");
	Procedures.Current_Row = -1;
	synchronized (Procedures)
		{Procedures.Restore ();}
	}
if ((DEBUG & DEBUG_PROCEDURES) != 0)
	System.out.println
		("<<< Manager.Reset_Procedures_Table");
}

/*------------------------------------------------------------------------------
	Log Monitor
*/
private void Log_Window ()
{
if ((DEBUG & DEBUG_LOG_MONITOR) != 0)
	System.out.println
		(">>> Manager.Log_Window");
Log_Monitor_Window = new JFrame (Conductor_Identification);

if ((DEBUG & DEBUG_LOG_MONITOR) != 0)
	System.out.println
		("    Log_Monitor setup");
Log_Monitor
	.Auto_Style (true)
	.Auto_Style (Conductor.STDOUT_NAME, stdout_STYLE)
	.Auto_Style (Conductor.STDERR_NAME, stderr_STYLE);
//	.Write (ID + NL + NL);
Log_Monitor.setPreferredSize (new Dimension
	(Log_Monitor_Width, Log_Monitor_Height));

if ((DEBUG & DEBUG_LOG_MONITOR) != 0)
	System.out.println
		("    Log_Monitor menus");
JMenuBar
	menu_bar = new JMenuBar ();
JMenu
	menu = Log_Monitor.File_Menu ();

menu.addSeparator ();

JMenuItem
	menu_item = new JMenuItem ("Close");
menu_item.setMnemonic (KeyEvent.VK_C);
menu_item.setAccelerator (KeyStroke.getKeyStroke
	(KeyEvent.VK_W, MENU_SHORTCUT_KEY_MASK));
menu_item.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
	{Close_Log_Monitor ();}});
menu.add (menu_item);
menu_bar.add (menu);

menu = Log_Monitor.View_Menu ();

menu.addSeparator ();

Log_While_Closed_Checkbox
	= new JCheckBoxMenuItem ("Log While Closed", DEFAULT_LOG_WHILE_CLOSED);
Log_While_Closed_Checkbox.setMnemonic (KeyEvent.VK_L);
Log_While_Closed_Checkbox.setAccelerator (KeyStroke.getKeyStroke
	(KeyEvent.VK_L, MENU_SHORTCUT_KEY_MASK));
Log_While_Closed_Checkbox.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
	{Log_While_Closed (Log_While_Closed_Checkbox.isSelected ());}});
menu.add (Log_While_Closed_Checkbox);
menu_bar.add (menu);

Log_Monitor_Window.setJMenuBar (menu_bar);

if ((DEBUG & DEBUG_LOG_MONITOR) != 0)
	System.out.println
		("    Log_Monitor panels");
JPanel
	content = new JPanel (new GridBagLayout ());
GridBagConstraints
	location = new GridBagConstraints ();
location.fill		= GridBagConstraints.BOTH;
location.weightx	= 1.0;
location.weighty	= 1.0;
location.gridwidth	= GridBagConstraints.REMAINDER;
content.add (Log_Monitor, location);
Log_Monitor_Window.setContentPane (content);
Log_Monitor_Window.pack ();

if ((DEBUG & DEBUG_LOG_MONITOR) != 0)
	System.out.println
		("    Log_Monitor window closing actions");
Log_Monitor_Window.setDefaultCloseOperation (JFrame.DO_NOTHING_ON_CLOSE);
Log_Monitor_Window.addWindowListener (new WindowAdapter ()
	{
	public void windowClosing (WindowEvent event)
		{Close_Log_Monitor ();}
	public void windowClosed (WindowEvent event)
		{Close_Log_Monitor ();}
	});
if ((DEBUG & DEBUG_LOG_MONITOR) != 0)
	System.out.println
		("<<< Manager.Log_Window");
}

/*==============================================================================
	Actions
*/
private void View_Configuration ()
{
if (Configuration_View == null)
	{
	Configuration_View = new Parameter_View
		(Conductor_Identification,  Conductor_Configuration);
	Configuration_View.setDefaultCloseOperation (JFrame.HIDE_ON_CLOSE);

	View_Locator
		locator = new View_Locator ()
			.Offsets (0, 0)
			.Vertical   (View_Locator.TOP   | View_Locator.INWARD)
			.Horizontal (View_Locator.RIGHT | View_Locator.OUTWARD);
	locator.Relocate (Configuration_View, this);
	Configuration_View.setVisible (true);
	}
else
	{
	Configuration_View.Parameter_Pane ().Parameter (Conductor_Configuration);
	Configuration_View.setVisible (true);
	Configuration_View.toFront ();
	}
}

/**	Processing_Button action method.
*/
private void Processing_Action ()
{
if ((DEBUG & (DEBUG_PROCESSING_STATE | DEBUG_PROCESSING_ACTIONS)) != 0)
	System.out.println
		(">>> Manager.Processing_Action: Processing_State "
			+ Processing_State);
if (Processing_State > 0)
	{
	if (Processing_State == Conductor.RUN_TO_WAIT)
		{
		//	Clear the stop-pending.
		if ((DEBUG & (DEBUG_PROCESSING_STATE | DEBUG_PROCESSING_ACTIONS)) != 0)
			System.out.println
				("    Start to clear run-to-wait");
		The_Management.Start ();
		}
	else
		Stop_Conductor ();
	}
else
	{
	if ((DEBUG & (DEBUG_PROCESSING_STATE | DEBUG_PROCESSING_ACTIONS)) != 0)
		System.out.println
			("    Procedures table reset");
	Reset_Procedures_Table ();
	if ((DEBUG & (DEBUG_PROCESSING_STATE | DEBUG_PROCESSING_ACTIONS)) != 0)
		System.out.println
			("    Start");
	try {The_Management.Start ();}
	catch (Remote_Management_Exception exception)
		{
		new Error_Report ("Management Protocol Failure",
			"Failed to start Conductor processing.",
			exception, this);
		}
	}
if ((DEBUG & (DEBUG_PROCESSING_STATE | DEBUG_PROCESSING_ACTIONS)) != 0)
	System.out.println
		("<<< Manager.Processing_Action");
}


public void Stop_Conductor ()
{
if (Processing_State > Conductor.RUN_TO_WAIT)
	{
	try {The_Management.Stop ();}
	catch (Remote_Management_Exception exception)
		{
		new Error_Report ("Management Protocol Failure",
			"Failed to stop Conductor processing.",
			exception, this);
		}
	}
}


private void Report_Processing_State
	(
	int		processing_state
	)
{
if ((DEBUG & (DEBUG_PROCESSING_STATE | DEBUG_PROCESSING_ACTIONS)) != 0)
	System.out.println
		(">>> Manager.Report_Processing_State: " + processing_state);
Processing_State = processing_state;
Status_Blinker_State ();
if (Processing_State > 0)
	{
	if (Processing_State == Conductor.RUN_TO_WAIT)
		Processing_Button.setText (WAIT_LABEL);
	else
		Processing_Button.setText (STOP_LABEL);
	}
else if (Quit_Menu_Item.isEnabled ())	//	i.e. not disabled.
	{
	if (Processing_State == Conductor.HALTED)
		TOOLKIT.beep ();

	if (Status_Blinker.getDelay () == RUNNING_BLINKER_RATE ||
		Status_Blinker.getDelay () == POLLING_BLINKER_RATE)
		Status_Blinker.stop ();
	Processing_Button.setText (START_LABEL);
	if ((DEBUG & (DEBUG_PROCESSING_STATE | DEBUG_PROCESSING_ACTIONS)) != 0)
		System.out.println
			("    Quit_Pending: " + Quit_Pending);
	if (Quit_Pending)
		Quit ();
	}
if ((DEBUG & (DEBUG_PROCESSING_STATE | DEBUG_PROCESSING_ACTIONS)) != 0)
	System.out.println
		("<<< Manager.Report_Processing_State");
}


private void Status_Blinker_State ()
{
Color
	color = null;
int
	rate = 0,
	state = Processing_State;
if (state != Conductor.HALTED &&
	Sequential_Failures > 0)
	state = WARNING;

switch (state)
	{
	case Conductor.RUN_TO_WAIT:
	case Conductor.RUNNING:
		color = Colors.RUNNING_BLINKER;
		rate  = RUNNING_BLINKER_RATE;
		break;
	case Conductor.POLLING:
		color = Colors.POLLING_BLINKER;
		rate  = POLLING_BLINKER_RATE;
		break;
	case Conductor.WAITING:
		break;
	case Conductor.HALTED:
		color = Colors.FAILURE_BLINKER;
		if (Sequential_Failures > 0)
			rate  = FAILURE_BLINKER_RATE;
		break;
	case WARNING:
		color = Colors.WARNING_BLINKER;
		rate  = WARNING_BLINKER_RATE;
		break;
	case EXITING:
		break;
	}

if (rate == 0)
	Status_Blinker.stop ();
else
	{
	Status_Blinker.setColor (color);
	Status_Blinker.setDelay (rate);
	Status_Blinking ();
	}
}


private void Status_Blinking ()
{
if (Annunciator_Checkbox.isSelected ())
	{
	if (Processing_State != Conductor.WAITING ||
		Sequential_Failures > 0)
		Status_Blinker.start ();
	}
else if (Status_Blinker.isRunning ())
	Status_Blinker.stop ();
}


private void Set_Poll_Interval ()
{
try
	{
	String
		value;
	while ((value = JOptionPane.showInputDialog
				(Sources_Table,
				"Polling interval" + NL
				+"for new source records (seconds):",
				String.valueOf (The_Management.Poll_Interval ())))
			!= null)
		{
		try
			{
			int
				interval = Integer.parseInt (value);
			if (interval <= 0)
				{
				interval = 0;
				if (! Dialog_Box.Confirm
					("When all available source records have been processed" + NL
					+"no polling for additional source records will be done.",
					Sources_Table))
					continue;
				}
			The_Management.Poll_Interval (interval);
			break;
			}
		catch (NumberFormatException exception)
			{Dialog_Box.Notice
				("Please enter a numeric value.", Sources_Table);}
		}
	}
catch (Remote_Management_Exception exception)
	{
	new Error_Report ("Management Protocol Failure",
		"Failed to set the poll interval.",
		exception, this);
	}
}


private void Set_Stop_on_Failure ()
{
String
	value;
try
	{
	while ((value = JOptionPane.showInputDialog
			(Sources_Table,
			"The number of sequential failures" + NL
			+"at which to stop processing.",
			String.valueOf (Stop_On_Failure)))
			!= null)
		{
		try
			{
			int
				stop_on_failure = Integer.parseInt (value);
			if (stop_on_failure <= 0)
				{
				stop_on_failure = 0;
				if (! Dialog_Box.Confirm
						("Processing will not be stopped" + NL
						+"regardless of the number of sequential failures.",
						Sources_Table));
					continue;
				}
			The_Management.Stop_on_Failure (stop_on_failure);
			break;
			}
		catch (NumberFormatException exception)
			{Dialog_Box.Notice
				("Please enter a numeric value.", Sources_Table);}
		}
	}
catch (Remote_Management_Exception exception)
	{
	new Error_Report ("Management Protocol Failure",
		"Failed to set the sequential failures limit.",
		exception, this);
	}
}


private void Set_Stop_on_Failure
	(
	int		stop_on_failure
	)
{
if (stop_on_failure < 0)
	stop_on_failure = 0;
if (stop_on_failure != Stop_On_Failure)
	Stop_On_Failure_Label.setText
		("/" + (Stop_On_Failure = stop_on_failure) + " Sequential");
}


private void Reset_Sequential_Failures ()
{
try {The_Management.Reset_Sequential_Failures ();}
catch (Remote_Management_Exception exception)
	{
	new Error_Report ("Management Protocol Failure",
		"Failed to reset sequential failures.",
		exception, this);
	return;
	}
}


private void Set_Default_Value ()
{
try
	{
	String
		value = The_Management.Resolver_Default_Value ();
	if (value == null)
		value = "";
	value = JOptionPane.showInputDialog
		(Procedures_Table,
		"Default for unresolved values." + NL
		+ NL
		+"No entry means an error condition." + NL
		+"Enter '' for the empty string value.",
		value);
	if (value == null)
		return;
	if (value.length () == 0)
		value = null;
	else
	if (value.equals ("''") ||
		value.equals ("\"\""))
		value = "";
	The_Management.Resolver_Default_Value (value);
	}
catch (Remote_Management_Exception exception)
	{
	new Error_Report ("Management Protocol Failure",
		"Failed to set the default for unresolved values.",
		exception, this);
	}
}


private void Set_Max_Sources ()
{
String
	value;
while ((value = JOptionPane.showInputDialog
			(Sources_Table,
			"Maximum number of source file records" + NL
			+"to be retained in the Sources table:",
			String.valueOf (Max_Sources_Rows)))
		!= null)
	{
	try
		{
		int
			max_sources_rows = Integer.parseInt (value);
		if (max_sources_rows < 1)
			max_sources_rows = 1;
		Max_Sources_Rows = max_sources_rows;
		break;
		}
	catch (NumberFormatException exception)
		{Dialog_Box.Notice
			("Please enter a numeric value.", Sources_Table);}
	}
}

/**	Signal the Conductor to quit.
<p>
	If the Conductor is in the running {@link
	Management#Processing_State() processing state} a confirmation
	dialog is presented before source processing is aborted.
<p>
	<b>Caution</b>: If the Conductor is polling for sources it is possible
	that it will acquire a source and begin processing it before the
	quit signal is received causing the processing of the source to be
	aborted.
*/
public void Quit_Conductor ()
{
if ((DEBUG & DEBUG_PROCESSING_ACTIONS) != 0)
	System.out.println
		(">>> Manager.Quit_Conductor");
if (Processing_State > 0)
	{
	int
		check = Dialog_Box.Check
			("Abort the active Conductor?",
			this);
	if (check <= 0)
		{
		if (check < 0)
			Quit_Pending = true;
		return;
		}
	}
Quit ();	//	This should generate an Exiting Processing Change.
if ((DEBUG & DEBUG_PROCESSING_ACTIONS) != 0)
	System.out.println
		("<<< Manager.Quit_Conductor");
}

/**	Close and dispose of this Manager.
<p>
	If the Conductor Management interface is a Conductor it will be told
	to {@link #Quit_Conductor() quit}. Otherwise the Conductor {@link
	Management#Remove_Log_Writer(Writer) Log stream} and {@link
	Management#Remove_Processing_Listener(Processing_Listener)
	processing event} listeners are unregistered and the Conductor is
	left in its current state.
*/
public void Close ()
{
if (Closing)
	return;
if ((DEBUG & DEBUG_GUI_ACTIONS) != 0)
	System.out.println
		(">>> Manager.Close" + NL
		+"    Local_Management: " + Local_Management (The_Management));
if (Local_Management (The_Management))
	Quit_Conductor ();
else
	{
	Closing = true;
	Disable ();

	//	Dispose of any sub-windows.
	if (Configuration_View != null)
		{
		Configuration_View.setVisible (false);
		Configuration_View.dispose ();
		}
	if (Log_Monitor_Window != null)
		{
		Log_Monitor_Window.setVisible (false);
		Log_Monitor_Window.dispose ();
		}
	
	//	Dispose of this window.
	if ((DEBUG & DEBUG_GUI_ACTIONS) != 0)
		System.out.println
			("    Window dispose");
	this.dispose ();
	}
if ((DEBUG & DEBUG_GUI_ACTIONS) != 0)
	System.out.println
		("<<< Manager.Close");
}


private void Quit ()
{
if ((DEBUG & DEBUG_PROCESSING_ACTIONS) != 0)
	System.out.println
		(">-< Quit the Conductor");
try {The_Management.Quit ();}
catch (Remote_Management_Exception exception)
	{
	new Error_Report ("Management Protocol Failure",
		"Failed to tell the Conductor to quit.",
		exception, this);
	return;
	}

//	Disable Conductor controls.
Disable ();
}


public void Disable ()
{
try
	{
	The_Management.Remove_Log_Writer (Log_Monitor.Writer ());
	The_Management.Remove_Processing_Listener (this);
	}
catch (Exception exception) {}

Options_Menu.setEnabled (false);
Quit_Menu_Item.setEnabled (false);
Status_Blinker.stop ();
Processing_Button.setText (DONE_LABEL);
Processing_Button.setEnabled (false);
}

/*------------------------------------------------------------------------------
	Log Monitor
*/
private void Show_Log_Monitor ()
{
if (! Log_Monitor_Window.isVisible ())
	{
	if (! Log_While_Closed_Checkbox.isSelected ())
		{
		Log_Monitor.Clear ();
		try {The_Management.Add_Log_Writer (Log_Monitor.Writer ());}
		catch (Remote_Management_Exception exception)
			{
			new Error_Report ("Management Protocol Failure",
				"Failed to register this Manager" + NL
				+"as a Conductor log stream listener.",
				exception, this);
			return;
			}
		}
	LOG_MONITOR_LOCATOR.Relocate (Log_Monitor_Window, this);
	Log_Monitor_Window.setVisible (true);
	}
Log_Monitor_Window.toFront ();
}

private static final View_Locator
	LOG_MONITOR_LOCATOR	= new View_Locator ()
		.Offsets (0, 0)
		.Vertical   (View_Locator.BOTTOM	| View_Locator.OUTWARD)
		.Horizontal (View_Locator.LEFT		| View_Locator.INWARD);


private void Close_Log_Monitor ()
{
if (Log_Monitor_Window.isVisible ())
	{
	if (! Log_While_Closed_Checkbox.isSelected ())
		{
		try {The_Management.Remove_Log_Writer (Log_Monitor.Writer ());}
		catch (Exception exception) {}
		}
	Log_Monitor_Window.setVisible (false);
	}
}


private void Log_While_Closed
	(
	boolean	enable
	)
{
if (enable)
	{
	try {The_Management.Add_Log_Writer (Log_Monitor.Writer ());}
	catch (Remote_Management_Exception exception)
		{
		new Error_Report ("Management Protocol Failure",
			"Failed to register this Manager" + NL
			+"as a Conductor log stream listener.",
			exception, this);
		return;
		}
	}
else if (! Log_Monitor_Window.isVisible ())
	{
	try {The_Management.Remove_Log_Writer (Log_Monitor.Writer ());}
	catch (Exception exception) {}
	}
}

/*==============================================================================
	Processing_Listener
*/
public void Processing_Event_Occurred
	(
	Processing_Event	event
	)
{
synchronized (Processing_Event_Queue)
	{Processing_Event_Queue.add (event);}
SwingUtilities.invokeLater (new Runnable ()
{public void run () {Processing_Event_Disposition ();}});
}


private Vector
	Processing_Event_Queue			= new Vector ();


private void Processing_Event_Disposition ()
{
if ((DEBUG & DEBUG_PROCESSING_LISTENER) != 0)
	System.out.println
		(">>> Manager.Processing_Event_Disposition");

//	Pull the next Processing_Event from the front of the queue.
Processing_Event
	event;
synchronized (Processing_Event_Queue)
	{
	/*
		>>> WARNING <<< No check is being done for an empty queue because
		Processing_Event_Disposition is run from the event queue after a
		Processing_Event_Occurred enqueues the delivered event, and only
		one event is removed from the queue each time
		Processing_Event_Disposition is run. If this one-to-one
		relationship between Processing_Event_Occurred and
		Processing_Event_Disposition is ever changed an empty queue check
		must be added.
	*/
	event = (Processing_Event)Processing_Event_Queue.remove (0);
	}

if ((DEBUG & DEBUG_PROCESSING_LISTENER) != 0)
	System.out.print (event.Changes);

if (event.Changes.Procedures_Changed ())
	Refresh_Procedure_Records ();

Configuration
	configuration = event.Changes.Configuration ();
if (configuration != null)
	Report_Configuration_Change (configuration);

Vector
	record = event.Changes.Source_Record ();
if (record != null)
	Report_Source_Record (record);

record = event.Changes.Procedure_Record ();
if (record != null)
	Report_Procedure_Record (record);

//	Report processing state before reporting sequential failures.
if (event.Changes.Processing_State () != 0)
	Report_Processing_State (event.Changes.Processing_State ());
else
if (event.Changes.Exiting ())
	Report_Processing_State (EXITING);

if (event.Changes.Sequential_Failures () >= 0)
	Report_Sequential_Failures
		(event.Changes.Sequential_Failures ());

Report_Error_Condition (event.Changes.Error_Condition ());

if ((DEBUG & DEBUG_PROCESSING_LISTENER) != 0)
	System.out.println
		("<<< Manager.Processing_Event_Disposition");
}

/*==============================================================================
	Helpers
*/
/**	Loads the application icons used by the GUI.
*/
private void Load_Icons ()
{
if (Success_Icon == null)
	{
	if ((DEBUG & DEBUG_ICONS) != 0)
		System.out.println
			(">>> Manager.Load_Icons");
	Icons.Directory ("Icons", this.getClass ());
	if ((DEBUG & DEBUG_ICONS) != 0)
		System.out.println
			("    Loading icons from " + Icons.Directory ());
	Success_Icon = Icons.Load_Icon (SUCCESS_ICON_NAME);
	Failure_Icon = Icons.Load_Icon (FAILURE_ICON_NAME);
	Warning_Icon = Icons.Load_Icon (WARNING_ICON_NAME);
	if ((DEBUG & DEBUG_ICONS) != 0)
		System.out.println
			("    The Icons were "
				+ ((Success_Icon == null) ? "not " : "") + "loaded" + NL
			+"<<< Manager.Load_Icons");
	}
}

/**	Test for a Management interface local to a Conductor.
<p>
	A local Management interface is implemented by the Conductor class;
	if the management is an instance of the Conductor class it is local.
<p>
	@param	management	A Management interface instance to test.
	@return	true if the management is a local Conductor instance; false
		otherwise.
*/
public static boolean Local_Management
	(
	Management	management
	)
{return management instanceof Conductor;}


}
