/**
* @license Apache-2.0
*
* Copyright (c) 2018 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

'use strict';

// MODULES //

var isNonNegativeInteger = require( '@stdlib/math/base/assert/is-nonnegative-integer' );
var factorialln = require( '@stdlib/math/base/special/factorialln' );
var isnan = require( '@stdlib/math/base/assert/is-nan' );
var ln = require( '@stdlib/math/base/special/ln' );
var NINF = require( '@stdlib/constants/float64/ninf' );
var PINF = require( '@stdlib/constants/float64/pinf' );


// MAIN //

/**
* Evaluates the natural logarithm of the probability mass function (PMF) for a Poisson distribution with mean parameter `lambda` at a value `x`.
*
* @param {number} x - input value
* @param {NonNegativeNumber} lambda - mean parameter
* @returns {number} evaluated logPMF
*
* @example
* var y = logpmf( 4.0, 3.0 );
* // returns ~-1.784
*
* @example
* var y = logpmf( 1.0, 3.0 );
* // returns ~-1.901
*
* @example
* var y = logpmf( -1.0, 2.0 );
* // returns -Infinity
*
* @example
* var y = logpmf( 0.0, NaN );
* // returns NaN
*
* @example
* var y = logpmf( NaN, 0.5 );
* // returns NaN
*
* @example
* // Invalid mean parameter:
* var y = logpmf( 2.0, -0.5 );
* // returns NaN
*/
function logpmf( x, lambda ) {
	if ( isnan( x ) || isnan( lambda ) || lambda < 0.0 ) {
		return NaN;
	}
	if ( lambda === 0.0 ) {
		return ( x === 0.0 ) ? 0.0 : NINF;
	}
	if ( isNonNegativeInteger( x ) && x !== PINF ) {
		return ( x * ln( lambda ) ) - lambda - factorialln( x );
	}
	return NINF;
}


// EXPORTS //

module.exports = logpmf;
