/*
 * @(#)BytecodeCountMeasureUTest.java
 *
 * Copyright (C) 2003 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the 
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in 
 *  all copies or substantial portions of the Software. 
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.
 */

package net.sourceforge.groboutils.codecoverage.v2.module;

import java.io.IOException;

import junit.framework.Test;
import junit.framework.TestCase;
import net.sourceforge.groboutils.autodoc.v1.AutoDoc;
import net.sourceforge.groboutils.codecoverage.v2.BytecodeLoaderUtil;
import net.sourceforge.groboutils.codecoverage.v2.CCCreatorUtil;
import net.sourceforge.groboutils.codecoverage.v2.IAnalysisModule;
import net.sourceforge.groboutils.codecoverage.v2.IAnalysisModuleUTestI;
import net.sourceforge.groboutils.codecoverage.v2.IMethodCode;
import net.sourceforge.groboutils.codecoverage.v2.compiler.ModifiedClass;
import net.sourceforge.groboutils.codecoverage.v2.compiler.ModifiedMethod;
import net.sourceforge.groboutils.codecoverage.v2.logger.TestLogger;
import net.sourceforge.groboutils.junit.v1.iftc.CxFactory;
import net.sourceforge.groboutils.junit.v1.iftc.InterfaceTestSuite;


/**
 * Tests the BytecodeCountMeasure class.
 *
 * @author    Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @version   $Date: 2004/04/15 05:48:28 $
 * @since     January 26, 2003
 */
public class BytecodeCountMeasureUTest extends TestCase
{
    //-------------------------------------------------------------------------
    // Standard JUnit Class-specific declarations
    
    private static final Class THIS_CLASS = BytecodeCountMeasureUTest.class;
    private static final AutoDoc DOC = new AutoDoc( THIS_CLASS );
    
    public BytecodeCountMeasureUTest( String name )
    {
        super( name );
    }


    //-------------------------------------------------------------------------
    // Tests
    
    
    public static class MyTestClass
    {
        public static void main( String[] args )
        {
            int i = 0;
            i += 10;
            System.out.println( ""+i );
        }
    }
    
    
    public void testAnalyze1() throws Exception
    {
        BytecodeCountMeasure bcm = new BytecodeCountMeasure();
        ModifiedClass mc = CCCreatorUtil.createModifiedClass(
            TestLogger.createPCL(), MyTestClass.class );
        ModifiedMethod mm = CCCreatorUtil.getMainModifiedMethod( mc );
        IMethodCode imc = createMethodCode( MyTestClass.class, mm, bcm );
        int expectedCount = mm.getInstructionList().getInstructionCount();
        
        // populate the method with marks
        bcm.analyze( imc );
        
        // Generate the class from the modified bytecode, and run it.
        Class c = BytecodeLoaderUtil.loadClassFromBytecode(
            mc.getClassName(), mc.getModifiedClass() );
        TestLogger.reset();
        BytecodeLoaderUtil.runMain( c );
        assertEquals(
            "Did not mark every bytecode in method.",
            expectedCount,
            TestLogger.size() );
    }
    
    
    
    //-------------------------------------------------------------------------
    // Helpers
    
    
    protected IMethodCode createMethodCode( Class c, ModifiedMethod mm,
            IAnalysisModule am )
    {
        return CCCreatorUtil.createIMethodCode( c, mm,
            CCCreatorUtil.createAnalysisModuleSet(
                new IAnalysisModule[] { am } ),
            0 );
    }
    
    
    //-------------------------------------------------------------------------
    // Standard JUnit declarations
    
    
    public static Test suite()
    {
        InterfaceTestSuite suite = IAnalysisModuleUTestI.suite();
        suite.addTestSuite( THIS_CLASS );
        suite.addFactory( new CxFactory( "A" ) {
            public Object createImplObject() throws IOException {
                return new BytecodeCountMeasure();
            }
        } );
        
        return suite;
    }
    
    public static void main( String[] args )
    {
        String[] name = { THIS_CLASS.getName() };
        
        // junit.textui.TestRunner.main( name );
        // junit.swingui.TestRunner.main( name );
        
        junit.textui.TestRunner.main( name );
    }
    
    
    /**
     * 
     * @exception Exception thrown under any exceptional condition.
     */
    protected void setUp() throws Exception
    {
        super.setUp();

       
        // set ourself up
    }
    
    
    /**
     * 
     * @exception Exception thrown under any exceptional condition.
     */
    protected void tearDown() throws Exception
    {
        // tear ourself down
        
        
        super.tearDown();
    }
}

