//  Copyright (c) 2013, Facebook, Inc.  All rights reserved.
//  This source code is licensed under the BSD-style license found in the
//  LICENSE file in the root directory of this source tree. An additional grant
//  of patent rights can be found in the PATENTS file in the same directory.
//  This source code is also licensed under the GPLv2 license found in the
//  COPYING file in the root directory of this source tree.
#pragma once

#include <limits>
#include <list>

#include "util/mutexlock.h"

namespace rocksdb {

//
// Simple synchronized queue implementation with the option of
// bounding the queue
//
// On overflow, the elements will be discarded
//
template <class T>
class BoundedQueue {
 public:
  explicit BoundedQueue(
      const size_t max_size = std::numeric_limits<size_t>::max())
      : cond_empty_(&lock_), max_size_(max_size) {}

  virtual ~BoundedQueue() {}

  void Push(T&& t) {
    MutexLock _(&lock_);
    if (max_size_ != std::numeric_limits<size_t>::max() &&
        size_ + t.Size() >= max_size_) {
      // overflow
      return;
    }

    size_ += t.Size();
    q_.push_back(std::move(t));
    cond_empty_.SignalAll();
  }

  T Pop() {
    MutexLock _(&lock_);
    while (q_.empty()) {
      cond_empty_.Wait();
    }

    T t = std::move(q_.front());
    size_ -= t.Size();
    q_.pop_front();
    return std::move(t);
  }

  size_t Size() const {
    MutexLock _(&lock_);
    return size_;
  }

 private:
  mutable port::Mutex lock_;
  port::CondVar cond_empty_;
  std::list<T> q_;
  size_t size_ = 0;
  const size_t max_size_;
};

}  // namespace rocksdb
