/*
   +----------------------------------------------------------------------+
   | HipHop for PHP                                                       |
   +----------------------------------------------------------------------+
   | Copyright (c) 2010- Facebook, Inc. (http://www.facebook.com)         |
   +----------------------------------------------------------------------+
   | This source file is subject to version 3.01 of the PHP license,      |
   | that is bundled with this package in the file LICENSE, and is        |
   | available through the world-wide-web at the following url:           |
   | http://www.php.net/license/3_01.txt                                  |
   | If you did not receive a copy of the PHP license and are unable to   |
   | obtain it through the world-wide-web, please send a note to          |
   | license@php.net so we can mail you a copy immediately.               |
   +----------------------------------------------------------------------+
*/

#include "hphp/php7/compiler.h"

#include "hphp/php7/analysis.h"
#include "hphp/php7/util.h"
#include "hphp/util/match.h"

#include <folly/Format.h>

#include <iostream>

namespace HPHP { namespace php7 {

namespace {

// CGS holds the Compile Global State for the compilation.
struct CGS {
  Unit* unit;
} g_cgs;

void buildFunction(Function* func, const zend_ast* ast);

CFG compileZvalLiteral(const zval* ast);
CFG compileConstant(const zend_ast* ast);
CFG compileVar(const zend_ast* ast, Destination destination);
CFG compileAssignment(const zend_ast* ast);
CFG compileBind(const zend_ast* ast);
CFG compileAssignOp(const zend_ast* ast);
CFG compileFunctionCall(const zend_ast* ast);
CFG compileMethodCall(const zend_ast_list* params);
CFG compileNew(const zend_ast* ast);
CFG compileCall(const zend_ast_list* params);
CFG compileArray(const zend_ast* ast);

CFG compileGlobalDeclaration(const zend_ast* ast);
CFG compileCatch(Function* func, const zend_ast_list* catches);
CFG compileTry(Function* func, const zend_ast* ast);
CFG compileIf(Function* func, const zend_ast* ast);
CFG compileWhile(Function* func,
    const zend_ast* cond,
    const zend_ast* body,
    bool bodyFirst);
CFG compileFor(Function* func, const zend_ast* ast);

Bytecode opForBinaryOp(const zend_ast* op);
IncDecOp getIncDecOpForNode(zend_ast_kind kind);
SetOpOp getSetOpOp(zend_ast_attr attr);

std::unique_ptr<Lvalue> getLvalue(const zend_ast* ast);

CFG compileUnaryOp(const zend_ast* op);
CFG compileIncDec(const zend_ast* op);
CFG compileIncludeEval(const zend_ast* op);

CFG fixFlavor(Destination dest, Flavor actual);

// Zend reuses the visibility bits to tag classes with additional info.
// pppok should be set to true when visibility attributes are desired.
Attr translateAttr(uint32_t flags, bool pppok);

void compileClassStatement(Class* cls, const zend_ast* ast);
void compileMethod(Class* cls, const zend_ast* ast);
void compileProp(Class* cls, const zend_ast* ast);

bool serializeInitilizer(std::string& out, const zend_ast* ast);

} // namespace

using namespace bc;

void compileProgram(Unit* unit, const zend_ast* ast) {
  assert(ast->kind == ZEND_AST_STMT_LIST);

  auto pseudomain = unit->getPseudomain();
  g_cgs.unit = unit;

  pseudomain->cfg = compileStatement(pseudomain, ast)
    .then(Int{1})
    .thenReturn(Flavor::Cell)
    .makeExitsReal()
    .tagSrcLoc(zend_ast_get_lineno(ast))
    .inRegion(std::make_unique<Region>(Region::Kind::Entry));
  simplifyCFG(pseudomain->cfg);
}

void compileFunction(Unit* unit, const zend_ast* ast) {
  buildFunction(unit->makeFunction(), ast);
}

CFG compileClass(Unit* unit, const zend_ast* ast) {
  auto decl = zend_ast_get_decl(ast);
  auto parent = decl->child[0];
  auto implements = decl->child[1];
  auto statements = zend_ast_get_list(decl->child[2]);

  auto cls = unit->makeClass();

  auto name = decl->name ? ZSTR_VAL(decl->name)
                         : folly::sformat("class@anonymous$;{}", cls->index);
  cls->name = name;

  cls->attr = translateAttr(decl->flags, false);

  if (parent) {
    cls->parentName = zval_to_string(zend_ast_get_zval(parent));
  }
  if (implements) {
    assertx(implements->kind == ZEND_AST_NAME_LIST);
    auto list = zend_ast_get_list(implements);
    for (uint32_t i = 0; i < list->children; i++) {
      auto item = list->child[i];

      cls->implements.emplace_back(zval_to_string(zend_ast_get_zval(item)));
    }
  }

  for (uint32_t i = 0; i < statements->children; i++) {
    compileClassStatement(cls, statements->child[i]);
  }

  uint32_t lineno = zend_ast_get_lineno(ast);
  /* Force the creation of a ctor if we don't have one yet */
  cls->getConstructor(lineno);

  cls->buildPropInit(lineno);

  return CFG(DefCls{cls->index})
    .tagSrcLoc(lineno);
}

void panic(const std::string& msg) {
  throw CompilerException(folly::sformat("panic: {}", msg));
}

namespace {

void buildFunction(Function* func, const zend_ast* ast) {
  auto decl = zend_ast_get_decl(ast);
  auto name = ZSTR_VAL(decl->name);
  auto params = zend_ast_get_list(decl->child[0]);
  auto body = decl->child[2];

  func->name = name;
  func->startLineno = decl->start_lineno;
  func->endLineno = decl->end_lineno;

  func->attr = translateAttr(decl->flags, true);

  func->params.reserve(params->children);
  for (uint32_t i = 0; i < params->children; i++) {
    auto param = params->child[i];
    auto param_name = zval_to_string(zend_ast_get_zval(param->child[1]));
    auto param_default = param->child[2];
    auto param_attrs = param->attr;

    if (param_default) {
      panic("default parameter values not supported");
    }

    func->params.push_back({
      param_name,
      static_cast<bool>(param_attrs & ZEND_PARAM_REF)
    });
  }

  func->cfg = compileStatement(func, body)
    .then(Null{})
    .thenReturn(func->returnsByReference()
      ? Flavor::Ref
      : Flavor::Cell)
    .makeExitsReal()
    .tagSrcLoc(zend_ast_get_lineno(ast))
    .inRegion(std::make_unique<Region>(Region::Kind::Entry));
  simplifyCFG(func->cfg);
}

CFG compileZvalLiteral(const zval* zv) {
  switch (Z_TYPE_P(zv)) {
    case IS_LONG:
      return { Int{Z_LVAL_P(zv)} };
    case IS_NULL:
      return { Null{} };
    case IS_FALSE:
      return { False{} };
    case IS_TRUE:
      return { True{} };
    case IS_DOUBLE:
      return { Double{Z_DVAL_P(zv)} };
    case IS_STRING:
      return { String{Z_STRVAL_P(zv)} };
    default:
      panic("unsupported literal");
  }
}

CFG compileConstant(const zend_ast* ast) {
  auto name = ast->child[0];
  const char* str = Z_STRVAL_P(zend_ast_get_zval(name));
  if (name->attr & ZEND_NAME_NOT_FQ) {
    if (strcasecmp(str, "true") == 0) {
      return { True{} };
    } else if (strcasecmp(str, "false") == 0) {
      return { False{} };
    } else if (strcasecmp(str, "null") == 0) {
      return { Null{} };
    } else if (strcasecmp(str, "__COMPILER_FRONTEND__") == 0) {
      return { String{"php7/hhvm"} };
    } else {
      return { Cns{str} };
    }
  } else {
    panic("unknown constant");
  }
}


std::unique_ptr<Lvalue> getLvalue(const zend_ast* ast){
  if (auto ret = Lvalue::getLvalue(ast)) {
    return ret;
  }

  panic("can't make lvalue");
}

CFG compileVar(const zend_ast* ast, Destination dest) {
  switch (dest.flavor) {
    case Drop:
      return getLvalue(ast)->getC()
        .then(PopC{});
    case Cell:
      return getLvalue(ast)->getC();
    case Ref:
      return getLvalue(ast)->getV();
    case FuncParam:
      return getLvalue(ast)->getF(dest.slot);
    case Return:
      panic("Can't get a local with R flavor");
  }
  panic("bad destination flavor");
}

Bytecode opForBinaryOp(const zend_ast* ast) {
  // NB: bizarrely, greater-than (>,>=) have their own AST type
  // and there is no ZEND_IS_GREATER since it doesn't correspond to a VM
  // instruction
  if (ast->kind == ZEND_AST_GREATER) {
    return Gt{};
  } else if (ast->kind == ZEND_AST_GREATER_EQUAL) {
    return Gte{};
  }

  switch (ast->attr) {
    case ZEND_ADD: return Add{};
    case ZEND_SUB: return Sub{};
    case ZEND_MUL: return Mul{};
    case ZEND_DIV: return Div{};
    case ZEND_POW: return Pow{};
    case ZEND_MOD: return Mod{};
    case ZEND_SL: return Shl{};
    case ZEND_SR: return Shr{};
    case ZEND_BW_OR: return BitOr{};
    case ZEND_BW_AND: return BitAnd{};
    case ZEND_BW_XOR: return BitXor{};
    case ZEND_CONCAT: return Concat{};

    case ZEND_IS_IDENTICAL: return Same{};
    case ZEND_IS_NOT_IDENTICAL: return NSame{};
    case ZEND_IS_EQUAL: return Eq{};
    case ZEND_IS_NOT_EQUAL: return Neq{};
    case ZEND_IS_SMALLER: return Lt{};
    case ZEND_IS_SMALLER_OR_EQUAL: return Lte{};
    case ZEND_SPACESHIP: return Cmp{};
    default:
      panic("unknown binop");
  }
}

CFG compileUnaryOp(const zend_ast* ast) {
  switch (ast->kind) {
    case ZEND_AST_UNARY_MINUS:
      return CFG()
        .then(Int{0})
        .then(compileExpression(ast->child[0], Flavor::Cell))
        .then(Sub{});
    case ZEND_AST_UNARY_PLUS:
      return CFG()
        .then(Int{0})
        .then(compileExpression(ast->child[0], Flavor::Cell))
        .then(Add{});
    case ZEND_AST_UNARY_OP:
      return CFG()
        .then(compileExpression(ast->child[0], Flavor::Cell))
        .then(ast->attr == ZEND_BOOL_NOT
            ? Bytecode{Not{}}
            : Bytecode(BitNot{}));
  }
  panic("unknown unop");
}

IncDecOp getIncDecOpForNode(zend_ast_kind kind) {
  switch (kind) {
    case ZEND_AST_PRE_INC:
      return IncDecOp::PreInc;
    case ZEND_AST_PRE_DEC:
      return IncDecOp::PreDec;
    case ZEND_AST_POST_INC:
      return IncDecOp::PostInc;
    case ZEND_AST_POST_DEC:
      return IncDecOp::PostDec;
    default:
      panic("not a inc/dec node");
  }
}

SetOpOp getSetOpOp(zend_ast_attr attr) {
  switch (attr) {
    case ZEND_ASSIGN_ADD:
      return SetOpOp::PlusEqual;
    case ZEND_ASSIGN_SUB:
      return SetOpOp::MinusEqual;
    case ZEND_ASSIGN_MUL:
      return SetOpOp::MulEqual;
    case ZEND_ASSIGN_POW:
      return SetOpOp::PowEqual;
    case ZEND_ASSIGN_DIV:
      return SetOpOp::DivEqual;
    case ZEND_ASSIGN_CONCAT:
      return SetOpOp::ConcatEqual;
    case ZEND_ASSIGN_MOD:
      return SetOpOp::ModEqual;
    case ZEND_ASSIGN_BW_AND:
      return SetOpOp::AndEqual;
    case ZEND_ASSIGN_BW_OR:
      return SetOpOp::OrEqual;
    case ZEND_ASSIGN_BW_XOR:
      return SetOpOp::XorEqual;
    case ZEND_ASSIGN_SL:
      return SetOpOp::SlEqual;
    case ZEND_ASSIGN_SR:
      return SetOpOp::SrEqual;
    default:
      panic("unsupported set-op");
  }
}

CFG compileIncDec(const zend_ast* ast) {
  auto op = getIncDecOpForNode(ast->kind);
  auto var = ast->child[0];

  return getLvalue(var)->incDec(op);
}

CFG compileAssignment(const zend_ast* ast) {
  auto lhs = ast->child[0];
  auto rhs = ast->child[1];

  return getLvalue(lhs)->assign(rhs);
}

CFG compileBind(const zend_ast* ast) {
  auto lhs = ast->child[0];
  auto rhs = ast->child[1];

  return getLvalue(lhs)->bind(rhs);
}

CFG compileAssignOp(const zend_ast* ast) {
  auto rhs = ast->child[1];
  auto op = getSetOpOp(ast->attr);
  auto lhs = ast->child[0];

  return getLvalue(lhs)->assignOp(op, rhs);
}

CFG compileFunctionCall(const zend_ast* ast) {
  auto callee = ast->child[0];
  auto params = zend_ast_get_list(ast->child[1]);

  auto zv = callee->kind == ZEND_AST_ZVAL
      ? zend_ast_get_zval(callee)
      : nullptr;

  if (zv && Z_TYPE_P(zv) == IS_STRING) {
    return CFG(FPushFuncD{
      params->children,
      Z_STRVAL_P(zend_ast_get_zval(callee))
    }).then(compileCall(params));
  } else {
    return compileExpression(callee, Flavor::Cell)
      .then(FPushFunc{
        params->children
      })
      .then(compileCall(params));
  }
}

CFG compileMethodCall(const zend_ast* ast) {
  auto target = ast->child[0];
  auto method = ast->child[1];
  auto params = zend_ast_get_list(ast->child[2]);

  auto zv = method->kind == ZEND_AST_ZVAL
      ? zend_ast_get_zval(method)
      : nullptr;

  auto targetCfg = compileExpression(target, Flavor::Cell);

  if (zv && Z_TYPE_P(zv) == IS_STRING) {
    return targetCfg
      .then(FPushObjMethodD{
        params->children,
        Z_STRVAL_P(zv),
        ObjMethodOp::NullThrows
      })
      .then(compileCall(params));
  } else {
    return compileExpression(method, Flavor::Cell)
      .then(std::move(targetCfg))
      .then(FPushObjMethod{
        params->children,
        ObjMethodOp::NullThrows
      })
      .then(compileCall(params));
  }
}

CFG compileStaticCall(const zend_ast* ast) {
  auto cls = ast->child[0];
  auto method = ast->child[1];
  auto params = zend_ast_get_list(ast->child[2]);

  ClassrefSlot slot;
  folly::Optional<SpecialClsRef> special;
  CFG cfg = compileClassref(cls, slot, &special)
    .then(compileExpression(method, Flavor::Cell));

  if (special) {
    return cfg
      .then(FPushClsMethodS{
        params->children,
        *special
      })
      .then(compileCall(params));
  }
  return cfg
    .then(FPushClsMethod{
      params->children,
      slot
    })
    .then(compileCall(params));
}

CFG compileNew(const zend_ast* ast) {
  auto cls = ast->child[0];
  auto params = zend_ast_get_list(ast->child[1]);

  auto zv = cls->kind == ZEND_AST_ZVAL
      ? zend_ast_get_zval(cls)
      : nullptr;

  // push the FPI record
  if (zv && Z_TYPE_P(zv) == IS_STRING) {
    return CFG(FPushCtorD{
      params->children,
      Z_STRVAL_P(zv)
    })
      .then(compileCall(params))
      .then(PopR{});
  } else if (cls->kind == ZEND_AST_CLASS) {
    // Anonymous class
    auto index = g_cgs.unit->nextClassId();

    return compileClass(g_cgs.unit, cls)
      .then(FPushCtorI{params->children, index})
      .then(compileCall(params))
      .then(PopR{});
  } else {
    panic("new with variable classref");
  }
}

CFG compileConditional(const zend_ast* ast, Destination dest) {
  CFG cfg;
  auto end = cfg.makeBlock();

  auto condition = ast->child[0];
  auto iftrue = ast->child[1];
  auto iffalse = ast->child[2];

  cfg
    .then(compileExpression(condition, Flavor::Cell))
    .branchNZ(
      compileExpression(iftrue, dest)
        .thenJmp(end)
    );
  // false condition.
  return cfg
    .then(compileExpression(iffalse, dest))
    .thenJmp(end)
    .continueFrom(end);
}

CFG compileCall(const zend_ast_list* params) {
  CFG call;
  for (uint32_t i = 0; i < params->children; i++) {
    call.then(compileExpression(params->child[i], Destination::Param(i)));
  }

  return call.then(FCall{params->children});
}

CFG compileArray(const zend_ast* ast) {
  // TODO: handle static array literals, too
  auto list = zend_ast_get_list(ast);

  // NB: array() and list() share the syntax node
  // ZEND_AST_ARRAY--hence this odd check
  if (list->attr == ZEND_ARRAY_SYNTAX_LIST) {
    throw LanguageException("Cannot use list() as a standalone expression");
  }

  CFG cfg(NewArray{list->children});
  for (uint32_t i = 0; i < list->children; i++) {
    auto item = list->child[i];

    if (!item) {
      throw LanguageException("Cannot use empty array elements in arrays");
    }

    // NB: there's actually no constant for this in the parser; it's currently
    // just set to 1 for ref-y items
    bool ref = item->attr != 0;
    auto flavor = ref ? Flavor::Ref : Flavor::Cell;

    // if the second child is set, this is a name-value pair
    if (item->child[1]) {
      auto key = item->child[1];
      auto val = item->child[0];
      cfg.then(compileExpression(key, Flavor::Cell))
         .then(compileExpression(val, flavor));
      if (ref) {
        cfg.then(AddElemV{});
      } else {
        cfg.then(AddElemC{});
      }
    } else {
      auto val = item->child[0];
      cfg.then(compileExpression(val, flavor));
      if (ref) {
        cfg.then(AddNewElemV{});
      } else {
        cfg.then(AddNewElemC{});
      }
    }
  }
  return cfg;
}

} // namespace

CFG compileClassref(const zend_ast* ast, ClassrefSlot slot,
                    folly::Optional<SpecialClsRef>* special) {
  auto clsString = ast->kind == ZEND_AST_ZVAL
      ? Z_STRVAL_P(zend_ast_get_zval(ast))
      : nullptr;

  if (clsString) {
    if (0 == strcasecmp(clsString, "self")) {
      if (special) {
        *special = SpecialClsRef::Self;
        return {};
      }
      return { Self{slot} };
    } else if (0 == strcasecmp(clsString, "parent")) {
      if (special) {
        *special = SpecialClsRef::Parent;
        return {};
      }
      return { Parent{slot} };
    } else if (0 == strcasecmp(clsString, "static")) {
      if (special) {
        *special = SpecialClsRef::Static;
        return {};
      }
      return { LateBoundCls{slot} };
    }
  }

  return compileExpression(ast, Flavor::Cell).then(ClsRefGetC{slot});
}

CFG compileExpression(const zend_ast* ast, Destination dest) {
  return [&]() {
    switch (ast->kind) {
      case ZEND_AST_ZVAL:
        return compileZvalLiteral(zend_ast_get_zval(ast))
          .then(fixFlavor(dest, Flavor::Cell));
      case ZEND_AST_CONST:
        return compileConstant(ast)
          .then(fixFlavor(dest, Flavor::Cell));
      case ZEND_AST_UNARY_MINUS:
      case ZEND_AST_UNARY_PLUS:
      case ZEND_AST_UNARY_OP:
        return compileUnaryOp(ast)
          .then(fixFlavor(dest, Flavor::Cell));
      case ZEND_AST_BINARY_OP:
      case ZEND_AST_GREATER:
      case ZEND_AST_GREATER_EQUAL:
        return CFG()
          .then(compileExpression(ast->child[0], Flavor::Cell))
          .then(compileExpression(ast->child[1], Flavor::Cell))
          .then(opForBinaryOp(ast))
          .then(fixFlavor(dest, Flavor::Cell));
      case ZEND_AST_POST_INC:
      case ZEND_AST_POST_DEC:
      case ZEND_AST_PRE_INC:
      case ZEND_AST_PRE_DEC:
        return compileIncDec(ast)
          .then(fixFlavor(dest, Flavor::Cell));
      case ZEND_AST_VAR:
      case ZEND_AST_DIM:
      case ZEND_AST_PROP:
      case ZEND_AST_STATIC_PROP:
        return compileVar(ast, dest);
      case ZEND_AST_ASSIGN:
        return compileAssignment(ast)
          .then(fixFlavor(dest, Flavor::Cell));
      case ZEND_AST_ASSIGN_REF:
        return compileBind(ast)
          .then(fixFlavor(dest, Flavor::Ref));
        break;
      case ZEND_AST_ASSIGN_OP:
        return compileAssignOp(ast)
          .then(fixFlavor(dest, Flavor::Cell));
      case ZEND_AST_CALL:
        return compileFunctionCall(ast)
          .then(fixFlavor(dest, Flavor::Return));
      case ZEND_AST_METHOD_CALL:
        return compileMethodCall(ast)
          .then(fixFlavor(dest, Flavor::Return));
      case ZEND_AST_STATIC_CALL:
        return compileStaticCall(ast)
          .then(fixFlavor(dest, Flavor::Return));
      case ZEND_AST_NEW:
        return compileNew(ast)
          .then(fixFlavor(dest, Flavor::Cell));
      case ZEND_AST_ARRAY:
        return compileArray(ast)
          .then(fixFlavor(dest, Flavor::Cell));
      case ZEND_AST_INCLUDE_OR_EVAL:
        return compileIncludeEval(ast)
          .then(fixFlavor(dest, Flavor::Cell));
      case ZEND_AST_CONDITIONAL:
        return compileConditional(ast, dest);
      case ZEND_AST_INSTANCEOF:
        return CFG()
          .then(compileExpression(ast->child[0], Flavor::Cell))
          .then(compileExpression(ast->child[1], Flavor::Cell))
          .then(InstanceOf{})
          .then(fixFlavor(dest, Flavor::Cell));
      default:
        panic("unsupported expression");
    }
  }().tagSrcLoc(zend_ast_get_lineno(ast));
}

namespace {

CFG fixFlavor(Destination dest, Flavor actual) {
  switch(dest.flavor) {
    case Drop:
      switch (actual) {
        case Drop:
          return {};
        case Cell:
          return { PopC{} };
        case Ref:
          return { PopV{} };
        case Return:
          return { PopR{} };
        case FuncParam:
          panic("Can't drop param");
      }
    case Ref:
      switch (actual) {
        case Cell:
          return { Box{} };
        case Ref:
          return {};
        case Return:
          return { BoxR{} };
        case Drop:
        case FuncParam:
          panic("Can't make ref");
      }
    case Cell:
      switch (actual) {
        case Cell:
          return {};
        case Ref:
          return { Unbox{} };
        case Return:
          return { UnboxR{} };
        case Drop:
        case FuncParam:
          panic("Can't make cell");
      }
    // nobody should be asking for a Return flavor unless they gosh darn know
    // the expression being compiled is a call >:/
    case Return:
      switch (actual) {
        case Return:
          return {};
        default:
          panic("can't coerce to return value");
      }
    case FuncParam:
      auto slot = dest.slot;
      switch (actual) {
        case Cell:
          return { FPassCE{slot, FPassHint::Any} };
        case Ref:
          return { FPassV{slot, FPassHint::Any} };
        case Return:
          return { FPassR{slot, FPassHint::Any} };
        case Drop:
        case FuncParam:
          panic("Can't make function param");
      }
  }
  panic("bad destination flavor");
}

} // namespace

CFG compileStatement(Function* func, const zend_ast* ast) {
  if (!ast) return CFG{};
  return [&]() {
    switch (ast->kind) {
      case ZEND_AST_STMT_LIST: {
        // just a block, so recur please :)
        CFG cfg;
        auto list = zend_ast_get_list(ast);
        for (uint32_t i = 0; i < list->children; i++) {
          if (!list->child[i]) {
            continue;
          }
          cfg.then(compileStatement(func, list->child[i]));
        }
        return cfg;
      }
      case ZEND_AST_ECHO:
        return CFG()
          .then(compileExpression(ast->child[0], Flavor::Cell))
          .then(Print{})
          .then(PopC{});
      case ZEND_AST_IF:
        return compileIf(func, ast);
      case ZEND_AST_WHILE:
        return compileWhile(func, ast->child[0], ast->child[1], false);
      case ZEND_AST_DO_WHILE:
        return compileWhile(func, ast->child[1], ast->child[0], true);
      case ZEND_AST_FOR:
        return compileFor(func, ast);
      case ZEND_AST_BREAK:
        return CFG().thenBreak();
      case ZEND_AST_GLOBAL:
        return compileGlobalDeclaration(ast);
      case ZEND_AST_CONTINUE:
        return CFG().thenContinue();
      case ZEND_AST_RETURN: {
        auto flavor = func->returnsByReference()
          ? Flavor::Ref
          : Flavor::Cell;
        auto expr = ast->child[0]
          ? compileExpression(ast->child[0], flavor)
          : CFG(Null{}).then(fixFlavor(flavor, Flavor::Cell));
        return expr.thenReturn(flavor);
      }
      case ZEND_AST_TRY:
        return compileTry(func, ast);
      case ZEND_AST_THROW:
        return compileExpression(ast->child[0], Flavor::Cell)
          .thenThrow();
      case ZEND_AST_FUNC_DECL:
        compileFunction(func->parent, ast);
        return CFG();
      case ZEND_AST_CLASS:
        return compileClass(func->parent, ast);
      default:
        return compileExpression(ast, Flavor::Drop);
    }
  }().tagSrcLoc(zend_ast_get_lineno(ast));
}

namespace {

CFG compileGlobalDeclaration(const zend_ast* ast) {
  auto var = ast->child[0];
  auto zv = var->child[0];
  auto name = zval_to_string(zend_ast_get_zval(zv));

  return {
    String{name},
    VGetG{},
    BindL{NamedLocal{name}},
    PopV{}
  };
}

CFG compileCatch(Function* func, const zend_ast_list* catches) {
  CFG cfg;
  auto end = cfg.makeBlock();

  for (uint32_t i = 0; i < catches->children; i++) {
    auto clause = catches->child[i];
    auto types = zend_ast_get_list(clause->child[0]);
    auto capture = zval_to_string(zend_ast_get_zval(clause->child[1]));
    auto body = clause->child[2];

    auto handler = CFG({
      SetL{NamedLocal{capture}},
      PopC{}
    }).then(compileStatement(func, body))
      .thenJmp(end);

    for (uint32_t j = 0; j < types->children; j++) {
      auto name = zval_to_string(zend_ast_get_zval(types->child[j]));
      cfg.then(Dup{})
        .then(InstanceOfD{name})
        .branchNZ("handler");
    }

    cfg.replace("handler", std::move(handler));
  }

  cfg.thenThrow();

  return cfg.continueFrom(end);
}


CFG compileTry(Function* func, const zend_ast* ast) {
  auto body = ast->child[0];
  auto catches = zend_ast_get_list(ast->child[1]);
  auto finally = ast->child[2];

  if (!finally && catches->children == 0) {
    throw LanguageException("Cannot use try without catch or finally");
  }

  CFG cfg = compileStatement(func, body);

  // don't bother adding a catch region if there's no catches
  if (catches->children > 0) {
    cfg.addExnHandler(compileCatch(func, catches));
  }

  if (finally) {
    cfg.addFinallyGuard(compileStatement(func, finally));
  }

  return cfg;
}

CFG compileIf(Function* func, const zend_ast* ast) {
  auto list = zend_ast_get_list(ast);

  CFG cfg;
  auto end = cfg.makeBlock();

  for (uint32_t i = 0; i <list->children; i++) {
    auto elem = list->child[i];
    auto condition = elem->child[0];
    auto contents = elem->child[1];
    if (!condition) {
      // if no condition is provided, this is the 'else' branch
      cfg.then(compileStatement(func, contents));
    } else {
      cfg
        .then(compileExpression(condition, Flavor::Cell))
        .branchNZ(
          compileStatement(func, contents)
            .thenJmp(end)
        );
    }
  }

  cfg.thenJmp(end);

  return cfg.continueFrom(end);
}

CFG compileWhile(Function* func,
    const zend_ast* condition,
    const zend_ast* contents,
    bool bodyFirst) {
  return CFG::Labeled(
    ".start",
    CFG().then(bodyFirst
      ? ".loop-body"
      : ".loop-header"),

    ".loop-body",
    compileStatement(func, contents)
      .linkLoop(
        CFG().then(".end"),
        CFG().then(".loop-header"))
      .then(".loop-header"),

    ".loop-header",
    compileExpression(condition, Flavor::Cell)
      .branchNZ(".loop-body"),

    ".end",
    CFG()
  );

  return {};
}

CFG compileFor(Function* func, const zend_ast* ast) {
  auto initializers = zend_ast_get_list(ast->child[0]);
  auto tests = zend_ast_get_list(ast->child[1]);
  auto increments = zend_ast_get_list(ast->child[2]);
  auto body = ast->child[3];

  CFG cfg;

  const auto doAll = [&] (const zend_ast_list* list) {
    CFG all;
    if (list) {
      for (uint32_t i = 0; i < list->children; i++) {
        all.then(compileExpression(list->child[i], Flavor::Drop));
      }
    }
    return all;
  };

  CFG testCFG;
  if (tests) {
    for (uint32_t i = 0; i < tests->children; i++) {
      // ignore all but the last test
      auto flavor = (i + 1 < tests->children)
        ? Flavor::Drop
        : Flavor::Cell;
      testCFG.then(compileExpression(tests->child[i], flavor));
    }
    testCFG.branchNZ(".body");
  } else {
    testCFG.then(".body");
  }

  // compile the initializers
  cfg.then(CFG::Labeled(
    ".initializers",
    doAll(initializers)
      .then(".test"),

    ".body",
    compileStatement(func, body)
      .linkLoop(
        CFG().then(".end"),
        CFG().then(".increment"))
      .then(".increment"),

    ".increment",
    doAll(increments).then(".test"),

    ".test",
    std::move(testCFG),

    ".end",
    CFG()
  ));

  return cfg;
}

Attr translateAttr(uint32_t flags, bool pppok) {
  Attr attr{};
  if (flags & ZEND_ACC_PPP_MASK && pppok) {
    if (flags & ZEND_ACC_PUBLIC) {
      // We don't mark flags & ZEND_ACC_IMPLICIT_PUBLIC as public because hhvm
      // should implicitly mark them as public as well.
      attr |= Attr::AttrPublic;
    }
    if (flags & ZEND_ACC_PROTECTED) {
      attr |= Attr::AttrProtected;
    }
    if (flags & ZEND_ACC_PRIVATE) {
      attr |= Attr::AttrPrivate;
    }
  }
  if (flags & ZEND_ACC_STATIC) {
    attr |= Attr::AttrStatic;
  }
  if (flags & ZEND_ACC_ABSTRACT
      || flags & ZEND_ACC_IMPLEMENTED_ABSTRACT
      || flags & ZEND_ACC_IMPLICIT_ABSTRACT_CLASS
      || flags & ZEND_ACC_EXPLICIT_ABSTRACT_CLASS) {
    attr |= Attr::AttrAbstract;
  }
  if (flags & ZEND_ACC_FINAL) {
    attr |= Attr::AttrFinal;
  }
  if (flags & ZEND_ACC_INTERFACE) {
    attr |= Attr::AttrInterface;
  }
  if (flags & ZEND_ACC_TRAIT) {
    attr |= Attr::AttrTrait;
  }
  if (flags & ZEND_ACC_RETURN_REFERENCE) {
    attr |= Attr::AttrReference;
  }

  return attr;
  // Unused flags:
  //
  // method flag (bc only), any method that has this flag can be used statically
  // and non statically.
  // ZEND_ACC_ALLOW_STATIC
  //
  // ZEND_ACC_CHANGED
  //
  // method flags (special method detection)
  // ZEND_ACC_CTOR
  // ZEND_ACC_DTOR
  //
  // method flag used by Closure::__invoke()
  // ZNED_ACC_USER_ARG_INFO
  //
  // shadow of parent's private method/property
  // ZEND_ACC_SHADOW
  //
  // deprecation flag
  // ZEND_ACC_DEPRECATED
  //
  // ZEND_ACC_CLOSURE
  // ZEND_ACC_FAKE_CLOSURE
  // ZEND_ACC_GENERATOR
  //
  // ZEND_ACC_NO_RT_ARENA
  //
  // call through user function trampoline. e.g. __call, __callstatic
  // ZEND_ACC_CALL_VIA_TRAMPOLINE
  //
  // call through internal function handler. e.g. Closure::invoke()
  // ZEND_ACC_CALL_VIA_HANDLER     ZEND_ACC_CALL_VIA_TRAMPOLINE
  //
  // disable inline caching
  // ZEND_ACC_NEVER_CACHE
  //
  // ZEND_ACC_VARIADIC
  //
  // ZEND_ACC_RETURN_REFERENCE
  // ZEND_ACC_DONE_PASS_TWO
  //
  // class has magic methods __get/__set/__unset/__isset that use guards
  // ZEND_ACC_USE_GUARDS
  //
  // function has typed arguments
  // ZEND_ACC_HAS_TYPE_HINTS
  //
  // op_array has finally blocks
  // ZEND_ACC_HAS_FINALLY_BLOCK
  //
  // internal function is allocated at arena
  // ZEND_ACC_ARENA_ALLOCATED
  //
  // Function has a return type (or class has such non-private function)
  // ZEND_ACC_HAS_RETURN_TYPE
  //
  // op_array uses strict mode types
  // ZEND_ACC_STRICT_TYPES
  //
  // ZEND_ACC_ANON_CLASS
  // ZEND_ACC_ANON_BOUND
  // ZEND_ACC_INHERITED
  //
  // class implement interface(s) flag
  // ZEND_ACC_IMPLEMENT_INTERFACES
  // ZEND_ACC_IMPLEMENT_TRAITS
  //
  // class constants updated
  // ZEND_ACC_CONSTANTS_UPDATED
}

CFG compileIncludeEval(const zend_ast* ast) {
  CFG cfg = compileExpression(ast->child[0], Flavor::Cell);
  switch (ast->attr) {
    case ZEND_EVAL:
      cfg.then(Eval{});
      break;
    case ZEND_INCLUDE:
      cfg.then(Incl{});
      break;
    case ZEND_INCLUDE_ONCE:
      cfg.then(InclOnce{});
      break;
    case ZEND_REQUIRE:
      cfg.then(Req{});
      break;
    case ZEND_REQUIRE_ONCE:
      cfg.then(ReqOnce{});
      break;
    default:
      panic("unsupported include/eval type");
  }
  return cfg;
}

void compileClassStatement(Class* cls, const zend_ast* ast) {
  switch(ast->kind) {
    case ZEND_AST_METHOD:
      compileMethod(cls, ast);
      break;
    case ZEND_AST_PROP_DECL:
      compileProp(cls, ast);
      break;
    case ZEND_AST_USE_TRAIT: {
      assert(ast->child[0]->kind == ZEND_AST_NAME_LIST);
      auto list = zend_ast_get_list(ast->child[0]);
      for (uint32_t i = 0; i < list->children; i++) {
        auto item = list->child[i];

        cls->traits.emplace_back(zval_to_string(zend_ast_get_zval(item)));
      }
      break;
    }
    default:
      panic("unsupported class statement");
  }
}

void compileMethod(Class* cls, const zend_ast* ast) {
  auto func = cls->makeMethod();
  buildFunction(func, ast);
  if (func->attr & Attr::AttrAbstract) {

    if (func->attr & Attr::AttrPrivate
        || func->attr & Attr::AttrFinal) {
      throw LanguageException(
        folly::sformat(
          "Cannot declare abstract method {}::{}() {}",
          cls->name,
          func->name,
          func->attr & Attr::AttrPrivate ? "private" : "final"
        )
      );
    }
    if (!(cls->attr & Attr::AttrAbstract)
        && !(cls->attr & Attr::AttrInterface)) {
      throw LanguageException(
        folly::sformat(
          "Class {} contains abstract {} and must therefore be declared "
          "abstract",
          cls->name,
          func->name
        )
      );
    }
  }
}

bool serializeZval(std::string& out, const zval* zv) {
  switch (Z_TYPE_P(zv)) {
    case IS_LONG:
      out.append(folly::sformat("i:{};", Z_LVAL_P(zv)));
      break;
    case IS_NULL:
      out.append("N;");
      break;
    case IS_FALSE:
      out.append("b:0;");
      break;
    case IS_TRUE:
      out.append("b:1;");
      break;
    case IS_DOUBLE:
      out.append(folly::sformat("d:{};", Z_DVAL_P(zv)));
      break;
    case IS_STRING:
      {
        auto str = Z_STRVAL_P(zv);
        out.append(folly::sformat("s:{}:\\\"{}\\\";", strlen(str), str));
      }
      break;
    default:
      panic("unsupported literal");
  }
  return true;
}

bool serializeConstant(std::string& out, const zend_ast* ast) {
  auto name = ast->child[0];
  const char* str = Z_STRVAL_P(zend_ast_get_zval(name));
  if (name->attr & ZEND_NAME_NOT_FQ) {
    if (strcasecmp(str, "true") == 0) {
      out.append("b:1;");
    } else if (strcasecmp(str, "false") == 0) {
      out.append("b:0;");
    } else if (strcasecmp(str, "null") == 0) {
      out.append("N;");
    } else {
      panic("unknown unqualified constant");
    }
  } else {
    panic("unknown constant");
  }
  return true;
}

bool serializeArray(std::string& out, const zend_ast* ast) {
  auto list = zend_ast_get_list(ast);

  if (list->attr == ZEND_ARRAY_SYNTAX_LIST) {
    return false;
  }

  bool ret = true;
  out.append(folly::sformat("a:{}:{{", list->children));
  zend_long cursor = 0;
  for (uint32_t i = 0; i < list->children && ret; i++) {
    auto item = list->child[i];
    if (!item) {
      return false;
    }

    if (item->child[1]) {
      ret &= serializeInitilizer(out, item->child[1]);
      ret &= serializeInitilizer(out, item->child[0]);
      if (item->child[1]->kind == ZEND_AST_ZVAL) {
        auto zv = zend_ast_get_zval(item->child[1]);
        if (Z_TYPE_P(zv) == IS_LONG) {
          cursor = std::max(Z_LVAL_P(zv), cursor);
        }
      }
    } else {
      out.append(folly::sformat("i:{};", cursor));
      ret &= serializeInitilizer(out, item->child[0]);
    }
    cursor++;
  }
  out.append("}");
  return ret;
}

bool serializeInitilizer(std::string& out, const zend_ast* ast) {
  switch (ast->kind) {
    case ZEND_AST_ZVAL:
      return serializeZval(out, zend_ast_get_zval(ast));
    case ZEND_AST_CONST:
      return serializeConstant(out, ast);
    case ZEND_AST_ARRAY:
      return serializeArray(out, ast);
    default:
      return false;
  }
}

void compileProp(Class* cls, const zend_ast* ast) {
  auto pelem = ast->child[1];
  assert(pelem->kind == ZEND_AST_PROP_ELEM);
  auto name = pelem->child[0];
  auto init = pelem->child[1];

  assert(name->kind == ZEND_AST_ZVAL);
  auto zv = zend_ast_get_zval(name);
  assert(Z_TYPE_P(zv) == IS_STRING);

  std::string initString;
  initString.append("\"\"\"");
  bool staticInit = serializeInitilizer(initString, init);
  initString.append( "\"\"\"");
  if (staticInit) {
    cls->properties.emplace_back(Class::Property{
      Z_STRVAL_P(zv),
      translateAttr(ast->attr, true),
      initString,
      CFG{},
    });
  } else {
    cls->properties.emplace_back(Class::Property{
      Z_STRVAL_P(zv),
      translateAttr(ast->attr, true),
      "uninit",
      compileExpression(init, Flavor::Cell),
    });
  }
}

} // namespace

}} // HPHP::php7
