/* preferences.cxx
     $Id: preferences.cxx,v 1.11 2001/11/27 23:57:36 elf Exp $

   written by Marc Singer
   27 October 1996

   This file is part of the project CurVeS.  See the file README for
   more information.

   Copyright (C) 1996 Marc Singer

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   with your Debian GNU/Linux system, in
   /usr/share/common-licenses/GPL, or with the Debian GNU/Linux hello
   source package as the file COPYING. If not, write to the Free
   Software Foundation, Inc., 59 Temple Place -Suite 330, MA
   02111-1307, USA.

   -----------
   DESCRIPTION
   -----------

   Preference options management class.  It handles queries and
   updates.  Essentially, this is a simple database class with keys
   and data.  Keys and data are both strings and we make no provisions
   for typing them.  We may choose to include methods to interpret the
   assumed data type, but this may NOT be done.


   --------------------
   IMPLEMENTATION NOTES
   --------------------

   -- Preferences File Syntax

   The file is line oriented.  Preferences are <keyword> = <data>
   pairs.  Whitespace around the keyword and the data are ignored.
   Comments lines start with '#' and all text on a line from the
   comment character to the end-of-line are ignored.  Blank lines are
   ignored, too.

   -- Set Representation

   We start with a VERY simple implementation for key/data matching.
   Presently it is a flat-file mapping.  We may choose to go to a
   hierarchy and to some sort of balanced tree such as AVL.

*/

#include "std.h"
#include <sys/stat.h>

#include "mman.h"
#include "preferences.h"
#include "path.h"

LPreferences g_preferences;	// Instantiate global preference object


/* LPreferences::as_*

   these functions return the preference data typed.  This is for the
   convenience of the user and the programmer.  For users, they can
   specify t, true, yes, y, or Y for boolean truth.  For programmers,
   they can make one call to determine the truth of the option.

*/

bool LPreferences::as_bool (const char* szKeyword)
{
  const char* szValue = fetch (szKeyword);
  if (!szValue || !*szValue)
    return false;

  switch (tolower (*szValue)) {
  case '1':
    if (szValue[1] == 0)
      return true;
    break;
  case '0':
    if (szValue[1] == 0)
      return false;
    break;
  case 'y':
    if (szValue[1] == 0
	|| strcasecmp (szValue, "yes") == 0
	|| strcasecmp (szValue, "yep") == 0)
      return true;
    break;
  case 't':
    if (szValue[1] == 0 || strcasecmp (szValue, "true") == 0)
      return true;
  case 'n':
    if (szValue[1] == 0
	|| strcasecmp (szValue, "no") == 0
	|| strcasecmp (szValue, "nope") == 0)
      return false;
    break;
  case 'f':
    if (szValue[1] == 0 || strcasecmp (szValue, "false") == 0)
      return false;
    break;
  }
  TRACE((T_PREFS_ERR, "unrecognized preference value %s='%s'",
     szKeyword, szValue));
  return false;
}

int LPreferences::as_int (const char* szKeyword)
{
  const char* szValue = fetch (szKeyword);
  int value = strtol (szValue, (char**) NULL, 0);
  return value;
}


/* LPreferences::associate

   matches a keyword to a value.  This is the user version that
   allocates space for copies of the strings if either is new.

*/

void LPreferences::associate (const char* szKeyword, const char* szValue)
{
  LPreferenceElement* pElement = _find (szKeyword);
  if (!pElement) {
    pElement = (LPreferenceElement*) malloc (sizeof (LPreferenceElement));
    memset (pElement, 0, sizeof (*pElement));
    pElement->pNext = m_pElements;
    m_pElements = pElement;
  }
  assert_ (pElement);
  if (!pElement->szKeyword) {
    char* sz = (char*) malloc (strlen (szKeyword) + 1);
    strcpy (sz, szKeyword);
    pElement->szKeyword = sz;
  }
  if (pElement->szValue)
    free (pElement->szValue);
  char* sz = (char*) malloc (strlen (szValue) + 1);
  strcpy (sz, szValue);
  pElement->szValue = sz;
}

void LPreferences::associate (const char* szKeyword, int value)
{
  char szValue[20];
  sprintf (szValue, "%d", value);
  associate (szKeyword, szValue);
}


/* LPreferences::associate

   matches a keyword to a value.  This is the internet version that
   assumes that space was already allocated for the keyword and value
   strings. 

*/

void LPreferences::_associate (char* szKeyword, char* szValue)
{
  LPreferenceElement* pElement = _find (szKeyword);
  if (!pElement) {
    pElement = (LPreferenceElement*) malloc (sizeof (LPreferenceElement));
    memset (pElement, 0, sizeof (*pElement));
    pElement->pNext = m_pElements;
    m_pElements = pElement;
  }
  assert_ (pElement);
  if (!pElement->szKeyword)
    pElement->szKeyword = szKeyword;
  else
    free (szKeyword);
  if (pElement->szValue)
    free (pElement->szValue);
  pElement->szValue = szValue;
				// Mitigation, won't be good, but it'll work
  if (strlen (szValue) > CB_PREFERENCE_MAX) {
    TRACE((T_PREFS_ERR, "truncation too long preference %s='%s'",
	   pElement->szKeyword, szValue));
    szValue[CB_PREFERENCE_MAX] = 0;
  }
}


const char* LPreferences::fetch (const char* szKeyword)
{
  for (LPreferenceElement* pElement = m_pElements; pElement; 
       pElement = pElement->pNext)
    if (strcasecmp (pElement->szKeyword, szKeyword) == 0)
      return pElement->szValue ? pElement->szValue : pElement->szValueDefault;
  return NULL;
}


LPreferenceElement* LPreferences::_find (const char* szKeyword)
{
  for (LPreferenceElement* pElement = m_pElements; pElement; 
       pElement = pElement->pNext)
    if (strcasecmp (pElement->szKeyword, szKeyword) == 0)
      return pElement;
  
  return NULL;
}
    

/* LPreferences::grab

   strips the whitespace that surrounds the keyword or data item.  It
   allocates a bit of memory for the string and returns a copy of it.

*/

char* LPreferences::grab (char* pb, int cch)
{
  while (cch && isspace (*pb))
    ++pb, --cch;		// Yeah, and I meant it.
  while (cch && isspace (pb[cch - 1]))
    --cch;

  char* sz = (char*) malloc (cch + 1);
  memcpy (sz, pb, cch);
  sz[cch] = 0;
  return sz;
}


void LPreferences::read (const char* szPath)
{
  LPath path (szPath);

  int fh = -1;
  char* pbMap = NULL;
  struct stat stat;
  if (!::stat (path, &stat) && stat.st_size
      && ((fh = open (path, O_RDONLY)) != -1)
      && (pbMap = (char*) mmap (NULL, stat.st_size + 1, PROT_READ, 
				MAP_FILE | MAP_PRIVATE, fh, 0))) {
    //    pbMap[stat.st_size] = 0;
    int iLine = 1;
    char* szKeyword = NULL;
    char* szValue = NULL;
    int state = 0;
    int cch;
    for (char* pb = pbMap; pb < pbMap + stat.st_size; pb += cch) {
      if (*pb == '\n') {
	++iLine;
	++pb;
      }
      cch = strcspn (pb, "\n=#");
      if (pb + cch > pbMap + stat.st_size)
	cch = pbMap + stat.st_size - pb;

      switch (state) {
      case 0:			// Looking for keyword
	if (pb[cch] != '=') {
	  cch += strcspn (pb + cch, "\n");
	  break;
	}
	szKeyword = grab (pb, cch);
	state = 1;
	++cch;			// Skip '='
	break;

      case 1:			// Looking for value
	szValue = grab (pb, cch);
	_associate (szKeyword, szValue);
	szKeyword = szValue = NULL;
	cch += strcspn (pb + cch, "\n");
	state = 0;
	break;
      }
    }      
  }
  if (pbMap)
    munmap (pbMap, stat.st_size);
  if (fh != -1)
    close (fh);
}


void LPreferences::release_entries (void)
{
  for (LPreferenceElement* pElement = m_pElements; pElement;
       pElement = m_pElements) {
    m_pElements = pElement->pNext;
    if (pElement->szKeyword)
      free (pElement->szKeyword);
    if (pElement->szValue)
      free (pElement->szValue);
    if (pElement->szValueDefault)
      free (pElement->szValueDefault);
    free (pElement);
  }
}

void LPreferences::release_settings (void)
{
  for (LPreferenceElement* pElement = m_pElements; pElement;
       pElement = pElement->pNext)
    if (pElement->szValue) {
      free (pElement->szValue);
      pElement->szValue = NULL;
    }
}

void LPreferences::release_this (void)
{
  release_entries ();
}

void LPreferences::transfer_defaults (void)
{
  for (LPreferenceElement* pElement = m_pElements; pElement;
       pElement = pElement->pNext) {
    if (pElement->szValueDefault)
      free (pElement->szValueDefault);
    pElement->szValueDefault = pElement->szValue;
    pElement->szValue = NULL;
  }
}
