;;
;; This file is part of q-tools, a collection of performance tools
;; Copyright (c) 2003 Hewlett-Packard Development Company, L.P.
;; Contributed by David Mosberger-Tang <davidm@hpl.hp.com>
;;
;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2 of the License, or
;; (at your option) any later version.
;;
;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.
;;
;; You should have received a copy of the GNU General Public License
;; along with this program; if not, write to the Free Software
;; Foundation, Inc., 59 Temple Place, Suite 330,
;; Boston, MA  02111-1307  USA
;;
(provide 'q-lib)

;; For SLIB (portable Scheme library) documentation, see:
;;   http://www.swiss.ai.mit.edu/~jaffer/slib_toc
;;
(use-modules (ice-9 regex) (ice-9 popen))

;; Blech, neither RHAS3 nor SLES 9 ship the GUILE SLIB so we make
;; up for the missing printf by including it here explicitly.
(load "slib-printf.scm")

;;(setenv "LTDL_LIBRARY_PATH" "/home/davidm/src/x1/libguile-qprof")
;;(load-extension "libguile-qprof" "init_qprof")

;;
;; Description of symbols used by this library:
;;
;; Symbol:			Description:
;;
;; q:name			The name of the object.
;; q:file			The filename (path) associated with the object.
;; q:event-name
;; q:x-unit-label
;; q:x-granularity
;; q:y-unit-label
;; q:y-unit-conversion-factor	Conversion factor to y sample values to
;;				weights.
;; q:y-granularity		Sampling granularity.
;; q:weight-EVENTNAME
;; q:child-weight-EVENTNAME
;; q:callers			Incoming edges in the form '((SYM . COUNT) ...)
;; q:callees			Outgoing edges if the form '((SYM . COUNT) ...)
;; q:call-count		Total number of times the object was called.
;; q:symbol-bind		A character indicating the binding of the obj.
;; q:topo-order		The topological sort-order number assigned to
;;				this object.  When traversing a graph, an
;;				an edge from A to B is a back-edge ("recursive
;;				call) iff the topological sort-order number of
;;				B is less-than-or-equal to that of A.
;; q:value			The value of the symbol (usually an address).
;; q:object			The loda-module which defines this object.
;; q:size			The size of the object.
;; q:to-visit			A temporary marker symbol used for graph
;;				traversals.
;;

;;
;; Including topological-sort is very slow in guile (presumably because
;; of its dependencies).  So instead of
;;  (require 'topological-sort)
;; we justs replicate the code here in line:

;;; "tsort.scm" Topological sort
;;; Copyright (C) 1995 Mikael Djurfeldt
;
; This code is in the public domain.

;;; The algorithm is inspired by Cormen, Leiserson and Rivest (1990)
;;; "Introduction to Algorithms", chapter 23
(define (tsort dag pred)
  (if (null? dag)
      '()
      (let* ((adj-table (make-hash-table
			 (* 2 (length dag) 1)))
	     (sorted '()))
	(letrec ((visit
		  (lambda (u adj-list)
		    ;; Color vertex u
		    (hash-set! adj-table u 'colored)
		    ;; Visit uncolored vertices which u connects to
		    (for-each (lambda (v)
				(let ((val (hash-ref adj-table v)))
				  (if (not (eq? val 'colored))
				      (visit v (or val '())))))
			      adj-list)
		    ;; Since all vertices downstream u are visited
		    ;; by now, we can safely put u on the output list
		    (set! sorted (cons u sorted)))))
	  ;; Hash adjacency lists
	  (for-each (lambda (def)
		      (hash-set! adj-table (car def) (cdr def)))
		    (cdr dag))
	  ;; Visit vertices
	  (visit (caar dag) (cdar dag))
	  (for-each (lambda (def)
		      (let ((val (hash-ref adj-table (car def))))
			(if (not (eq? val 'colored))
			    (visit (car def) (cdr def)))))
		    (cdr dag)))
	sorted)))

(define q:dir-name (or (getenv "Q_DIR") ".q"))

(define q:object-hash (make-hash-table 99)) ;; hash-table of (ELF-) objects

(define (q:hash-length table)
  "Return the number of elements in a hash-table."
  (hash-fold (lambda (key value seed) (1+ seed)) 0 table))

(define (q:open-file filename mode)
  "Open the file named FILENAME with access-mode MODE.  MODE has the
same meaning as for the open-file procedure.  If FILENAME does not
exist, look in the qprof directory (initialized from
environment-variable QPROF_DIR).  If the version-number is omitted,
find the highest-numbered version of the file and open it."
  (let ((default-path (string-append q:dir-name "/" filename))
	(port #f))
    (cond

     ((access? filename F_OK)
      (let ((dir (dirname filename)))
	(if dir (set! q:dir-name dir))
	(set! port (open-file filename mode))))

     ((access? default-path F_OK)
      (set! port (open-file default-path mode)))

     (else
      (let* ((dir-path (if (string-index filename #\/)
			   (dirname filename)
			   q:dir-name))
	     (fname (basename filename))
	     (len (string-length fname))
	     (dir-port (opendir dir-path))
	     (max-vers -1))
	(set! q:dir-name dir-path)
;;	(printf "filename=%s dir-path=%s fname=%s\n" filename dir-path fname)
	(do ((entry (readdir dir-port) (readdir dir-port)))
	    ((eof-object? entry) (closedir dir-port))
	  (if (and (> (string-length entry) (1+ len))
		   (string=? fname (substring entry 0 len))
		   (eq? #\# (string-ref entry len)))
	      (set! max-vers (max max-vers
				  (or (string->number (substring entry
								 (+ 1 len)))
				      -1)))))
	(if (>= max-vers 0)
	    (set! port (open-file (string-append dir-path "/" fname "#"
						 (number->string max-vers))
				  mode))))))
    (if (not port)
	(error (sprintf #f "Failed to open file `%s'\n" filename)))
    port))

;;
;; Helper routines for managing symbols.  A symbol consists of a
;; cons-cell whose car is the cell-name, and the cdr is an
;; association-list.  KEY/VALUE pairs in the assocation-list can be
;; managed via q:sym-ref, q:sym-set, and q:sym-remove!,
;; respectively.
;;
(define (q:sym name alist)
  "Create a symbol named NAME and with association-list ALIST."
  (cons name alist))

(define (q:sym-name sym)
  "Return the name of symbol SYM."
  (car sym))

(define (q:sym-ref sym key)
  "For symbol SYM, get the value associated with KEY."
  (assq-ref (cdr sym) key))

(define (q:sym-set! sym key value)
  "For symbol SYM, set KEY to VALUE."
  (let ((new-alist (assq-set! (cdr sym) key value)))
    (if (not (eq? new-alist (cdr sym)))
	(set-cdr! sym new-alist))))

(define (q:sym-remove! sym key)
  "For symbol SYM, remove KEY and its value."
  (let ((new-alist (assq-remove! (cdr sym) key)))
    (if (not (eq? new-alist (cdr sym)))
	(set-cdr! sym new-alist))))

(define (q:sym-compare left right keys)
  "Compare two symbols (which must be alists).  Returns a negative
value is LEFT is smaller than RIGHT, 0 if they are equal, and a
positive value otherwise.  The comparison is done based on the keys
passed in KEYS.  For example, if KEYS is ('q:weight-time
'q:call-count), then the primary key is 'q:weight-time.  If LEFT and
RIGHT have the same value for this key, then the 'q:call-count values
of LEFT and RIGHT are used as a secondary key."
  (if (null? keys)
      0
      (let* ((key (car keys))
	     (lval (q:sym-ref left key))
	     (rval (q:sym-ref right key)))
	(cond
	 ((and lval (not rval)) 1)
	 ((and (not lval) rval) -1)
	 ((or (and (not lval)) (not rval) (= lval rval))
	  (q:sym-compare left right (cdr keys)))
	 (else (- lval rval))))))


(define (q:symtab-enter! sym-tab name bind value object)
  (define (fully-qualified-name obj name)
    (if obj (string-append name "<" (assq-ref obj 'q:name) ">") name))
  ;; Check if we have a name-collision (same name pointing to
  ;; different objects; if so, change the new symbol's name to the
  ;; fully-qualified name (SYMNAME<OBJNAME>).  We leave the existing
  ;; symbol name unchanged.  The logic here is to simulate dynamic
  ;; name resolution: the first object defining a symbol will be the
  ;; one that "wins" from the perspective of the main-program.
  (define (unique-name name val obj)
    (let* ((old-sym (hash-ref sym-tab name))
	   (old-obj (if old-sym (q:sym-ref old-sym 'q:object) obj))
	   (old-val (if old-sym (q:sym-ref old-sym 'q:value) val)))
      (if (eq? obj old-obj)
	  (if (eq? val old-val)
	      ;; same object, same name, same value, same thing!
	      name
	      ;; same object, same name, but different value (can
	      ;; happen due to overlapping mappings) => append value
	      ;; to symbol-name:
	      (string-append name "@" (number->string value 16)))
	  ;; different objects => prefix name with object name
	  (unique-name (fully-qualified-name obj name) val obj))))
  (set! name (unique-name name value object))
  ;(printf "%016x -> %s [%c]\n" value name bind)
  (hash-set! sym-tab name (cons name (acons 'q:symbol-bind bind
					    (acons 'q:value value
						   (acons 'q:object object
							  '()))))))

(define (q:set-kallsyms! sym-tab port postfix)
  "Read a symbol-table in the /proc/kallsyms format from PORT and
install the symbols in SYM-TAB with POSTFIX as a postfix."
  (let (
	(rx (make-regexp "([0-9a-f]*) (.) ([^\t]*)(.[[]([^]]*)[]])?"))
	(line #f) (m #f) (val #f) (name #f) (bind #f) (module #f) (object #f)
	(p1 #f) (p2 #f) (p3 #f) (p4 #f))
    (do ((line (read-line port) (read-line port)))
	((eof-object? line))
      (set! m (regexp-exec rx line))
      (set! val (string->number (match:substring m 1) 16))
      (set! bind (string-ref (match:substring m 2) 0))
      (set! name (string-append (match:substring m 3) postfix))
      (set! module (match:substring m 5))
      ;; ignore absolute & undefined symbols:
      (if (not (string-index "aU" bind))
	  (begin
	    (if module
		(set! object (hash-ref q:object-hash module))
		(set! object (hash-ref q:object-hash "vmlinux")))
	    (q:symtab-enter! sym-tab name bind val object))))))

(define (q:read-elf-phdr path)
  (let* ((rx (make-regexp " *LOAD *0x([^ ]*) 0x([^ ]*)"))
	 (cmd (string-append "readelf -l " path "| grep '^  LOAD '"))
	 (pipe (open-input-pipe cmd))
	 (i 0) (m #f) (offset #f) (vaddr #f) (num_phdrs 0) (phdr_list '())
	 (v #f))
    (do ((line (read-line pipe) (read-line pipe)))
	((eof-object? line))
      (set! m (regexp-exec rx line))
      (set! offset (string->number (match:substring m 1) 16))
      (set! vaddr (string->number (match:substring m 2) 16))
      (set! num_phdrs (1+ num_phdrs))
      (set! phdr_list (append phdr_list (list (cons vaddr offset)))))
    (set! v (make-vector num_phdrs))
    (for-each (lambda (phdr)
		(vector-set! v i phdr)
		(set! i (1+ i)))
	      phdr_list)
    (stable-sort v (lambda (left right)
		     < (car left) (car right)))
    v))

(define (q:set-object-syms! sym-tab object allowed-symbol-type)
  "Read a symbol-table from object file OBJECT and install the symbols
in SYM-TAB.  SYMBOL-TYPE is a string specifying the type of symbols
that will be entered into the symbol-table (e.g., \"FUNC\" for
functions, \"OBJECT\" for data objects) or #f to allow all
symbol-types."

  (define (create-symbol-for-mapping mapping)
    (let ((addr (assq-ref mapping 'q:addr)))
      (q:symtab-enter! sym-tab (string-append (assq-ref object 'q:name)
					      "@" (number->string addr 16))
		       #\A addr object)))

  (define (enter-mapped-symbol name bind val mapping phdr-offset)
    (let* ((start (assq-ref mapping 'q:addr))
	   (end (+ start (assq-ref mapping 'q:size)))
	   (off (assq-ref mapping 'q:offset))
	   (base (- start off phdr-offset))
	   (abs-val (if (eq? bind #\A) val (+ val base))))
      (if (and (>= abs-val start) (< abs-val end))
	  (q:symtab-enter! sym-tab name bind
			   abs-val object))))

  (let ((filename (assq-ref object 'q:file)))
    (if filename
	(begin
;	  (printf "Doing %s:\n" filename)
	  (let* ((value-table (make-hash-table 10000))
		 (maps (assq-ref object 'q:maps))
		 (path (string-append q:dir-name "/" filename))
		 (phdrs (q:read-elf-phdr path))
		 ;; Sort symbols in reverse alphabetical order, such
		 ;; that symbols starting with "_" come after symbols
		 ;; starting with a letter; this is useful for the
		 ;; Nvidia drivers where the same value may be
		 ;; attached to both an obfuscated and a normal
		 ;; symbol.
		 (cmd (string-append "(nm -f sysv -C --defined " path
				     ";nm -f sysv -C --defined -D " path
				     ")2>/dev/null|sort -r|uniq"))
		 (rx (make-regexp
	   "(.*)[|]([0-9a-fA-F]+)[|] *([^ ]*) *[|] *([^|]*)[|]([^|]*)[|]"))
		 (trailing-ws-rx (make-regexp "(.*[^ ])[ ]*$"))
		 (pipe (open-input-pipe cmd))
		 (m #f) (val #f) (type #f) (name #f) (addr 0)
		 (size #f) (bind #f))

	    (define (enter-symbol name bind val)
	      (let ((padj 0))
		(array-for-each (lambda (phdr)
				  (if (>= val (car phdr))
				      (set! padj (- (car phdr)
						    (cdr phdr)))))
				phdrs)
		(if maps
		    (for-each (lambda (mapping)
				(enter-mapped-symbol name bind val mapping
						     padj))
			      maps)
		    (q:symtab-enter! sym-tab name bind val object))))


	    ;; For each mapping, create an artificial symbol of the
	    ;; form <OBJECTNAME.MAPPINGNUMBER>.  This ensures that
	    ;; addresses with no matching symbol get mapped to the
	    ;; correct object at least.
	    (if maps (for-each create-symbol-for-mapping maps))
	    (do ((line (read-line pipe) (read-line pipe)))
		((eof-object? line))
;	      (printf "line='%s'\n" line)
	      (if (string-index line #\|)
		  (begin
		    (set! m (regexp-exec rx line))
		    (set! name (match:substring
				(regexp-exec trailing-ws-rx
					     (match:substring m 1)) 1))
		    (set! val (string->number (match:substring m 2) 16))
		    (set! bind (string-ref (match:substring m 3) 0))
		    (set! type (match:substring m 4))
;		    (set! size (string->number (match:substring m 5) 16))
;		    (printf "%016x %40s %c %s %x\n" val name bind type size)
		    (if (and (not (hash-ref value-table val))
			     (or (not allowed-symbol-type)
				 (string=? allowed-symbol-type type)))
			(begin
			  (hash-set! value-table val #t)
			  (enter-symbol name bind val))))))
	    (close-pipe pipe)

	    ;; Look at the unwind info and make up a symbol name for
	    ;; all nameless procedures.  That makes it more likely
	    ;; that the flat profile aggregation and the call-count
	    ;; profiles are meaningful, even if the names themselves
	    ;; are not.  Just try looking at a profile running an
	    ;; OpenGL application with the Nvidia driver, if you don't
	    ;; think this matters...
	    (set! cmd (string-append "readelf -u " path "|fgrep '<>'"))
	    (set! pipe (open-input-pipe cmd))
	    (set! rx (make-regexp
		      "[^[]*0x0*([0-9a-fA-F]+)-0x0*([0-9a-fA-F]+)[]],"))
	    (do ((line (read-line pipe) (read-line pipe)))
		((eof-object? line))
 	      (set! m (regexp-exec rx line))
	      (set! val (string->number (match:substring m 1) 16))
;	      (set! size (- (string->number (match:substring m 2) 16) val))
;	      (printf "  val=%016x\n" val)
	      (if (not (hash-ref value-table val))
		  (enter-symbol
		   (string-append "0x" (match:substring m 1)
				  "<" (assq-ref object 'q:name) ">") #\t val)))
	    (close-pipe pipe))))))

;;
;; Helper routines for managing an address-map, which is a mapping
;; from addresses to symbols.
;;
(define (q:addr-map sym-tab unaccounted-sym)
  "Build an address-map (lookup-table sorted by 'q:value) for
symbol-table SYM-TAB.  The first element in the table refers to the
special symbol UNACCOUNTED-SYM, which is returned when an
address-lookup would otherwise fail."
  (let* ((num-symbols (1+ (q:hash-length sym-tab)))
	 (v (make-vector num-symbols)))
    (vector-set! v 0 unaccounted-sym)
    (hash-fold (lambda (key value i)
		 (vector-set! v i value)
		 (1+ i)) 1 sym-tab)
    (stable-sort! v (lambda (left right)
		      (cond ((eq? right unaccounted-sym) #f)
			    ((eq? left unaccounted-sym) #t)
			    (else (< (q:sym-ref left 'q:value)
				     (q:sym-ref right 'q:value))))))
    ;; Print the final address map:
;     (let ((i 0) (len (vector-length v)))
;       (while (< i len)
;  	     (printf "%x -> %s\n"
;  		     (q:sym-ref (vector-ref v i) 'q:value)
;  		     (q:sym-name (vector-ref v i)))
;  	     (set! i (1+ i))))
    v))

(define (q:addr-map-ref map addr)
  "Search address-map MAP for an symbol that matches ADDR.  The
address-map must be a sorted vector of symbols, where each symbol is
represented by an association list.  Each association list _must_
define a 'q:value key, which specifies the starting address of the
symbol, and, optionally, a 'q:size key to specify the length of the
address-range covered by the symbol.  If a matching symbol is found,
the corresponding association-list is returned, otherwise #f.  The
search is performed in logarithmic time (via a binary search)."
  (let ((lo 1)
	(mid 0)
	(midval #f)
	(start 0)
	(end 0)
	(size 0)
	(hi (vector-length map))
	(result #f))
    (while (and (not result) (> hi lo))
	   (set! mid (quotient (+ lo hi) 2))
	   (set! midval (vector-ref map mid))
	   (set! start (q:sym-ref midval 'q:value))
	   (if (< addr start)
	       (set! hi mid)
	       (begin
		 (set! size (q:sym-ref midval 'q:size))
		 (if size
		     (set! end (+ start size))
		     (set! end
			   (if (< (1+ mid) (vector-length map))
			       (q:sym-ref (vector-ref map (1+ mid))
					   'q:value)
			       (1+ addr))))
		 (if (< addr end)
		     (set! result midval)	;; found it
		     (set! lo mid)))))
    ;(printf "%x>%s\n" addr (q:sym-name (or result (vector-ref map 0))))
    (or result (vector-ref map 0))))

(define (q:hist->weight-key hist-info)
  "Extract the event-name from the histogram info and convert it into
the key used to collect the histogram weights.  The resulting key is a
symbol named 'q:weight-EVENTNAME."
  (let ((event-name (assq-ref hist-info 'q:event-name)))
    (if (not event-name)
	(set! event-name "time"))
    (string->symbol (string-append "q:weight-" event-name))))

(define (q:hist->child-weight-key hist-info)
  "Extract the event-name from the child histogram info and convert it
into the key used to collect the histogram weights.  The resulting key
is a symbol named 'q:child-weight-EVENTNAME."
  (let ((event-name (assq-ref hist-info 'q:event-name)))
    (if (not event-name)
	(set! event-name "time"))
    (string->symbol (string-append "q:child-weight-" event-name))))

(define (q:read-histogram proc port)
  (let ((rx (make-regexp "0x([0-9a-f]*) ([0-9]*)"))
	(line #f) (m #f) (addr #f) (count #f))
    (do ((line (read-line port) (read-line port)))
	((eof-object? line))
      (set! m (regexp-exec rx line))
      (set! addr (string->number (match:substring m 1) 16))
      (set! count (string->number (match:substring m 2)))
      (proc addr count))))

(define (q:read-edges proc port)
  (let ((rx (make-regexp "0x([0-9a-f]*) 0x([0-9a-f]*) ([0-9]*)"))
	(line #f) (m #f) (from #f) (to #f) (count #f))
    (do ((line (read-line port) (read-line port)))
	((eof-object? line))
      (set! m (regexp-exec rx line))
      (set! from (string->number (match:substring m 1) 16))
      (set! to (string->number (match:substring m 2) 16))
      (set! count (string->number (match:substring m 3)))
      (proc from to count))))

(define (q:hist-assign-weights! sym-tab port hist-info addr-map)
  "Read a histogram data file from PORT and assign the weights to the
symbols in SYM-TAB.  HIST-INFO is used to convert the bin-counts to
weights (based on 'q:y-unit-conversion-factor) and to determine the
key under which to store the weights (based on 'q:event-name).
ADDR-MAP is an address-map which is used to map the labels in the
histogram data file to symbols."
  (let* ((histogram-key (q:hist->weight-key hist-info))
	 (y-factor (assq-ref hist-info 'q:y-unit-conversion-factor)))
    (if (not y-factor)
	(set! y-factor 1))
    (q:read-histogram
     (lambda (addr count)
       (let* ((sym (q:addr-map-ref addr-map addr))
	      (old-count (or (q:sym-ref sym histogram-key) 0)))
	  (q:sym-set! sym histogram-key (+ old-count (* y-factor count)))))
     port)))
;     (hash-fold
;      (lambda (key value prior-result)
;        (let* ((sym (q:addr-map-ref addr-map key))
; 	      (old-val (or (q:sym-ref sym histogram-key) 0)))
; 	 (q:sym-set! sym histogram-key
; 		      (+ old-val (* y-factor value)))))
;      '()
;      (q:read-hist port))))

(define (q:assign-call-counts! sym-tab port info addr-map)
  (q:read-edges
   (lambda (from to count)
     (let* ((from-sym (q:addr-map-ref addr-map from))
	    (to-sym (q:addr-map-ref addr-map to))
	    (from-cc (or (q:sym-ref from-sym 'q:callees) '()))
	    (to-cc   (or (q:sym-ref to-sym   'q:callers) '())))
;       (printf "%s->%s %u\n" (q:sym-name from-sym) (q:sym-name to-sym) count)
       (q:sym-set! from-sym 'q:callees
		    (assq-set! from-cc to-sym
			       (+ (or (assq-ref from-cc to-sym) 0) count)))
       (q:sym-set! to-sym 'q:callers
		    (assq-set! to-cc from-sym
			       (+ (or (assq-ref to-cc from-sym) 0) count)))
       (q:sym-set! to-sym 'q:call-count
		    (+ (or (q:sym-ref to-sym 'q:call-count) 0) count))))
   port))


;;
;; Call-graph related definitions.
;;

(define (q:call-graph-node? sym)
  "Return #t if symbol SYM is part of the call-graph (is a callee
and/or a caller."
  (not (and (null? (or (q:sym-ref sym 'q:callers) '()))
	    (null? (or (q:sym-ref sym 'q:callees) '())))))

(define (q:call-graph-root? sym)
  "Return #t if symbol SYM is a root in the call-graph.  A node is
considered to be a root if it calls other symbols but is never called
itself."
  (and (null? (or (q:sym-ref sym 'q:callers) '()))
       (not (null? (or (q:sym-ref sym 'q:callees) '())))))

(define (q:call-graph-dag sym-tab)
  "Create a DAG from the call-graph given by the 'q:callee edges in
symbol table SYMTAB.  The DAG is represented as a list of
topologically sorted symbols.  Each symbol has a topological order
number assigned under key 'q:topo-order, which can be used to detect
back-edges: an edge is a back-edge if the child's topological number
is less-than or equal to the parent's."
  ;; For the topological sort to work, we need a _connected_ DAG, not
  ;; a forest.  To guarantee this, we create an artificial ueber-root,
  ;; whose neighbors are all symbols with callees.  After the
  ;; topological sort, we remove that ueberroot again (we know it will
  ;; be first in the sorted list.
  (let* ((graph '())
	(ueber-root '())
	(ueber-root-adj-list (list ueber-root))
	(i 1))
    ;; Traverse the symbol table and for each symbol with
    ;; callees, build the ajdancency lists and append it to
    ;; the graph.  Also, if the symbol is a root, we add it
    ;; to the ueber-root-adj-list.
    (hash-fold (lambda (key sym rest)
		 (let ((adj-list '())
		       (edges (q:sym-ref sym 'q:callees)))
		   (if edges
		       (begin
			 (set! ueber-root-adj-list
			       (cons sym ueber-root-adj-list))
			 (for-each (lambda (edge)
				     (set! adj-list (cons (car edge)
							  adj-list)))
				   edges)
			 (set! graph (cons (cons sym adj-list) graph))))))
	       #f sym-tab)

    (set! graph (tsort (cons (cons ueber-root ueber-root-adj-list) graph) eq?))
;    (if (not (eq? (car graph) ueber-root))
;	(printf "INTERNAL ERROR!\n")
;	(printf "OK\n"))
    (set! graph (cdr graph))
    (for-each (lambda (sym)
;		(printf "setting %s to %u\n" (q:sym-name sym) i)
		(q:sym-set! sym 'q:topo-order i)
		(set! i (1+ i)))
	      graph)
    graph))

(define (q:propagate-child-weights! graph weight-key child-weight-key)
  "Walk a topologically sorted call-graph and distribute the
childrens' weight to their parents (linearly weighted by call-count).
Back-edges in the call-graph (due to recursive calls) are ignored."
  (define (node-weight! node)
    (let ((edges (or (q:sym-ref node 'q:callees) '()))
	  (node-order (q:sym-ref node 'q:topo-order))
	  (total 0))
      (for-each (lambda (edge)
		  (let* ((child (car edge))
			 (calls (cdr edge))
			 (child-order (q:sym-ref child 'q:topo-order)))
		    (if (and child-order (> child-order node-order))
			(begin
			  (let* ((child-weight (node-weight! child))
				 (child-called (q:sym-ref child
							  'q:call-count))
				 (prop (/ calls child-called)))
			    (set! total (+ total (* prop child-weight))))))))
		edges)
      (q:sym-set! node child-weight-key total)
      (+ total (or (q:sym-ref node weight-key) 0))))
  (for-each (lambda (node)
	      (if (q:call-graph-root? node)
		  (node-weight! node)))
	    graph))

;;
;; Provide default implementations for the qp file tags.  If they
;; don't suite your needs, override them after loading this file.
;;

(define q:info-list '())
(define q:histogram-list '())
(define q:call-count-list '())
(define q:map-file-list '())

(define (q:info alist)
  "Accumulates the auxiliary info in q:info-list."
  (set! q:info-list (append q:info-list (list alist))))

(define (q:object alist)
  "Accumulates all objects in the q:objects list for later use."
  (hash-set! q:object-hash (assq-ref alist 'q:name) alist))

(define (q:histogram alist)
  "Collects histogram information."
  (set! q:histogram-list (append q:histogram-list (list alist))))

(define (q:call-counts alist)
  "Collects call-count information."
  (set! q:call-count-list (append q:call-count-list (list alist))))

(define (q:map-file filename postfix)
  "Collects absolute map-file information."
  (set! q:map-file-list (append q:map-file-list
				(list (cons filename postfix)))))

(define (q:kallsyms filename)
  "Collects /proc/kallsyms information."
  (q:map-file filename #f))
