/* ListOfIntegersValueDelegate.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.invariants.values;

import be.ugent.caagt.swirl.lists.ArrayListModel;
import be.ugent.caagt.swirl.lists.TypedListModel;

import java.awt.Dialog;
import java.awt.Frame;
import java.awt.Point;
import java.awt.Window;
import java.awt.event.ActionEvent;
import javax.swing.AbstractAction;
import javax.swing.JComponent;
import javax.swing.JDialog;
import javax.swing.KeyStroke;
import javax.swing.SwingUtilities;

import org.grinvin.invariants.Invariant;
import org.grinvin.invariants.details.BarChartAndList;
import org.grinvin.io.IOFormatException;
import org.grinvin.list.graphs.GraphListElement;

import org.jdom.DataConversionException;
import org.jdom.Element;

/**
 *
 */
public class ListOfIntegersValueDelegate extends AbstractInvariantValueDelegate<ListOfIntegersValue> {
    
    //
    private static ListOfIntegersValueDelegate INSTANCE = new ListOfIntegersValueDelegate();
    
    //
    private ListOfIntegersValueDelegate() {}

    public static ListOfIntegersValueDelegate getInstance() {
        return INSTANCE;
    }
    
    //
    public ListOfIntegersValue loadValue(Element element, Invariant invariant, String computerId, String computerVersion) throws IOFormatException {
        int[] values;
        try {
            int size = element.getAttribute("length").getIntValue();
            values = new int[size];
            int i = 0;
            for (Object obj: element.getChildren("element")) {
                if (i >= size)
                    throw new IOFormatException("Too many element children");
                values[i] = Integer.valueOf(((Element)obj).getTextTrim());
                i ++;
            }
            if (i < size)
                throw new IOFormatException("Not enough element children");
        } catch (DataConversionException ex) {
            throw new IOFormatException("Invalid 'length' attribute", ex);
        }

        return new ListOfIntegersValue(values, invariant, computerId, computerVersion);
    }
    
    //
    public void saveValue(Element element, ListOfIntegersValue listOfIntegersValue) {
        element.setAttribute("length", Integer.toString(listOfIntegersValue.getValues().length));
        for (int value: listOfIntegersValue.getValues()) {
            Element child = new Element ("element").setText (Integer.toHexString(value));
            element.addContent(child);
        }
    }
    
    public Class<ListOfIntegersValue> getType() {
        return ListOfIntegersValue.class;
    }
    
    /**
     * Pops up a dialog with the reals displayed as bar charts.
     */
    public void showDetails(ListOfIntegersValue value, GraphListElement element, JComponent component,
            int x, int y) {
        
        Window window = SwingUtilities.getWindowAncestor(component);
        String title = element.getName() + " - " + value.getInvariant().getName();
        
        final JDialog dialog;
        if (window instanceof Frame)
            dialog = new JDialog((Frame)window, title, true);
        else if (window instanceof Dialog)
            dialog = new JDialog((Dialog)window, title, true);
        else
            dialog = new JDialog((Frame)null, title, true);
        
        TypedListModel<Integer> model = new ArrayListModel<Integer> (value.getValues().length);
        for (int d : value.getValues())
            model.add(d);
        dialog.setDefaultCloseOperation(JDialog.DISPOSE_ON_CLOSE);
        dialog.setContentPane(new BarChartAndList(model));
        dialog.pack();
        
        Point point = component.getLocationOnScreen();
        dialog.setLocation(point.x + x - dialog.getWidth()/2, 
                point.y + y - dialog.getHeight()/2);
        
        dialog.getRootPane().getInputMap(JComponent.WHEN_IN_FOCUSED_WINDOW).put(KeyStroke.getKeyStroke("ESCAPE"), "closeDialog");
        dialog.getRootPane().getActionMap().put("closeDialog", new AbstractAction() {
            public void actionPerformed(ActionEvent ev) {
                dialog.dispose();
            }
        });

        
        dialog.setVisible(true);
        
    }
    
    /**
     * This invariant value can display details.
     */
    public boolean hasDetails(ListOfIntegersValue value) {
        return value.getValues().length > 0;
    }
    
}
