use crate::drive::MediaCompatibility;

#[derive(Debug, Clone, Copy)]
pub(crate) enum DriveType {
    Drive,
    Disk,
    Card,
    Disc,
}

pub(crate) struct MediaData {
    pub id: MediaCompatibility,
    pub media_name: &'static str,
    pub media_family: &'static str,
    pub media_icon: &'static str,
    pub media_icon_symbolic: &'static str,
    pub media_type: DriveType,
    pub drive_icon: &'static str,
    pub drive_icon_symbolic: &'static str,
}

impl MediaData {
    //just used to generate a nicer interface for the `MEDIA_DATA` below, no need for builder
    //pattern or secondary struct
    #[allow(clippy::too_many_arguments)]
    const fn new(
        id: MediaCompatibility,
        media_name: &'static str,
        media_family: &'static str,
        media_icon: &'static str,
        media_icon_symbolic: &'static str,
        media_type: DriveType,
        drive_icon: &'static str,
        drive_icon_symbolic: &'static str,
    ) -> Self {
        //TODO: translate
        //https://github.com/storaged-project/udisks/blob/0b3879ab1d429b8312eaad0deb1b27e5545e39c1/udisks/udisksobjectinfo.c#L159
        //Translators: 'Thumb' here refers to "USB thumb drive", see http://en.wikipedia.org/wiki/Thumb_drive
        Self {
            id,
            media_name,
            media_family,
            media_icon,
            media_icon_symbolic,
            media_type,
            drive_icon,
            drive_icon_symbolic,
        }
    }
}

pub(crate) const MEDIA_DATA: [MediaData; 32] = [
    /* Translators: 'Thumb' here refers to "USB thumb drive", see http://en.wikipedia.org/wiki/Thumb_drive */
    MediaData::new(
        MediaCompatibility::Thumb,
        "Thumb",
        "Thumb",
        "media-removable",
        "media-removable-symbolic",
        DriveType::Drive,
        "media-removable",
        "media-removable-symbolic",
    ),
    MediaData::new(
        MediaCompatibility::Floppy,
        "Floppy",
        "Floppy",
        "media-floppy",
        "media-floppy-symbolic",
        DriveType::Disk,
        "drive-removable-media-floppy",
        "drive-removable-media-symbolic",
    ),
    MediaData::new(
        MediaCompatibility::FloppyZip,
        "Zip",
        "Zip",
        "media-floppy-jaz",
        "media-floppy-symbolic",
        DriveType::Disk,
        "drive-removable-media-floppy-jaz",
        "drive-removable-media-symbolic",
    ),
    MediaData::new(
        MediaCompatibility::FloppyJaz,
        "Jaz",
        "Jaz",
        "media-floppy-zip",
        "media-floppy-symbolic",
        DriveType::Disk,
        "drive-removable-media-floppy-zip",
        "drive-removable-media-symbolic",
    ),
    MediaData::new(
        MediaCompatibility::Flash,
        "Flash",
        "Flash",
        "media-flash",
        "media-flash-symbolic",
        DriveType::Card,
        "drive-removable-media-flash",
        "drive-removable-media-symbolic",
    ),
    MediaData::new(
        MediaCompatibility::FlashMs,
        "MemoryStick",
        "MemoryStick",
        "media-flash-ms",
        "media-flash-symbolic",
        DriveType::Card,
        "drive-removable-media-flash-ms",
        "drive-removable-media-symbolic",
    ),
    MediaData::new(
        MediaCompatibility::FlashSm,
        "SmartMedia",
        "SmartMedia",
        "media-flash-sm",
        "media-flash-symbolic",
        DriveType::Card,
        "drive-removable-media-flash-sm",
        "drive-removable-media-symbolic",
    ),
    MediaData::new(
        MediaCompatibility::FlashCf,
        "CompactFlash",
        "CompactFlash",
        "media-flash-cf",
        "media-flash-symbolic",
        DriveType::Card,
        "drive-removable-media-flash-cf",
        "drive-removable-media-symbolic",
    ),
    MediaData::new(
        MediaCompatibility::FlashMmc,
        "MMC",
        "SD",
        "media-flash-mmc",
        "media-flash-symbolic",
        DriveType::Card,
        "drive-removable-media-flash-mmc",
        "drive-removable-media-symbolic",
    ),
    MediaData::new(
        MediaCompatibility::FlashSd,
        "SD",
        "SD",
        "media-flash-sd",
        "media-flash-symbolic",
        DriveType::Card,
        "drive-removable-media-flash-sd",
        "drive-removable-media-symbolic",
    ),
    MediaData::new(
        MediaCompatibility::FlashSdxc,
        "SDXC",
        "SD",
        "media-flash-sd-xc",
        "media-flash-symbolic",
        DriveType::Card,
        "drive-removable-media-flash-sd-xc",
        "drive-removable-media-symbolic",
    ),
    MediaData::new(
        MediaCompatibility::FlashSdhc,
        "SDHC",
        "SD",
        "media-flash-sd-hc",
        "media-flash-symbolic",
        DriveType::Card,
        "drive-removable-media-flash-sd-hc",
        "drive-removable-media-symbolic",
    ),
    MediaData::new(
        MediaCompatibility::OpticalCd,
        "CD-ROM",
        "CD",
        "media-optical-cd-rom",
        "media-optical-symbolic",
        DriveType::Disc,
        "drive-optical",
        "drive-optical-symbolic",
    ),
    MediaData::new(
        MediaCompatibility::OpticalCdR,
        "CD-R",
        "CD",
        "media-optical-cd-r",
        "media-optical-symbolic",
        DriveType::Disc,
        "drive-optical-recorder",
        "drive-optical-symbolic",
    ),
    MediaData::new(
        MediaCompatibility::OpticalCdRw,
        "CD-RW",
        "CD",
        "media-optical-cd-rw",
        "media-optical-symbolic",
        DriveType::Disc,
        "drive-optical-recorder",
        "drive-optical-symbolic",
    ),
    MediaData::new(
        MediaCompatibility::OpticalDvd,
        "DVD",
        "DVD",
        "media-optical-dvd-rom",
        "media-optical-symbolic",
        DriveType::Disc,
        "drive-optical",
        "drive-optical-symbolic",
    ),
    MediaData::new(
        MediaCompatibility::OpticalDvdR,
        "DVD-R",
        "DVD",
        "media-optical-dvd-r",
        "media-optical-symbolic",
        DriveType::Disc,
        "drive-optical-recorder",
        "drive-optical-symbolic",
    ),
    MediaData::new(
        MediaCompatibility::OpticalDvdRw,
        "DVD-RW",
        "DVD",
        "media-optical-dvd-rw",
        "media-optical-symbolic",
        DriveType::Disc,
        "drive-optical-recorder",
        "drive-optical-symbolic",
    ),
    MediaData::new(
        MediaCompatibility::OpticalDvdRam,
        "DVD-RAM",
        "DVD",
        "media-optical-dvd-ram",
        "media-optical-symbolic",
        DriveType::Disc,
        "drive-optical-recorder",
        "drive-optical-symbolic",
    ),
    MediaData::new(
        MediaCompatibility::OpticalDvdPlusR,
        "DVD+R",
        "DVD",
        "media-optical-dvd-r-plus",
        "media-optical-symbolic",
        DriveType::Disc,
        "drive-optical-recorder",
        "drive-optical-symbolic",
    ),
    MediaData::new(
        MediaCompatibility::OpticalDvdPlusRw,
        "DVD+RW",
        "DVD",
        "media-optical-dvd-rw-plus",
        "media-optical-symbolic",
        DriveType::Disc,
        "drive-optical-recorder",
        "drive-optical-symbolic",
    ),
    MediaData::new(
        MediaCompatibility::OpticalDvdPlusRDl,
        "DVD+R DL",
        "DVD",
        "media-optical-dvd-dl-r-plus",
        "media-optical-symbolic",
        DriveType::Disc,
        "drive-optical-recorder",
        "drive-optical-symbolic",
    ),
    MediaData::new(
        MediaCompatibility::OpticalDvdPlusRwDl,
        "DVD+RW DL",
        "DVD",
        "media-optical-dvd-dl-r-plus",
        "media-optical-symbolic",
        DriveType::Disc,
        "drive-optical-recorder",
        "drive-optical-symbolic",
    ),
    MediaData::new(
        MediaCompatibility::OpticalBd,
        "BD-ROM",
        "Blu-Ray",
        "media-optical-bd-rom",
        "media-optical-symbolic",
        DriveType::Disc,
        "drive-optical",
        "drive-optical-symbolic",
    ),
    MediaData::new(
        MediaCompatibility::OpticalBdR,
        "BD-R",
        "Blu-Ray",
        "media-optical-bd-r",
        "media-optical-symbolic",
        DriveType::Disc,
        "drive-optical-recorder",
        "drive-optical-symbolic",
    ),
    MediaData::new(
        MediaCompatibility::OpticalBdRe,
        "BD-RE",
        "Blu-Ray",
        "media-optical-bd-re",
        "media-optical-symbolic",
        DriveType::Disc,
        "drive-optical-recorder",
        "drive-optical-symbolic",
    ),
    MediaData::new(
        MediaCompatibility::OpticalHddvd,
        "HDDVD",
        "HDDVD",
        "media-optical-hddvd-rom",
        "media-optical-symbolic",
        DriveType::Disc,
        "drive-optical",
        "drive-optical-symbolic",
    ),
    MediaData::new(
        MediaCompatibility::OpticalHddvdR,
        "HDDVD-R",
        "HDDVD",
        "media-optical-hddvd-r",
        "media-optical-symbolic",
        DriveType::Disc,
        "drive-optical-recorder",
        "drive-optical-symbolic",
    ),
    MediaData::new(
        MediaCompatibility::OpticalHddvdRw,
        "HDDVD-RW",
        "HDDVD",
        "media-optical-hddvd-rw",
        "media-optical-symbolic",
        DriveType::Disc,
        "drive-optical-recorder",
        "drive-optical-symbolic",
    ),
    MediaData::new(
        MediaCompatibility::OpticalMo,
        "MO",
        "CD",
        "media-optical-mo",
        "media-optical-symbolic",
        DriveType::Disc,
        "drive-optical",
        "drive-optical-symbolic",
    ),
    MediaData::new(
        MediaCompatibility::OpticalMrw,
        "MRW",
        "CD",
        "media-optical-mrw",
        "media-optical-symbolic",
        DriveType::Disc,
        "drive-optical-recorder",
        "drive-optical-symbolic",
    ),
    MediaData::new(
        MediaCompatibility::OpticalMrwW,
        "MRW-W",
        "CD",
        "media-optical-mrw-w",
        "media-optical-symbolic",
        DriveType::Disc,
        "drive-optical-recorder",
        "drive-optical-symbolic",
    ),
];
