#!/bin/bash
# SPDX-License-Identifier: MIT

PS4='${BASH_SOURCE[0]}:$LINENO: ${FUNCNAME[0]}:  '

set -e

TESTDIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
export PATH=$TESTDIR/setup-forgejo:$PATH

if test "$SCRATCHDIR"; then
  mkdir -p $SCRATCHDIR
else
  SCRATCHDIR=$(mktemp -d)
  trap "rm -fr $SCRATCHDIR" EXIT
fi

SELF_DIR="$TESTDIR/end-to-end"
SELF="$TESTDIR/end-to-end/end-to-end.sh"
FORGEJO_INSTANCE=https://code.forgejo.org
source $SELF_DIR/lib/lib.sh
source $SELF_DIR/upgrade/upgrade.sh

GO=$(go env GOROOT)/bin/go

HOVERFLY_VERSION=1.11.1 # renovate: datasource=github-tags depName=SpectoLabs/hoverfly
function hoverfly_version() {
  echo ${HOVERFLY_VERSION}
}

GITEA_VERSION=1.23 # renovate: datasource=docker depName=gitea/gitea
function gitea_version() {
  echo ${GITEA_VERSION}
}

FORGEJO_VERSION=11.0 # renovate: datasource=docker depName=data.forgejo.org/forgejo/forgejo
function forgejo_version() {
  echo ${FORGEJO_VERSION}
}

GITLAB_VERSION=17.11.3-ce.0 # renovate: datasource=docker depName=gitlab/gitlab-ce
function gitlab_version() {
  echo ${GITLAB_VERSION}
}

function test_run_coverage() {
  local name=$1
  shift

  local coveragedir="$SCRATCHDIR/coverage-$name"
  mkdir -p $coveragedir
  rm -f $coveragedir/*
  $GO test -cover -coverpkg code.forgejo.org/f3/gof3/... "$@" -args -test.gocoverdir=$coveragedir
}

function test_merge_coverage() {
  local coveragedirs=$(ls --directory --width=0 --format=commas $SCRATCHDIR/coverage-* | tr -d ' ')
  rm -fr $SCRATCHDIR/merged*
  mkdir -p $SCRATCHDIR/merged
  $GO tool covdata merge -i=$coveragedirs -o $SCRATCHDIR/merged
  $GO tool covdata textfmt -i=$SCRATCHDIR/merged -o $SCRATCHDIR/merged.out
}

function test_display_coverage() {
  test_merge_coverage
  echo "Coverage percentage per package"
  $GO tool covdata percent -i=$SCRATCHDIR/merged
}

function run_forgejo() {
  local version=$FORGEJO_VERSION

  echo "========= Forgejo driver compliance with version $version"

  stop_forgejo $TESTDIR/forgejo-app.ini
  reset_forgejo $TESTDIR/forgejo-app.ini
  start_forgejo $version $TESTDIR/forgejo-app.ini
}

function test_forgejo() {
  run_forgejo
  GOF3_FORGEJO_HOST_PORT=$(get_host_port $TESTDIR/forgejo-app.ini) test_run_coverage forgejo code.forgejo.org/f3/gof3/v3/...
}

function run_gitea() {
  local version=$GITEA_VERSION

  echo "========= Gitea driver compliance with version $version"

  stop_forgejo $TESTDIR/gitea-app.ini
  reset_forgejo $TESTDIR/gitea-app.ini
  start_gitea $version $TESTDIR/gitea-app.ini
}

function test_gitea() {
  run_gitea
  GOF3_GITEA_HOST_PORT=$(get_host_port $TESTDIR/gitea-app.ini) test_run_coverage gitea -run=TestF3Forge/gitea code.forgejo.org/f3/gof3/v3/tree/tests/...
}

function run_gitlab() {
  local version=gitlab/gitlab-ce:${GITLAB_VERSION}

  echo "========= GitLab driver compliance with version $version"

  stop_gitlab
  start_gitlab $version
}

function test_gitlab() {
  run_gitlab
  GOF3_GITLAB_HOST_PORT=$IP:8181 test_run_coverage gitlab -run=TestF3Forge/gitlab code.forgejo.org/f3/gof3/v3/tree/tests/...
}

function run() {
  test_forgejo
  test_gitlab
  test_gitea
  test_display_coverage
}

function prepare_container() {
  $SUDO apt-get install -y -qq skopeo wget # replace with gitlab_install_dependencies
  forgejo-dependencies.sh install_docker
  forgejo-binary.sh ensure_user forgejo
  mkdir -p /srv/forgejo-binaries
  chown -R forgejo /srv
  chmod -R +x /srv
  clobber
}

"${@:-run}"
