﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/signer/SignerRequest.h>
#include <aws/signer/Signer_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace signer {
namespace Model {

/**
 */
class ListSigningPlatformsRequest : public SignerRequest {
 public:
  AWS_SIGNER_API ListSigningPlatformsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListSigningPlatforms"; }

  AWS_SIGNER_API Aws::String SerializePayload() const override;

  AWS_SIGNER_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  ///@{
  /**
   * <p>The category type of a signing platform.</p>
   */
  inline const Aws::String& GetCategory() const { return m_category; }
  inline bool CategoryHasBeenSet() const { return m_categoryHasBeenSet; }
  template <typename CategoryT = Aws::String>
  void SetCategory(CategoryT&& value) {
    m_categoryHasBeenSet = true;
    m_category = std::forward<CategoryT>(value);
  }
  template <typename CategoryT = Aws::String>
  ListSigningPlatformsRequest& WithCategory(CategoryT&& value) {
    SetCategory(std::forward<CategoryT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Any partner entities connected to a signing platform.</p>
   */
  inline const Aws::String& GetPartner() const { return m_partner; }
  inline bool PartnerHasBeenSet() const { return m_partnerHasBeenSet; }
  template <typename PartnerT = Aws::String>
  void SetPartner(PartnerT&& value) {
    m_partnerHasBeenSet = true;
    m_partner = std::forward<PartnerT>(value);
  }
  template <typename PartnerT = Aws::String>
  ListSigningPlatformsRequest& WithPartner(PartnerT&& value) {
    SetPartner(std::forward<PartnerT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The validation template that is used by the target signing platform.</p>
   */
  inline const Aws::String& GetTarget() const { return m_target; }
  inline bool TargetHasBeenSet() const { return m_targetHasBeenSet; }
  template <typename TargetT = Aws::String>
  void SetTarget(TargetT&& value) {
    m_targetHasBeenSet = true;
    m_target = std::forward<TargetT>(value);
  }
  template <typename TargetT = Aws::String>
  ListSigningPlatformsRequest& WithTarget(TargetT&& value) {
    SetTarget(std::forward<TargetT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of results to be returned by this operation.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListSigningPlatformsRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Value for specifying the next set of paginated results to return. After you
   * receive a response with truncated results, use this parameter in a subsequent
   * request. Set it to the value of <code>nextToken</code> from the response that
   * you just received.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListSigningPlatformsRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_category;
  bool m_categoryHasBeenSet = false;

  Aws::String m_partner;
  bool m_partnerHasBeenSet = false;

  Aws::String m_target;
  bool m_targetHasBeenSet = false;

  int m_maxResults{0};
  bool m_maxResultsHasBeenSet = false;

  Aws::String m_nextToken;
  bool m_nextTokenHasBeenSet = false;
};

}  // namespace Model
}  // namespace signer
}  // namespace Aws
