﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/awstransfer/Transfer_EXPORTS.h>
#include <aws/awstransfer/model/As2ConnectorConfig.h>
#include <aws/awstransfer/model/ConnectorEgressType.h>
#include <aws/awstransfer/model/ConnectorStatus.h>
#include <aws/awstransfer/model/DescribedConnectorEgressConfig.h>
#include <aws/awstransfer/model/SftpConnectorConfig.h>
#include <aws/awstransfer/model/Tag.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace Transfer {
namespace Model {

/**
 * <p>Describes the parameters for the connector, as identified by the
 * <code>ConnectorId</code>.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/DescribedConnector">AWS
 * API Reference</a></p>
 */
class DescribedConnector {
 public:
  AWS_TRANSFER_API DescribedConnector() = default;
  AWS_TRANSFER_API DescribedConnector(Aws::Utils::Json::JsonView jsonValue);
  AWS_TRANSFER_API DescribedConnector& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_TRANSFER_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The unique Amazon Resource Name (ARN) for the connector.</p>
   */
  inline const Aws::String& GetArn() const { return m_arn; }
  inline bool ArnHasBeenSet() const { return m_arnHasBeenSet; }
  template <typename ArnT = Aws::String>
  void SetArn(ArnT&& value) {
    m_arnHasBeenSet = true;
    m_arn = std::forward<ArnT>(value);
  }
  template <typename ArnT = Aws::String>
  DescribedConnector& WithArn(ArnT&& value) {
    SetArn(std::forward<ArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The unique identifier for the connector.</p>
   */
  inline const Aws::String& GetConnectorId() const { return m_connectorId; }
  inline bool ConnectorIdHasBeenSet() const { return m_connectorIdHasBeenSet; }
  template <typename ConnectorIdT = Aws::String>
  void SetConnectorId(ConnectorIdT&& value) {
    m_connectorIdHasBeenSet = true;
    m_connectorId = std::forward<ConnectorIdT>(value);
  }
  template <typename ConnectorIdT = Aws::String>
  DescribedConnector& WithConnectorId(ConnectorIdT&& value) {
    SetConnectorId(std::forward<ConnectorIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The URL of the partner's AS2 or SFTP endpoint.</p> <p>When creating AS2
   * connectors or service-managed SFTP connectors (connectors without egress
   * configuration), you must provide a URL to specify the remote server endpoint.
   * For VPC Lattice type connectors, the URL must be null.</p>
   */
  inline const Aws::String& GetUrl() const { return m_url; }
  inline bool UrlHasBeenSet() const { return m_urlHasBeenSet; }
  template <typename UrlT = Aws::String>
  void SetUrl(UrlT&& value) {
    m_urlHasBeenSet = true;
    m_url = std::forward<UrlT>(value);
  }
  template <typename UrlT = Aws::String>
  DescribedConnector& WithUrl(UrlT&& value) {
    SetUrl(std::forward<UrlT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A structure that contains the parameters for an AS2 connector object.</p>
   */
  inline const As2ConnectorConfig& GetAs2Config() const { return m_as2Config; }
  inline bool As2ConfigHasBeenSet() const { return m_as2ConfigHasBeenSet; }
  template <typename As2ConfigT = As2ConnectorConfig>
  void SetAs2Config(As2ConfigT&& value) {
    m_as2ConfigHasBeenSet = true;
    m_as2Config = std::forward<As2ConfigT>(value);
  }
  template <typename As2ConfigT = As2ConnectorConfig>
  DescribedConnector& WithAs2Config(As2ConfigT&& value) {
    SetAs2Config(std::forward<As2ConfigT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Connectors are used to send files using either the AS2 or SFTP protocol. For
   * the access role, provide the Amazon Resource Name (ARN) of the Identity and
   * Access Management role to use.</p> <p> <b>For AS2 connectors</b> </p> <p>With
   * AS2, you can send files by calling <code>StartFileTransfer</code> and specifying
   * the file paths in the request parameter, <code>SendFilePaths</code>. We use the
   * file’s parent directory (for example, for <code>--send-file-paths
   * /bucket/dir/file.txt</code>, parent directory is <code>/bucket/dir/</code>) to
   * temporarily store a processed AS2 message file, store the MDN when we receive
   * them from the partner, and write a final JSON file containing relevant metadata
   * of the transmission. So, the <code>AccessRole</code> needs to provide read and
   * write access to the parent directory of the file location used in the
   * <code>StartFileTransfer</code> request. Additionally, you need to provide read
   * and write access to the parent directory of the files that you intend to send
   * with <code>StartFileTransfer</code>.</p> <p>If you are using Basic
   * authentication for your AS2 connector, the access role requires the
   * <code>secretsmanager:GetSecretValue</code> permission for the secret. If the
   * secret is encrypted using a customer-managed key instead of the Amazon Web
   * Services managed key in Secrets Manager, then the role also needs the
   * <code>kms:Decrypt</code> permission for that key.</p> <p> <b>For SFTP
   * connectors</b> </p> <p>Make sure that the access role provides read and write
   * access to the parent directory of the file location that's used in the
   * <code>StartFileTransfer</code> request. Additionally, make sure that the role
   * provides <code>secretsmanager:GetSecretValue</code> permission to Secrets
   * Manager.</p>
   */
  inline const Aws::String& GetAccessRole() const { return m_accessRole; }
  inline bool AccessRoleHasBeenSet() const { return m_accessRoleHasBeenSet; }
  template <typename AccessRoleT = Aws::String>
  void SetAccessRole(AccessRoleT&& value) {
    m_accessRoleHasBeenSet = true;
    m_accessRole = std::forward<AccessRoleT>(value);
  }
  template <typename AccessRoleT = Aws::String>
  DescribedConnector& WithAccessRole(AccessRoleT&& value) {
    SetAccessRole(std::forward<AccessRoleT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the Identity and Access Management (IAM)
   * role that allows a connector to turn on CloudWatch logging for Amazon S3 events.
   * When set, you can view connector activity in your CloudWatch logs.</p>
   */
  inline const Aws::String& GetLoggingRole() const { return m_loggingRole; }
  inline bool LoggingRoleHasBeenSet() const { return m_loggingRoleHasBeenSet; }
  template <typename LoggingRoleT = Aws::String>
  void SetLoggingRole(LoggingRoleT&& value) {
    m_loggingRoleHasBeenSet = true;
    m_loggingRole = std::forward<LoggingRoleT>(value);
  }
  template <typename LoggingRoleT = Aws::String>
  DescribedConnector& WithLoggingRole(LoggingRoleT&& value) {
    SetLoggingRole(std::forward<LoggingRoleT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Key-value pairs that can be used to group and search for connectors.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  DescribedConnector& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  DescribedConnector& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A structure that contains the parameters for an SFTP connector object.</p>
   */
  inline const SftpConnectorConfig& GetSftpConfig() const { return m_sftpConfig; }
  inline bool SftpConfigHasBeenSet() const { return m_sftpConfigHasBeenSet; }
  template <typename SftpConfigT = SftpConnectorConfig>
  void SetSftpConfig(SftpConfigT&& value) {
    m_sftpConfigHasBeenSet = true;
    m_sftpConfig = std::forward<SftpConfigT>(value);
  }
  template <typename SftpConfigT = SftpConnectorConfig>
  DescribedConnector& WithSftpConfig(SftpConfigT&& value) {
    SetSftpConfig(std::forward<SftpConfigT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The list of egress IP addresses of this connector. These IP addresses are
   * assigned automatically when you create the connector.</p>
   */
  inline const Aws::Vector<Aws::String>& GetServiceManagedEgressIpAddresses() const { return m_serviceManagedEgressIpAddresses; }
  inline bool ServiceManagedEgressIpAddressesHasBeenSet() const { return m_serviceManagedEgressIpAddressesHasBeenSet; }
  template <typename ServiceManagedEgressIpAddressesT = Aws::Vector<Aws::String>>
  void SetServiceManagedEgressIpAddresses(ServiceManagedEgressIpAddressesT&& value) {
    m_serviceManagedEgressIpAddressesHasBeenSet = true;
    m_serviceManagedEgressIpAddresses = std::forward<ServiceManagedEgressIpAddressesT>(value);
  }
  template <typename ServiceManagedEgressIpAddressesT = Aws::Vector<Aws::String>>
  DescribedConnector& WithServiceManagedEgressIpAddresses(ServiceManagedEgressIpAddressesT&& value) {
    SetServiceManagedEgressIpAddresses(std::forward<ServiceManagedEgressIpAddressesT>(value));
    return *this;
  }
  template <typename ServiceManagedEgressIpAddressesT = Aws::String>
  DescribedConnector& AddServiceManagedEgressIpAddresses(ServiceManagedEgressIpAddressesT&& value) {
    m_serviceManagedEgressIpAddressesHasBeenSet = true;
    m_serviceManagedEgressIpAddresses.emplace_back(std::forward<ServiceManagedEgressIpAddressesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The text name of the security policy for the specified connector.</p>
   */
  inline const Aws::String& GetSecurityPolicyName() const { return m_securityPolicyName; }
  inline bool SecurityPolicyNameHasBeenSet() const { return m_securityPolicyNameHasBeenSet; }
  template <typename SecurityPolicyNameT = Aws::String>
  void SetSecurityPolicyName(SecurityPolicyNameT&& value) {
    m_securityPolicyNameHasBeenSet = true;
    m_securityPolicyName = std::forward<SecurityPolicyNameT>(value);
  }
  template <typename SecurityPolicyNameT = Aws::String>
  DescribedConnector& WithSecurityPolicyName(SecurityPolicyNameT&& value) {
    SetSecurityPolicyName(std::forward<SecurityPolicyNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Current egress configuration of the connector, showing how traffic is routed
   * to the SFTP server. Contains VPC Lattice settings when using VPC_LATTICE egress
   * type.</p> <p>When using the VPC_LATTICE egress type, Transfer Family uses a
   * managed Service Network to simplify the resource sharing process.</p>
   */
  inline const DescribedConnectorEgressConfig& GetEgressConfig() const { return m_egressConfig; }
  inline bool EgressConfigHasBeenSet() const { return m_egressConfigHasBeenSet; }
  template <typename EgressConfigT = DescribedConnectorEgressConfig>
  void SetEgressConfig(EgressConfigT&& value) {
    m_egressConfigHasBeenSet = true;
    m_egressConfig = std::forward<EgressConfigT>(value);
  }
  template <typename EgressConfigT = DescribedConnectorEgressConfig>
  DescribedConnector& WithEgressConfig(EgressConfigT&& value) {
    SetEgressConfig(std::forward<EgressConfigT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Type of egress configuration for the connector. SERVICE_MANAGED uses Transfer
   * Family managed NAT gateways, while VPC_LATTICE routes traffic through customer
   * VPCs using VPC Lattice.</p>
   */
  inline ConnectorEgressType GetEgressType() const { return m_egressType; }
  inline bool EgressTypeHasBeenSet() const { return m_egressTypeHasBeenSet; }
  inline void SetEgressType(ConnectorEgressType value) {
    m_egressTypeHasBeenSet = true;
    m_egressType = value;
  }
  inline DescribedConnector& WithEgressType(ConnectorEgressType value) {
    SetEgressType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Error message providing details when the connector is in ERRORED status.
   * Contains information to help troubleshoot connector creation or operation
   * failures.</p>
   */
  inline const Aws::String& GetErrorMessage() const { return m_errorMessage; }
  inline bool ErrorMessageHasBeenSet() const { return m_errorMessageHasBeenSet; }
  template <typename ErrorMessageT = Aws::String>
  void SetErrorMessage(ErrorMessageT&& value) {
    m_errorMessageHasBeenSet = true;
    m_errorMessage = std::forward<ErrorMessageT>(value);
  }
  template <typename ErrorMessageT = Aws::String>
  DescribedConnector& WithErrorMessage(ErrorMessageT&& value) {
    SetErrorMessage(std::forward<ErrorMessageT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Current status of the connector. PENDING indicates creation/update in
   * progress, ACTIVE means ready for operations, and ERRORED indicates a failure
   * requiring attention.</p>
   */
  inline ConnectorStatus GetStatus() const { return m_status; }
  inline bool StatusHasBeenSet() const { return m_statusHasBeenSet; }
  inline void SetStatus(ConnectorStatus value) {
    m_statusHasBeenSet = true;
    m_status = value;
  }
  inline DescribedConnector& WithStatus(ConnectorStatus value) {
    SetStatus(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_arn;

  Aws::String m_connectorId;

  Aws::String m_url;

  As2ConnectorConfig m_as2Config;

  Aws::String m_accessRole;

  Aws::String m_loggingRole;

  Aws::Vector<Tag> m_tags;

  SftpConnectorConfig m_sftpConfig;

  Aws::Vector<Aws::String> m_serviceManagedEgressIpAddresses;

  Aws::String m_securityPolicyName;

  DescribedConnectorEgressConfig m_egressConfig;

  ConnectorEgressType m_egressType{ConnectorEgressType::NOT_SET};

  Aws::String m_errorMessage;

  ConnectorStatus m_status{ConnectorStatus::NOT_SET};
  bool m_arnHasBeenSet = false;
  bool m_connectorIdHasBeenSet = false;
  bool m_urlHasBeenSet = false;
  bool m_as2ConfigHasBeenSet = false;
  bool m_accessRoleHasBeenSet = false;
  bool m_loggingRoleHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
  bool m_sftpConfigHasBeenSet = false;
  bool m_serviceManagedEgressIpAddressesHasBeenSet = false;
  bool m_securityPolicyNameHasBeenSet = false;
  bool m_egressConfigHasBeenSet = false;
  bool m_egressTypeHasBeenSet = false;
  bool m_errorMessageHasBeenSet = false;
  bool m_statusHasBeenSet = false;
};

}  // namespace Model
}  // namespace Transfer
}  // namespace Aws
